<?php declare(strict_types = 1);

namespace EshopCatalog\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use EshopCatalog\Model\Entities\Availability;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductSupplier;

class AvailabilityService extends BaseFrontEntityService
{
	protected $entityClass = Availability::class;

	protected ?array $cAvs = null;

	protected static $updated = [];

	/**
	 * Kontrola stavu produktu pokud je vyprodán nebo skladem
	 *
	 * @param Product $product
	 */
	public function updateAvailabilityByQuantity(Product &$product): void
	{
		$allowExternal = (int) $this->settings->get('eshopCatalogDisableExternalStorage', 0) === 0;

		if ($product->unlimitedQuantity) {
			$product->setAvailability($this->getEr()->findOneBy(['ident' => Availability::IN_STOCK]));
		} else if ($product->quantity <= 0) {
			$supplierPos = null;
			$supplierAv  = null;

			if ($allowExternal) {
				foreach ($product->getSuppliers()->toArray() as $supp) {
					/** @var ProductSupplier $supp */
					if (!$supp->isActive)
						continue;

					if ($supp->availabilityAfterSoldOut && $supp->quantity > 0 && ($supplierPos === null || $supplierPos > $supp->availabilityAfterSoldOut->getPosition())) {
						$supplierAv  = $supp->availabilityAfterSoldOut;
						$supplierPos = $supp->availabilityAfterSoldOut->getPosition();
					}
				}
			}

			if (!$supplierAv && $product->availabilityAfterSoldOut)
				$supplierAv = $product->availabilityAfterSoldOut;

			$product->setAvailability($supplierAv ?: $this->getEr()->findOneBy(['ident' => Availability::SOLD_OUT]));
			self::$updated[$product->getId()] = $product->getId();
		} else if ($product->quantity > 0 && (!$product->getAvailability() || $product->getAvailability()->getIdent() != Availability::IN_STOCK)) {
			$product->setAvailability($this->getEr()->findOneBy(['ident' => Availability::IN_STOCK]));
		}
	}

	public function updateProductsWithoutSupplier()
	{
		$inStock = $this->getEr()->findOneBy(['ident' => Availability::IN_STOCK]);
		$soldOut = $this->getEr()->findOneBy(['ident' => Availability::SOLD_OUT]);

		$isIn   = [];
		$isOut  = [];
		$others = [];

		foreach ($this->em->getRepository(Product::class)->createQueryBuilder('p')
			         ->select('p.id, p.quantity, IDENTITY(p.availabilityAfterSoldOut) as avAfterSoldOut')
			         ->leftJoin('p.suppliers', 'ps')
			         ->andWhere('ps.product IS NULL')
			         ->getQuery()->getResult() as $row) {
			if (isset(self::$updated[$row['id']]))
				continue;

			if ($row['avAfterSoldOut'])
				$others[$row['avAfterSoldOut']][] = $row['id'];
			else if ($row['quantity'] > 0)
				$isIn[] = $row['id'];
			else
				$isOut[] = $row['id'];
		}

		foreach (array_chunk($isIn, 100) as $chunk) {
			$this->em->getConnection()
				->executeStatement("UPDATE eshop_catalog__product SET id_availability = {$inStock->getId()} WHERE id IN (" . implode(',', $chunk) . ")");
		}

		foreach (array_chunk($isOut, 100) as $chunk) {
			$this->em->getConnection()
				->executeStatement("UPDATE eshop_catalog__product SET id_availability = {$soldOut->getId()} WHERE id IN (" . implode(',', $chunk) . ")");
		}

		foreach ($others as $avId => $ids) {
			foreach (array_chunk($ids, 100) as $chunk) {
				$this->em->getConnection()
					->executeStatement("UPDATE eshop_catalog__product SET id_availability = {$avId} WHERE id IN (" . implode(',', $chunk) . ")");
			}
		}
	}

	protected function getAvailability(): array
	{
		if ($this->cAvs === null) {
			$this->cAvs = [];

			foreach ($this->getEr()->findAll() as $row) {
				/** @var Availability $row */
				$this->cAvs[$row->getIdent()] = $row;
			}
		}

		return $this->cAvs;
	}
}
