<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Navigation;

use Core\Model\Sites;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Event\RouteInFindAliasEvent;
use EshopCatalog\Model\Helpers\ProductsHelper;
use Navigations\Model\Event\RouteInEvent;
use Navigations\Model\Event\RouteOutEvent;
use Nette\Application\LinkGenerator;
use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Dao\SiteMapUrl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\FrontModule\Model\ProductQuery;
use Navigations\Model\Providers\INavigationItem;
use Nette\Caching\Cache;
use Nette\Utils\ArrayHash;
use Nette\Utils\DateTime;
use Nette\Utils\Strings;
use Pages\AdminModule\Model\TemplatePages;

class Home extends BaseNavigation implements INavigationItem
{
	public const COMPONENT_NAME = 'eshopCatalog.navigation.home';

	public const DEFAULT_PRESENTER = 'EshopCatalog:Front:Default';

	public const ACTION_PRODUCT  = 'product';
	public const ACTION_CATEGORY = 'category';

	protected $title = 'home';

	protected $action = 'default';

	protected $componentChildStart = 0;
	protected $componentChildLimit = 4000;
	protected $cComponentChildGalleries;

	protected ?string $lastSite = null;

	const CACHE_PRODUCT = 'eshopCatalogProduct';

	/** @var TemplatePages */
	protected $templatePagesService;

	/** @var LinkGenerator */
	public $linkGenerator;

	protected ProductsFacade $productsFacade;

	protected ProductsHelper $productsHelper;

	protected Sites $sites;

	public function __construct(TemplatePages $templatePages, ProductsFacade $productsFacade, ProductsHelper $productsHelper, Sites $sites)
	{
		$this->templatePagesService = $templatePages;
		$this->productsFacade       = $productsFacade;
		$this->productsHelper       = $productsHelper;
		$this->sites                = $sites;
	}

	/**
	 * @param BaseForm $baseForm
	 *
	 * @return BaseContainer
	 */
	public function getFormContainer(BaseForm $baseForm)
	{
		$container = new BaseContainer();

		$site  = $this->navFormSite ?: $baseForm->getCustomData('loadComponent')['site'];
		$pages = $this->templatePagesService->getBySiteForSelectOption($site ?? $baseForm['site']->getValue());

		$container->addSelect('templateHome', 'eshopCatalog.navigationBuilder.templateHome', $pages);
		$container->addSelect('templateProduct', 'eshopCatalog.navigationBuilder.templateDetail', $pages);
		$container->addSelect('templateCategory', 'eshopCatalog.navigationBuilder.templateCategory', $pages);

		if ($baseForm->getParent()->lookup(null, false)) {
			$componentId = $baseForm->getParent()->getParameter('componentId');
			if ($componentId)
				$container['templateHome']->setAttribute('data-custom-link', $baseForm->getParent()->link('componentCustom!', [$componentId]));
		} else {
			$baseForm->onAnchor[] = function() use ($container, $baseForm) {
				$container['templateHome']->setAttribute('data-custom-link', $baseForm->getParent()->link('componentCustom!', [$baseForm->getValues()->componentType]));
			};
		}

		return $container;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{

	}

	public function updateCacheDep(array &$cache, $urlParams)
	{
		$cache[Cache::TAGS][] = self::CACHE_PRODUCT . '/' . $urlParams['id'];

		parent::updateCacheDep($cache, $urlParams);
	}

	/*******************************************************************************************************************
	 * ===========================  Route
	 */

	public function presenterActionCheck($presenter, $action)
	{
		return self::DEFAULT_PRESENTER == $presenter && in_array($action, [$this->action, self::ACTION_PRODUCT,
				self::ACTION_CATEGORY]);
	}

	/**
	 * @param DaoNavigationItem $navigation
	 * @param array             $urlParams
	 *
	 * @return array|mixed
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 * @throws \Nette\Application\UI\InvalidLinkException
	 * @throws \Throwable
	 */
	public function routerIn($navigation, $urlParams = [])
	{
		$returnDefault = [
			'presenter' => self::DEFAULT_PRESENTER,
			'action'    => $this->action,
			'id'        => null,
		];

		if (isset($urlParams['getHomepage']))
			return $returnDefault;

		if (isset($urlParams['path']) && $navigation->alias != $urlParams['path']) {
			$params    = explode('/', $urlParams['path']);
			$lastParam = end($params);
			$id        = explode('-', $lastParam, 2);
			$resolver  = Config::load('useOldUrls') ? $this->productsHelper->getOldUrlResolver() : null;

			if ($resolver) {
				$return = $resolver->routeIn($navigation, $urlParams, $returnDefault);

				if ($return)
					return $return;
			}

			if (is_numeric($id[0]) && $this->productsFacade->getProduct((int) $id[0])) {
				return [
					'presenter' => self::DEFAULT_PRESENTER,
					'action'    => self::ACTION_PRODUCT,
					'id'        => $id[0],
				];
			} else {
				$return = [];
				$this->eventDispatcher->dispatch(new RouteInFindAliasEvent($urlParams['path'], $urlParams, $return), BaseNavigation::class . '::routeInFindAlias');
				if ($return) {
					if (!isset($return['activeNavigation']))
						$return['activeNavigation'] = $navigation;

					return $return;
				}
			}
		}

		$returnEvent = [];
		$this->eventDispatcher->dispatch(new RouteInEvent(null, $urlParams, $returnEvent), Home::class . '::routeInStart');
		if ($returnEvent)
			return $returnEvent;

		return $returnDefault;
	}

	public function routerOut(DaoNavigationItem $navigation, &$urlParams): ?string
	{
		$cfUrl = $navigation->isHomepage ? '/' : '/' . $navigation->alias;
		if ($urlParams['action'] == self::ACTION_PRODUCT && isset($urlParams['id'])) {
			$resolver = Config::load('useOldUrls') ? $this->productsHelper->getOldUrlResolver() : null;

			if ($resolver) {
				$out = $resolver->routeOut($navigation, $urlParams);

				if ($out)
					return $out;
			}

			$product = $this->productsFacade->productsService->getForNavigation((int) $urlParams['id']);

			if ($product)
				return ($navigation->isHomepage ? '' : '/' . $navigation->alias) . '/' . $product['id'] . '-' . Strings::webalize($product['name']);
		} else {
			$this->eventDispatcher->dispatch(new RouteOutEvent($navigation, $urlParams, $cfUrl), BaseNavigation::class . '::routeOut');
		}

		return '/' . ltrim((string) $cfUrl, '/');
	}

	/**
	 * @param DaoNavigationItem|null $navigation
	 * @param bool                   $siteMapData
	 *
	 * @return array
	 */
	public function getComponentChild(string $lang, $navigation = null, $siteMapData = false)
	{
		$arr         = [];
		$i           = 0;
		$currentSite = $this->sites->getCurrentSite()->getIdent();

		if ($this->lastSite != $currentSite) {
			$this->componentChildStart = 0;
		}

		//		if ($this->cComponentChildGalleries === null) {
		//			if ($siteMapData) {
		//				foreach ($this->productsService->em->getRepository(Image::class)->createQueryBuilder('i')
		//					         ->select('i.path, i.filename, it.title, p.id as productId')
		//					         ->leftJoin('i.texts', 'it', 'WITH', 'it.lang = :lang')
		//					         ->join('i.album', 'a')->leftJoin(Product::class, 'p', 'WITH', 'p.gallery = a.id')
		//					         ->setParameter('lang', $lang)
		//					         ->getQuery()->getScalarResult() as $row) {
		//					$this->cComponentChildGalleries[$row['productId']][] = $row;
		//				}
		//
		//				$this->productsService->em->clear(Image::class);
		//			}
		//		}
		//		$galleries = $this->cComponentChildGalleries;

		foreach ((new ProductQuery($lang))
			         ->withTexts()
			         ->inSite($currentSite)
			         ->getQueryBuilder($this->productsFacade->productsService->getEr())
			         ->select('p.id, pt.name, p.modified, pt.seo, p.modified')
			         ->where('p.isPublished = 1')
			         ->andWhere('sites.isActive = 1')
			         ->setFirstResult($this->componentChildStart)->setMaxResults($this->componentChildLimit)
			         ->getQuery()->getScalarResult() as $row) {
			$id = $row['id'];

			$data = [
				'presenter' => self::DEFAULT_PRESENTER,
				'action'    => self::ACTION_PRODUCT,
				'id'        => $id,
				'locale'    => $lang,
			];

			if ($siteMapData) {
				$siteMapUrl = new SiteMapUrl();
				if ($row['modified'])
					$siteMapUrl->setModified(DateTime::from($row['modified']));

				$seo = $row['seo'] ? @unserialize($row['seo']) : null;

				if (!$seo) {
					$seo = [];
					foreach (['addToSiteMap', 'siteMapChangeFreq', 'siteMapPriority', 'robots'] as $col)
						$seo[$col] = $navigation->getSeo($col);
				}

				if (is_array($seo)) {
					if (!isset($seo['addToSiteMap']))
						$seo['addToSiteMap'] = '1';
					if ($seo['addToSiteMap'] === '0' || is_string($seo['robots']) && strpos('noindex', $seo['robots']) !== false)
						continue;

					$siteMapUrl->setSeo($seo, null, true);
				}

				//				if (isset($galleries[$id])) {
				//					foreach ($galleries[$id] as $k => $rowImg) {
				//						$img           = new SiteMapImage();
				//						$img->title    = $rowImg['title'];
				//						$img->filePath = $rowImg['path'] . '/' . $rowImg['filename'];
				//						$siteMapUrl->addImage($img);
				//					}
				//				}

				$data['siteMapData']['default'] = $siteMapUrl;
			}

			$arr[] = $data;
			$i++;
		}

		$this->productsFacade->productsService->em->clear();

		$this->componentChildStart += $this->componentChildLimit;
		$this->lastSite            = $currentSite;

		if (!empty($arr))
			$arr['continue'] = true;

		return $arr;
	}
}
