<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model\FormContainers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\UI\AdminPresenter;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\FrontPresenter;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Package;
use EshopCatalog\Model\Entities\PackageItem;
use EshopCatalog\Model\Entities\Product;
use Nette\Application\LinkGenerator;

class ProductPackageContainer
{
	protected EntityManagerDecorator $em;
	protected Products               $products;
	protected LinkGenerator          $linkGenerator;
	protected CacheService           $cacheService;

	public function __construct(
		EntityManagerDecorator $em,
		Products               $products,
		LinkGenerator          $linkGenerator,
		CacheService           $cacheService
	)
	{
		$this->em            = $em;
		$this->products      = $products;
		$this->linkGenerator = $linkGenerator;
		$this->cacheService  = $cacheService;
	}

	public function getContainer(BaseForm $form, string $containerName): BaseContainer
	{
		$container = new BaseContainer();

		if (!Config::load('product.allowPackage', false)) {
			return $container;
		}

		$productLink = $this->linkGenerator->link('EshopCatalog:Admin:Products:edit', ['__ID__']);

		$container->addText('product', 'eshopCatalog.productPackage.searchAndAddToPackage')
			->setHtmlAttribute('data-package-container-name', $containerName)
			->setHtmlAttribute('data-autocomplete-name', 'packageProduct')
			->setHtmlAttribute('data-autocomplete-keys', 'id,code1,ean,name')
			->setHtmlAttribute('data-product-link', $productLink)
			->setOmitted();

		$form->monitor(AdminPresenter::class, function(AdminPresenter $presenter) use ($container) {
			$container->getComponent('product')
				->setHtmlAttribute('data-autocomplete-url', $this->linkGenerator->link('EshopCatalog:Cron:Products:loadAll', [
					'excluded'        => [$presenter->getParameter('id')],
					'excludePackages' => '1',
				]));
		});

		$container->addCustomData('productLink', $productLink);
		$container->addCustomData('template', __DIR__ . '/ProductPackageContainer.latte');
		$form->addCustomData('packagesContainerName', $containerName);

		return $container;
	}

	public function saveData(array $values, Product $product): void
	{
		unset($values['product']);

		$package = $product->package;

		// If package not exist and data post -> create
		if (!empty($values) && $package === null) {
			$package = new Package;

			$product->package = $package;

			$this->em->persist($package);
			$this->em->persist($product);
		}

		// If no package -> return
		if (!$package) {
			return;
		}

		$currentItems = $package->items->toArray();

		// Create
		$create = array_diff_key($values, $currentItems);

		$isPackages = $create
			? $this->em->getRepository(Product::class)->createQueryBuilder('p', 'p.id')
				->select('p.id, IDENTITY(p.package) as package')
				->andWhere('p.id IN (' . implode(',', array_keys($create)) . ')')
				->getQuery()->getArrayResult()
			: [];
		foreach ($create as $productId => $v) {
			if ($isPackages[$productId]['package'] !== null) {
				continue;
			}

			$packageItem           = new PackageItem($package, $this->products->getReference($productId));
			$packageItem->quantity = (int) $v['quantity'];

			$this->em->persist($packageItem);

			$package->items->set($productId, $packageItem);
		}

		// Delete
		foreach (array_diff_key($currentItems, $values) as $productId => $packageItem) {
			$this->em->remove($packageItem);

			$package->items->remove($productId);
		}

		// Update
		foreach (array_intersect_key($currentItems, $values) as $productId => $packageItem) {
			$packageItem->quantity = (int) $values[$productId]['quantity'];

			$this->em->persist($packageItem);
		}

		if ($package->items->count() === 0) {
			$this->em->remove($package);

			$product->package = null;
			$this->em->persist($product);
		}

		$this->em->flush();
	}

	public function setDefaults(BaseContainer $container, Package $package): void
	{
		$defaults = [];

		foreach ($package->items->toArray() as $item) {
			$product = $item->getProduct();

			$defaults[$item->getProduct()->getId()] = [
				'code1'    => $product->code1,
				'name'     => $product->getText()->name,
				'quantity' => $item->quantity,
			];
		}

		$container->addCustomData('defaults', $defaults);
	}

	/**
	 * @param int[] $packages
	 */
	public function clearCache(array $packages): void
	{
		$this->cacheService->removePackage($packages);
	}
}
