<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\PseudoWarehouse;
use EshopOrders\AdminModule\Components\Order\OrderItemAddForm;
use EshopOrders\AdminModule\Components\Order\OrderItemForm;
use EshopOrders\AdminModule\Model\Event\OrderItemEvent;
use Nette\Utils\Validators;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class OrderItemChangedSubscriber implements EventSubscriberInterface
{
	protected EntityManagerDecorator $em;
	protected PseudoWarehouse $pseudoWarehouse;

	public function __construct(EntityManagerDecorator $em, PseudoWarehouse $pseudoWarehouse)
	{
		$this->em              = $em;
		$this->pseudoWarehouse = $pseudoWarehouse;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			OrderItemAddForm::class . '::formSuccess' => ['orderItemAdded', 100],
			OrderItemForm::class . '::beforeDelete'   => ['orderItemRemoved', 100],
			OrderItemForm::class . '::beforeSave'     => ['orderItemSaved', 100],
		];
	}

	public function orderItemAdded(OrderItemEvent $event): void
	{
		if (!Config::load('pseudoWarehouse') || $event->orderItem->getOrder()->isCanceled()) {
			return;
		}

		$productId = $event->orderItem->getProductId();

		if (!Validators::isNone($productId) && Validators::isNumericInt($productId)) {
			$this->pseudoWarehouse->lowerQuantities([$productId => $event->orderItem->getQuantity()], $event->orderItem->getOrder());
		}
	}

	public function orderItemRemoved(OrderItemEvent $event): void
	{
		if (!Config::load('pseudoWarehouse') || $event->orderItem->getOrder()->isCanceled()) {
			return;
		}

		$productId = $event->orderItem->getProductId();

		if (!Validators::isNone($productId) && Validators::isNumericInt($productId)) {
			$this->pseudoWarehouse->raiseQuantities([$event->orderItem->getProductId() => $event->orderItem->getQuantity()], $event->orderItem->getOrder());
		}
	}

	public function orderItemSaved(OrderItemEvent $event): void
	{
		if (!Config::load('pseudoWarehouse') || $event->orderItem->getOrder()->isCanceled()) {
			return;
		}

		$orderItemId = $event->orderItem->getId();
		if ($orderItemId === null) {
			return;
		}

		$conn   = $this->em->getConnection();
		$result = $conn->fetchAssociative('SELECT quantity FROM eshop_orders__order_item WHERE id = ' . $orderItemId);

		if (!$result) {
			return;
		}

		$oldOrderItemQuantity = $result['quantity'];
		$newOrderItem         = $event->orderItem;
		$changedQuantity      = $newOrderItem->getQuantity();
		$diff                 = abs($newOrderItem->getQuantity() - $oldOrderItemQuantity);

		if ($newOrderItem->getQuantity() > $oldOrderItemQuantity) {
			// doslo k snizeni
			$this->pseudoWarehouse->lowerQuantities([$event->orderItem->getProductId() => $diff], $event->orderItem->getOrder());
		} else if ($newOrderItem->getQuantity() < $oldOrderItemQuantity) {
			// doslo ke navyseni
			$newOrderItem->setQuantity($diff);
			$this->pseudoWarehouse->raiseQuantities([$event->orderItem->getProductId() => $diff], $event->orderItem->getOrder());
		}

		$newOrderItem->setQuantity($changedQuantity);
	}

}