<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Presenters;

use Core\Model\Dao\Site;
use Core\Model\Sites as SitesService;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Components\Categories\CategoriesGrid;
use EshopCatalog\AdminModule\Components\Categories\CategoryForm;
use EshopCatalog\AdminModule\Components\Categories\ICategoriesGridFactory;
use EshopCatalog\AdminModule\Components\Categories\ICategoryFormFactory;
use EshopCatalog\AdminModule\Model\Categories;
use Nette\Application\UI\Multiplier;

class CategoriesPresenter extends BasePresenter
{
	protected Categories   $categoriesService;
	protected SitesService $defaultSitesService;

	public function __construct(
		Categories   $categoriesService,
		SitesService $defaultSitesService
	)
	{
		parent::__construct();

		$this->categoriesService   = $categoriesService;
		$this->defaultSitesService = $defaultSitesService;
	}

	protected function startup()
	{
		parent::startup();
		$this->setHeader('eshopCatalog.title.categories', 'fa fa-list');
	}

	protected function canUpdateGrid(): bool { return $this->categoriesService->getSessionSection()->categoriesGridUpdateEnabled ?? true; }

	/*******************************************************************************************************************
	 * ==================================   Actions
	 */
	public function actionDefault(?string $site = null): void
	{
		$this->template->gridUpdateEnabled = $this->canUpdateGrid();
		$sites                             = $this->sitesService->getAll();
		$selectedSite                      = null;

		if (!$site) {
			$site = $this->defaultSitesService->getCurrentSite()->getIdent();
		}

		foreach ($sites as $row) {
			if ($row->getIdent() === $site) {
				$selectedSite = $row;
				break;
			}
		}

		$this->template->selectedSite = $selectedSite;
		$this->template->sites        = $sites;

		/** @var Site $firstSite */
		$firstSite = array_values($sites)[0];

		$this['navigation']->setData(['header' => [
			[
				'title' => 'eshopCatalog.menu.addCategory',
				'link'  => $this->link('add!', count($sites) > 1 ? ['site-ident' => $firstSite->getIdent()] : []),
				'ico'   => 'plus',
				'class' => 'ajax',
			],
		]]);
	}

	public function actionEdit(int $id): void
	{
		$category = $this->categoriesService->get($id);

		if (!$category) {
			$this->error();
		}

		$this['categoryForm']->setCategory($id);
		$this->setHeader(
			$this->t('eshopCatalog.title.editCategory') . ' ' . ($category->getCategoryText() ? $category->getCategoryText()->getName() : ''),
			'fa fa-list',
		);
	}

	/*******************************************************************************************************************
	 * ==================================   Handle
	 */

	public function handleUpdateGridSet(): void
	{
		$this->categoriesService->getSessionSection()->categoriesGridUpdateEnabled = (bool) $this->getHttpRequest()->getPost('val');

		$this->sendPayload();
	}

	public function handleAdd(): void
	{
		$this->template->modalTitle       = $this->t('eshopCatalog.title.newCategory');
		$this->template->modal            = 'categoryForm';
		$this->template->modalDialogClass = 'modal-xl';
		$this->redrawControl('modal');
	}

	/*******************************************************************************************************************
	 * ==================================   Components
	 */

	protected function createComponentCategoriesGrid(ICategoriesGridFactory $factory): Multiplier
	{
		return new Multiplier(function(string $sideIdent) use ($factory): CategoriesGrid {
			return $factory->create($sideIdent);
		});
	}

	protected function createComponentCategoryForm(ICategoryFormFactory $category): CategoryForm
	{
		$control = $category->create();
		if ($this->getParameter('id')) {
			$control->setCategory((int) $this->getParameter('id'));
		}

		$control['form']->onAnchor[] = function() use ($control) {
			$control['form']->onSuccessSave[]         = function(BaseForm $form) {
				if ($this->canUpdateGrid()) {
					$this['categoriesGrid-' . $form->getCustomData('siteIdent')]->redrawControl('list');
				}

				$this->payload->hideModal = true;
				$this->redrawControl('flashes');
			};
			$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
				if ($this->getAction() === 'edit') {
					$this->redirect('default');
				} else {
					if ($this->canUpdateGrid()) {
						$this['categoriesGrid-' . $form->getCustomData('siteIdent')]->redrawControl('list');
					}

					$this->payload->hideModal = true;
					$this->redrawControl('flashes');
				}
			};

			if ($this->getAction() === 'edit') {
				$control['form']->onCancel[] = function() {
					$this->redirect('default');
				};
			} else {
				$control['form']->getComponent('saveControl')->closeModalOnCancel();
			}
		};

		return $control;
	}

}
