<?php declare(strict_types = 1);

namespace EshopCatalog\CronModule\Model;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Lang\Langs;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Entities\CategoryProduct;
use EshopCatalog\Model\Entities\ProductInSite;
use EshopCatalog\Model\Entities\ProductVariant;

class ProductsVariants
{
	protected EntityManagerDecorator $em;
	protected CacheService           $cacheService;
	protected Langs                  $langsService;

	public function __construct(
		EntityManagerDecorator $em,
		CacheService           $cacheService,
		Langs                  $langsService
	)
	{
		$this->em           = $em;
		$this->cacheService = $cacheService;
		$this->langsService = $langsService;
	}

	public function reSetAllCategories(): void
	{
		$variants = [];
		$map      = [];
		foreach ($this->em->createQueryBuilder()->select('IDENTITY(pv.product) as prod, pv.isDefault, pv.variantId')
			         ->from(ProductVariant::class, 'pv')
			         ->orderBy('pv.isDefault', 'DESC')
			         ->getQuery()->getArrayResult() as $row) {
			if ($row['isDefault'] == 1) {
				$variants[$row['prod']] = [];
				$map[$row['variantId']] = $row['prod'];
			} else {
				$variants[$map[$row['variantId']]][] = $row['prod'];
			}
		}

		$categories = [];
		foreach ($this->em->createQueryBuilder()->select('IDENTITY(cp.product) as prod, IDENTITY(cp.category) as cat')
			         ->from(CategoryProduct::class, 'cp')
			         ->getQuery()->getScalarResult() as $row) {
			$categories[$row['prod']][] = $row['cat'];
		}

		$sites = [];
		foreach ($this->em->createQueryBuilder()->select('IDENTITY(ps.product) as prod, IDENTITY(ps.site) as site, ps.isActive, IDENTITY(ps.category) as cat')
			         ->from(ProductInSite::class, 'ps')
			         ->getQuery()->getScalarResult() as $row) {
			$sites[$row['prod']][$row['site']] = $row;
		}

		foreach ($variants as $baseProdId => $prods) {
			if (!$prods) {
				continue;
			}

			$this->em->getConnection()->executeStatement("DELETE FROM eshop_catalog__category_product WHERE id_product IN (" . implode(', ', $prods) . ")");

			$ins = [];
			foreach ($categories[$baseProdId] ?? [] as $cat) {
				foreach ($prods as $prodId) {
					$ins[] = "($prodId, $cat)";
				}
			}

			if ($ins) {
				$this->em->getConnection()
					->executeStatement("INSERT INTO eshop_catalog__category_product (id_product, id_category) VALUES " . implode(', ', $ins));
			}

			$this->em->getConnection()->executeStatement("DELETE FROM eshop_catalog__product_in_site WHERE product_id IN (" . implode(', ', $prods) . ")");

			$ins = [];
			foreach ($sites[$baseProdId] as $site => $vals) {
				foreach ($prods as $prodId) {
					$ins[] = '(' . implode(', ', [
							$prodId,
							"'$site'",
							(int) $vals['isActive'],
							(is_null($vals['cat']) ? 'NULL' : (int) $vals['cat']),
						]) . ')';
				}
			}

			/** @phpstan-ignore-next-line */
			if ($ins) {
				$this->em->getConnection()
					->executeStatement("INSERT INTO eshop_catalog__product_in_site (product_id, site, is_active, category_id) VALUES " . implode(', ', $ins));
			}
		}
	}

	public function setVariantMasters(): void
	{
		$variants = [];
		$skip     = [];

		foreach ($this->em->getConnection()->fetchAllAssociative("SELECT product_id, variant_id, is_default 
				FROM eshop_catalog__product_variant ORDER BY is_default DESC") as $row) {
			if (array_key_exists($row['variant_id'], $skip)) {
				continue;
			}

			if ($row['is_default'] == 1) {
				unset($variants[$row['variant_id']]);
				$skip[$row['variant_id']] = $row['variant_id'];
				continue;
			}

			$variants[$row['variant_id']][] = $row['product_id'];
		}

		foreach ($variants as $variantId => $prods) {
			if (count($prods) === 1) {
				$this->em->getConnection()->executeStatement("DELETE FROM eshop_catalog__product_variant WHERE product_id = {$prods[0]} AND variant_id = '{$variantId}'");
				continue;
			}

			$master = min($prods);

			$this->em->getConnection()->executeStatement("UPDATE eshop_catalog__product_variant SET is_default = 1 WHERE product_id = {$master} AND variant_id = '{$variantId}'");

			foreach ($prods as $prod) {
				foreach ($this->langsService->getLangs(false) as $lang) {
					$this->cacheService->productCache->remove('product/' . $prod . '/' . $lang->getTag());
				}
			}
		}
	}
}
