<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use EshopCatalog\Model\Entities\Manufacturer;
use Nette\Caching\Cache;
use EshopCatalog\FrontModule\Model\Dao;
use Throwable;

class Manufacturers extends BaseFrontEntityService
{
	public const CACHE_NAMESPACE = 'manufacturers';

	protected $entityClass = Manufacturer::class;
	protected CacheService $cacheService;
	protected ?array $cManufacturers = null;
	protected array $cacheDep = [
		Cache::TAGS   => ['manufacturers'],
		Cache::EXPIRE => '1 week',
	];

	public function __construct(CacheService $cacheService)
	{
		$this->cacheService = $cacheService;
	}

	public function getCache(): Cache
	{
		if ($this->cache === null) {
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);
		}

		return $this->cache;
	}

	/**
	 * @return Dao\Manufacturer[]
	 * @throws Throwable
	 */
	public function getManufacturers(): array
	{
		if ($this->cManufacturers === null) {
			$this->cManufacturers = [];

			$locale = $this->translator->getLocale();
			$key    = 'manufacturers/' . $locale;

			$this->cManufacturers = $this->cacheService->defaultCache->load($key, function(&$dep) use ($locale) {
				$dep    = $this->cacheDep;
				$result = [];

				foreach ($this->getEr()->createQueryBuilder('m')
					         ->addSelect('mt')
					         ->leftJoin('m.texts', 'mt', 'WITH', 'mt.lang = :lang')
					         ->andWhere('m.isPublished = 1')
					         ->setParameter('lang', $locale)
					         ->getQuery()->getArrayResult() as $row) {
					$result[$row['id']] = $this->fillDao($row);
				}

				return $result;
			});
		}

		return $this->cManufacturers;
	}

	public function get(int $id): ?Dao\Manufacturer
	{
		return $this->getManufacturers()[$id] ?? null;
	}

	protected function fillDao(array $a): Dao\Manufacturer
	{
		$locale = $this->translator->getLocale();

		$manufacturer = new Dao\Manufacturer;
		$manufacturer->id = $a['id'];
		$manufacturer->name = $a['name'];
		$manufacturer->logo = $a['logo'];
		$manufacturer->description = $a['texts'][$locale]['description'];

		return $manufacturer;
	}

}