<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Helpers\DaoHelper;
use Core\Model\Sites;
use Doctrine\ORM\Query;
use EshopCatalog\FrontModule\Model\Dao;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\ProductTag;
use EshopCatalog\Model\Entities\Tag;
use EshopOrders\Model\Entities\PaymentSpedition;
use EshopOrders\Model\Entities\Spedition;
use EshopOrders\Model\Helpers\EshopOrdersCache;
use Nette\Caching\Cache;
use Nette\Utils\DateTime;
use Throwable;

class Tags extends BaseFrontEntityService
{
	public const CACHE_NAMESPACE = 'eshopCatalogTags';

	protected $entityClass = Tag::class;

	protected ?Cache           $productsCache    = null;
	protected Sites            $sites;
	protected EshopOrdersCache $eshopOrdersCache;
	protected ?array           $cTags            = null;
	protected ?float           $cFreeFrom        = null;
	protected ?array           $cTagsForProducts = null;
	protected ?array           $freeFromArr      = null;
	protected array            $cacheDep         = [
		Cache::TAGS   => [self::CACHE_NAMESPACE, Products::CACHE_NAMESPACE],
		Cache::EXPIRE => '1 week',
	];

	public function __construct(Sites $sites, EshopOrdersCache $eshopOrdersCache)
	{
		$this->sites            = $sites;
		$this->eshopOrdersCache = $eshopOrdersCache;
	}

	public function getCache(): Cache
	{
		if ($this->cache === null) {
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);
		}

		return $this->cache;
	}

	public function getProductsCache(): Cache
	{
		if ($this->productsCache === null) {
			$this->productsCache = new Cache($this->cacheStorage, Products::CACHE_NAMESPACE);
		}

		return $this->productsCache;
	}

	/**
	 * @param Dao\Product|Dao\Product[] $product
	 *
	 * @throws Throwable
	 */
	public function loadTagsToProduct(&$product): void
	{
		$products = is_array($product) ? $product : [$product];

		$tagForPrice  = Config::load('tagWhenPriceIsLowerThenRetail');
		$freeDelivery = $this->get('freeDelivery');

		foreach ($products as $prod) {
			$freeDeliveryMinimum = $this->getFreeDeliveryMinimumPrice($prod->disabledSpeditions ?: []);

			if ($tagForPrice && Config::load('product.allowRetailPrice') && $prod->getPrice() < $prod->getRetailPrice()) {
				$prod->tags[$tagForPrice] = $this->get($tagForPrice);
			}

			if ($freeDeliveryMinimum !== null && $freeDelivery && $prod->getPrice() >= $freeDeliveryMinimum) {
				$freeDelivery->isAuto       = true;
				$prod->tags['freeDelivery'] = $freeDelivery;
			}
		}
	}

	/**
	 * @param int|string $id
	 *
	 * @throws Throwable
	 */
	public function get($id): ?Dao\Tag
	{
		return $this->getAll()[$id] ?? null;
	}

	/**
	 * @return Dao\Tag[]
	 * @throws Throwable
	 */
	public function getAll(): array
	{
		if ($this->cTags === null) {
			$lang        = $this->translator->getLocale();
			$this->cTags = $this->getCache()->load('allTags/' . $lang, function(&$dep) use ($lang) {
				$dep  = [
					Cache::TAGS       => [self::CACHE_NAMESPACE],
					Cache::EXPIRATION => '1 month',
				];
				$data = [];

				foreach ($this->getEr()->createQueryBuilder('t')
					         ->select('t.id, t.type, t.image, tt.name, t.color, t.bgColor')
					         ->join('t.texts', 'tt', 'WITH', 'tt.lang = :lang')
					         ->setParameter('lang', $lang)
					         ->getQuery()
					         ->getArrayResult() as $row) {
					$tag                = $this->fillDao($row);
					$data[$row['type']] = $tag;
					$data[$row['id']]   = $tag;
				}

				return $data;
			});
		}

		return $this->cTags ?? [];
	}

	protected function fillDao(array $tag): Dao\Tag
	{
		/** @var Dao\Tag $dao */
		$dao            = DaoHelper::fillDaoFromArray($tag, new Dao\Tag);
		$dao->textColor = $tag['color'];
		$dao->isAuto    = false;

		return $dao;
	}

	protected function loadFreeFromArr(): array
	{
		if ($this->freeFromArr === null) {
			$siteIdent = $this->sites->getCurrentSite()->getIdent();

			$this->freeFromArr = $this->eshopOrdersCache->getCache()->load('freeFrom_' . $siteIdent, function(&$dep) {
				$dep = [
					Cache::Tags       => ['freeFrom', 'spedition', 'payment'],
					Cache::EXPIRATION => '1 week',
				];

				$arr = [];

				foreach ($this->em->getRepository(Spedition::class)->createQueryBuilder('s')
					         ->select('s.id, s.freeFrom')
					         ->innerJoin(PaymentSpedition::class, 'ps', Query\Expr\Join::WITH, 'ps.spedition = s.id')
					         ->innerJoin('ps.sites', 'sites', Query\Expr\Join::WITH, 'sites.ident = :ident')
					         ->where('s.freeFrom > 0')->andWhere('s.freeFrom IS NOT NULL')
					         ->setParameter('ident', $this->sites->getCurrentSite()->getIdent())
					         ->groupBy('s.id')
					         ->getQuery()->getArrayResult() as $row) {
					$arr[$row['id']] = (float) $row['freeFrom'];
				}

				return $arr;
			});
		}

		return $this->freeFromArr;
	}

	protected function getFreeDeliveryMinimumPrice(array $skipSpeditions = []): ?float
	{
		if (!empty($skipSpeditions)) {
			$min            = 99999999;
			$skipSpeditions = array_flip($skipSpeditions);
			foreach ($this->loadFreeFromArr() as $id => $val) {
				if (isset($skipSpeditions[$id])) {
					continue;
				}

				if ($val < $min) {
					$min = $val;
				}
			}

			return $min;
		}

		if ($this->cFreeFrom === null) {
			if ($this->loadFreeFromArr()) {
				$this->cFreeFrom = min($this->loadFreeFromArr());
			}
		}

		return $this->cFreeFrom;
	}

}
