<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Core\Model\Entities\Country;
use Core\Model\Helpers\Strings;
use DateTimeInterface;
use Doctrine\ORM\Mapping as ORM;
use Core\Model\Entities\TId;
use Nette\Utils\DateTime;

/**
 * @ORM\Table(name="eshop_catalog__product_price_history_country", indexes={
 *     @ORM\Index(name="byCountry", columns={"direction", "country", "currency_code", "product"}),
 * })
 * @ORM\Entity
 * @ORM\EntityListeners({"EshopCatalog\Model\Listeners\ProductPriceCountryHistoryListener"})
 */
class ProductPriceCountryHistory
{
	use TId;

	/**
	 * @ORM\ManyToOne(targetEntity="Product")
	 * @ORM\JoinColumn(name="product", referencedColumnName="id", onDelete="CASCADE")
	 */
	public Product $product;

	/**
	 * @ORM\ManyToOne(targetEntity="Core\Model\Entities\Country")
	 * @ORM\JoinColumn(name="country", referencedColumnName="id", nullable=false, onDelete="CASCADE")
	 */
	public Country $country;

	/**
	 * @ORM\Column(name="currency_code", type="string", length=255, nullable=true)
	 */
	public ?string $currency = null;

	/**
	 * @var double|string
	 * @ORM\Column(type="decimal", precision=10, scale=2, nullable=false)
	 */
	public $price;

	/**
	 * @ORM\Column(type="datetime", nullable=false, options={"default": "CURRENT_TIMESTAMP"})
	 */
	protected DateTimeInterface $created;

	/**
	 * @ORM\Column(type="smallint", nullable=false, options={"default": 0})
	 */
	public int $direction = 0;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public ?string $reason = null;

	public function __construct(
		Product $product,
		Country $country,
		?float  $price,
		float   $newPrice,
		?string $currency = null,
		?string $reason = null
	)
	{
		if (!$price) {
			$price = 0.0;
		}

		$this->product   = $product;
		$this->country   = $country;
		$this->price     = Strings::formatEntityDecimal($price);
		$this->created   = new DateTime;
		$this->direction = $newPrice > $price ? 1 : 0;
		$this->currency  = $currency;
		$this->reason    = $reason;
	}

	public function getCreated(): DateTimeInterface { return $this->created; }

}