<?php declare(strict_types = 1);

namespace EshopCatalog\Model;

use Contributte\Translation\Translator;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopOrders\Model\Entities\Order;
use Nette\Http\Url;
use Nette\Utils\ArrayHash;
use Nette\Utils\Json;
use Nette\Utils\Validators;
use Tracy\Debugger;

class Personalization
{
	protected ArrayHash $config;
	protected ProductsFacade $productsFacade;
	protected Translator $translator;

	public function __construct(array $config, ProductsFacade $productsFacade, Translator $translator)
	{
		$this->config = ArrayHash::from($config);
		$this->productsFacade = $productsFacade;
		$this->translator = $translator;
	}

	protected function getBaseUrl(): Url
	{
		return new Url($this->config->baseApiUrl);
	}

	/**
	 * @return array{response: bool|string, httpCode: int, error: string}
	 */
	protected function get(string $url): array
	{
		$ch = curl_init();

		curl_setopt($ch, CURLOPT_URL, $url);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_HEADER, false);
		curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'GET');
		curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 0);
		curl_setopt($ch, CURLOPT_TIMEOUT, 0);

		$response = curl_exec($ch);
		$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
		$error = curl_error($ch);

		curl_close($ch);

		return ['response' => $response, 'httpCode' => (int) $httpCode, 'error' => $error];
	}

	/**
	 * @return array{response: bool|string, httpCode: int, error: string}
	 */
	protected function post(string $url, array $data): array
	{
		$ch = curl_init();

		$headers = [
			'X-Token: ' . $this->config->apiToken,
			'Content-Type: application/json'
		];

		curl_setopt($ch, CURLOPT_URL, $url);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_HEADER, false);
		curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
		curl_setopt($ch, CURLOPT_POSTFIELDS, Json::encode($data));
		curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'POST');
		curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 0);
		curl_setopt($ch, CURLOPT_TIMEOUT, 0);

		$response = curl_exec($ch);
		$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
		$error = curl_error($ch);

		curl_close($ch);

		return ['response' => $response, 'httpCode' => (int) $httpCode, 'error' => $error];
	}

	/**
	 * @return int[]
	 */
	public function getAlternativeProductIds(int $forProductId): array
	{
		$url = $this->getBaseUrl();
		$url->setPath('/api/v1/recommended-products/alternatives/' . $forProductId);
		$url->setQueryParameter('size', $this->config->alternativeProducts->size);
		$url->setQueryParameter('lang', $this->translator->getLocale());
		$url->setQueryParameter('currency', 'czk');

		$result = $this->get((string) $url);
		$response = (string) $result['response'];
		$httpCode = $result['httpCode'];
		$error = $result['error'];

		if ($httpCode === 200) {
			$productIds = [];
			foreach (Json::decode($response) as $value) {
				if ($this->productsFacade->getProduct((int) $value->databaseId)) {
					$productIds[] = (int) $value->databaseId;
				}
			}

			return $productIds;
		}

		Debugger::log(Validators::isNone($error) ? 'Api not working' : $error);
		return [];
	}

	public function putOrder(Order $order): void
	{
		$ipRestrict = Config::load('personalization.ipRestrict');
		if (!Config::load('personalization.crossSellProducts.enable') || ($ipRestrict !== [] && !in_array($_SERVER['REMOTE_ADDR'], $ipRestrict))) {
			return;
		}

		$url = $this->getBaseUrl();
		$url->setPath('/api/v1/orders/put');

		$data = [];
		$data['id'] = (string) $order->getId();

		foreach ($order->getOrderItems() as $item) {
			$data['items'][] = [
				'productId' => (string) $item->getProductId(),
				'quantity'  => $item->getQuantity(),
			];
		}

		$this->post((string) $url, $data);
	}

	public function increaseCtr(int $fromProductId, int $productId): void
	{
		$url = $this->getBaseUrl();
		$url->setPath('/api/v1/products/increase-ctr');

		$this->post((string) $url, ['productId' => $productId, 'fromProductId' => $fromProductId]);
	}

	public function getCrossSellProducts(int $productId): array
	{
		$url = $this->getBaseUrl();
		$url->setPath('/api/v1/recommended-products/cross-sell/' . $productId);
		$url->setQueryParameter('size', $this->config->crossSellProducts->size);
		$url->setQueryParameter('lang', $this->translator->getLocale());
		$url->setQueryParameter('currency', 'czk');

		$result = $this->get((string) $url);
		$response = (string) $result['response'];
		$httpCode = $result['httpCode'];
		$error = $result['error'];

		if ($httpCode === 200) {
			$productIds = [];
			foreach (Json::decode($response) as $value) {
				if ($this->productsFacade->getProduct((int) $value->databaseId)) {
					$productIds[] = (int) $value->databaseId;
				}
			}

			return $productIds;
		}

		Debugger::log(Validators::isNone($error) ? 'Api not working' : $error);
		return [];
	}

}
