<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Lang\DefaultLang;
use Doctrine\ORM\Query;
use EshopCatalog\FrontModule\Model\Dao;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductTag;
use EshopCatalog\Model\Entities\Tag;
use EshopOrders\Model\Entities\Spedition;
use Navigations\Model\Entities\Navigation;
use Nette\Caching\Cache;
use Nette\Utils\ArrayHash;
use Nette\Utils\DateTime;

/**
 * Class Tags
 * @package EshopCatalog\FrontModule\Model
 */
class Tags extends BaseFrontEntityService
{
	/** @var Cache */
	protected $productsCache;

	const CACHE_NAMESPACE = 'eshopCatalogTags';

	protected $entityClass = Tag::class;

	/** @var DefaultLang @inject */
	public $defaultLang;

	protected $cTag = [];

	/** @var int */
	protected $cFreeFrom;

	/** @var array */
	protected $cacheDep = [
		Cache::TAGS    => [self::CACHE_NAMESPACE, Products::CACHE_NAMESPACE],
		Cache::EXPIRE  => '1 week',
		Cache::SLIDING => true,
	];

	public function getCache()
	{
		if ($this->cache === null)
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);

		return $this->cache;
	}

	public function getProductsCache()
	{
		if ($this->productsCache === null)
			$this->productsCache = new Cache($this->cacheStorage, Products::CACHE_NAMESPACE);

		return $this->productsCache;
	}

	/**
	 * @param Dao\Product $product
	 *
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function loadTagsToProduct(Dao\Product &$product)
	{
		$tagIds = $this->getIdsForProduct($product->getId());

		if ($tagIds) {
			foreach ($tagIds as $id) {
				$t = $this->get($id);
				if ($t)
					$product->tags[$t->type] = $t;
			}
		}

		$freeDelivery = $this->get('freeDelivery');
		if ($freeDelivery && $product->getPrice() >= $this->getFreeDeliveryMinimumPrice()) {
			$freeDelivery->isAuto = true;
			$product->tags['freeDelivery'] = $freeDelivery;
		}
	}

	/**
	 * @param int $id
	 *
	 * @return Dao\Category
	 */
	public function getIdsForProduct($productId)
	{
		$key = 'tags_p' . $productId;
		$now = (new DateTime())->format('Y-m-d H:i:00');

		$data = $this->getProductsCache()->load($key, function(&$dep) use ($productId, $now) {
			$dep                = $this->cacheDep;
			$dep[Cache::TAGS][] = 'product/' . $productId;

			return $this->getEr()->createQueryBuilder('t')
				->select('t.id, pt.validFrom, pt.validTo')
				->join(ProductTag::class, 'pt', 'WITH', 'pt.tag = t AND pt.product = :product')
				->andWhere('pt.validFrom IS NULL OR pt.validFrom >= :now')
				->andWhere('pt.validTo IS NULL OR pt.validTo <= :now')
				->setParameter('now', $now)
				->setParameter('product', $productId)
				->getQuery()->getResult();
		});

		$ids = [];
		foreach ($data as $row)
			if (($row['validFrom'] == null || $row['validFrom']->format('Y-m-d H:i:s') >= $now)
				&& ($row['validTo'] == null || $row['validTo']->format('Y-m-d H:i:s') <= $now))
				$ids[] = $row['id'];

		return $ids;
	}

	/**
	 * @param int|string $id
	 *
	 * @return Dao\Tag|null
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function get($id)
	{
		if (!isset($this->cTag[$id])) {
			$tag = $this->getEr()->createQueryBuilder('t')
				->select('t.id, t.type, t.image, tt.name')
				->leftJoin('t.texts', 'tt', 'WITH', 'tt.lang = :lang')
				->setParameter('lang', $this->defaultLang->locale);

			if (is_numeric($id))
				$tag->where('t.id = :id');
			else
				$tag->where('t.type = :id');
			$tag = $tag->setParameter('id', $id)->getQuery()->useResultCache(true, 60)
				->getOneOrNullResult(Query::HYDRATE_ARRAY);
			if ($tag) {
				$tag                    = $this->fillDao($tag);
				$this->cTag[$tag->id]   = $tag;
				$this->cTag[$tag->type] = $tag;
			}
		}

		return $this->cTag[$id];
	}

	/**
	 * @param array $tag
	 *
	 * @return Dao\Tag
	 */
	protected function fillDao($tag)
	{
		$t = (new Dao\Tag())
			->setId($tag['id'])
			->setType($tag['type'])
			->setName($tag['name'])
			->setIsAuto(false)
			->setImage($tag['image']);

		return $t;
	}

	/**
	 * TODO uložit do cache doprav
	 *
	 * @return int
	 */
	protected function getFreeDeliveryMinimumPrice()
	{
		if (!$this->cFreeFrom) {
			$v = $this->em->getRepository(Spedition::class)->createQueryBuilder('s')->select('s.freeFrom')
				->andWhere('s.freeFrom > 0')
				->setMaxResults(1)->orderBy('s.freeFrom', 'ASC')
				->getQuery()->useResultCache(true, 60)->getOneOrNullResult();

			$this->cFreeFrom = $v['freeFrom'];
		}

		return $this->cFreeFrom;
	}
}
