<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Products;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\AdminModule\Model\Features;
use EshopCatalog\AdminModule\Model\FeatureValues;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Entities\FeatureProduct;
use EshopCatalog\Model\Entities\FeatureValue;
use EshopCatalog\Model\Helpers\VariantsHelper;
use Nette\Caching\Cache;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\AdminModule\Model\Products;
use Nette\Utils\Json;

/**
 * Class ProductsFeatureForm
 * @package EshopCatalog\AdminModule\Components\Products
 */
class ProductsFeatureForm extends BaseControl
{
	/** @var array */
	public $products;

	/** @var string */
	private $ids;

	protected Products $productServices;

	protected Features $featuresService;

	protected FeatureValues $featureValuesService;

	protected VariantsHelper $variantsHelper;

	protected CacheService $cacheService;

	/** @var array */
	protected $features;

	public function __construct(Products       $products, Features $features, FeatureValues $featureValues,
	                            VariantsHelper $variantsHelper, CacheService $cacheService)
	{
		$this->productServices      = $products;
		$this->featuresService      = $features;
		$this->featureValuesService = $featureValues;
		$this->variantsHelper       = $variantsHelper;
		$this->cacheService         = $cacheService;
	}

	public function render()
	{
		$this->template->featuresJson = Json::encode($this->getFeatures());
		$this->template->render($this->getTemplateFile());
	}

	protected function attached($presenter): void
	{
		parent::attached($presenter);
		$feature = $this->getPresenter()->getHttpRequest()->getPost('feature');
		if ($feature)
			$this['form']['value']->setItems($this->getFeatures()['values'][$feature]);
	}

	/*******************************************************************************************************************
	 * ========================  COMPONENTS
	 */

	protected function createComponentForm()
	{
		$form = $this->createForm();

		$form->addHidden('produtcIds', $this->ids);

		$form->addSelect('feature', 'eshopCatalog.productFeature.feature', $this->getFeatures()['features']);
		$form->addSelect('value', 'eshopCatalog.productFeature.value', $this->getFeatures()['values'][key($this->getFeatures()['features'])] ?? []);
		$form->addCheckbox('keepSameFeatures', 'eshopCatalog.productFeature.keepSameFeatures');
		$form->addCheckbox('applyToVariants', 'eshopCatalog.productFeature.applyToVariants');

		$form->addSaveCancelControl('saveControl');

		$form['saveControl']['save']->setValidationScope(null);
		$form['saveControl']['saveAndClose']->setValidationScope(null);

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			$featureId = $values->feature;
			$valueId   = $this->getPresenter()->getHttpRequest()->getPost('value');

			$ids    = [];
			$exists = [];

			foreach ($this->products as $prod) {
				$ids[]                  = $prod->getId();
				$exists[$prod->getId()] = false;
			}

			if ($values->applyToVariants) {
				$ids = array_merge($ids, ...$this->variantsHelper->getProductVariants($ids));
			}

			if ($values->keepSameFeatures) {
				foreach ($this->em->getRepository(FeatureProduct::class)->createQueryBuilder('fp')
					         ->select('fp.id, IDENTITY(fp.product) as prod, IDENTITY(fp.featureValue) as val')
					         ->where('fp.product IN (:ids)')
					         ->andWhere('fp.featureValue = :val')
					         ->setParameters([
						         'ids' => $ids,
						         'val' => $valueId,
					         ])->getQuery()->getArrayResult() as $row) {
					$exists[$row['prod']] = $row['id'];
				}
			} else {
				foreach ($this->em->getRepository(FeatureProduct::class)->createQueryBuilder('fp')
					         ->select('fp.id, IDENTITY(fp.product) as prod, IDENTITY(fp.featureValue) as val')
					         ->innerJoin('fp.featureValue', 'fv', Join::WITH, 'fv.feature = :featureId')
					         ->where('fp.product IN (:ids)')
					         ->setParameters([
						         'ids'       => $ids,
						         'featureId' => $featureId,
					         ])->getQuery()->getArrayResult() as $row) {
					$exists[$row['prod']] = $row['id'];
				}
			}

			foreach ($ids as $id) {
				$exist = $exists[$id] ?? false;

				/** @var FeatureProduct $featureProduct */
				if ($exist === false) {
					$featureProduct = new FeatureProduct(
						$this->productServices->getReference($id),
						$this->featureValuesService->getReference($valueId)
					);
				} else {
					$featureProduct = $this->em->getReference(FeatureProduct::class, $exist);
					$featureProduct->setFeatureValue($this->featureValuesService->getReference((int) $valueId));
				}

				$this->em->persist($featureProduct);
			}

			$this->em->flush();
			$form->addCustomData('productIds', $values->produtcIds);

			$this->cacheService->clean('product', [
				Cache::TAGS => array_map(fn($id) => 'product/' . $id, explode('-', (string) $values->produtcIds)),
			]);

			$this->getPresenter()->flashMessageSuccess('eshopCatalog.productForm.productSaved');
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}

		return true;
	}

	/*******************************************************************************************************************
	 * ========================  GET / SET
	 */

	public function setProducts($ids)
	{
		$this->ids  = $ids;
		$productIds = explode('-', $ids);
		foreach ($productIds as $id) {
			$this->products[$id] = $this->em->getRepository(Product::class)->find($id);
		}

		if (!$this->products)
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}

	protected function getFeatures()
	{
		if (empty($this->features)) {
			$features = [];
			foreach ($this->featuresService->getEr()->createQueryBuilder('f')->select('f.id, ft.name')
				         ->join('f.featureTexts', 'ft', 'WITH', 'ft.lang = :lang')
				         ->setParameter('lang', $this->translator->getLocale())
				         ->where('f.isPublished = 1')
				         ->getQuery()->getResult() as $v)
				$features[$v['id']] = $v['name'];

			$values = [];
			foreach ($this->featureValuesService->getEr()->createQueryBuilder('fv')->select('fv.id, fvt.name, IDENTITY(fv.feature) as featureId')
				         ->join('fv.featureValueTexts', 'fvt', 'WITH', 'fvt.lang = :lang')
				         ->setParameter('lang', $this->translator->getLocale())
				         ->where('fv.isPublished = 1')
				         ->getQuery()->getResult() as $v)
				$values[$v['featureId']][$v['id']] = $v['name'];

			$this->features = ['features' => $features, 'values' => $values];
		}

		return $this->features;
	}
}
