<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model\Import;

use Core\AdminModule\Model\Sites;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\Model\AvailabilityService;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Availability;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductInSite;
use EshopCatalog\Model\Entities\ProductSupplier;
use EshopCatalog\Model\Entities\ProductTexts;
use Gallery\Model\Entities\AlbumText;
use Import\AdminModule\Model\ProcessingType\ProcessingType;
use Import\Model\Entities\Import;
use Import\Model\ImportDebugger;
use Core\Model\Entities\EntityManagerDecorator;
use Nette\Localization\ITranslator;
use Nette\Utils\Strings;
use Tracy\Debugger;

/**
 * Class ProcessPairing
 * @package EshopCatalog\AdminModule\Model\Import
 */
class ProcessPairing extends ProcessingType
{
	public string $title = 'eshopCatalog.import.processingType.pairing';

	public string $name = 'eshopCatalog.pairing';

	protected EntityManagerDecorator $em;

	protected ITranslator $translator;

	protected ImportHelper $importHelper;

	protected AvailabilityService $avService;

	protected Sites $sites;

	public array $lastProductsRun = [];

	public function __construct(EntityManagerDecorator $em, ITranslator $translator, ImportHelper $importHelper, Sites $sites, AvailabilityService $avService)
	{
		$this->em           = $em;
		$this->translator   = $translator;
		$this->importHelper = $importHelper;
		$this->sites        = $sites;
		$this->avService    = $avService;
	}

	/**
	 * @param Import $import
	 * @param array  $data
	 *
	 * @return array|string
	 * @throws \Exception
	 */
	public function run($import, $data)
	{
		$error             = null;
		Debugger::$showBar = false;

		$pairing = $import->data;
		$result  = [
			'created'  => 0,
			'updated'  => 0,
			'notFound' => 0,
			'message'  => '',
		];

		$availabilities = [];
		foreach ($this->em->getRepository(Availability::class)->createQueryBuilder('a')->select('a.id, a.ident, a.position')
			         ->orderBy('a.position')->getQuery()->getArrayResult() as $row)
			$availabilities[$row['id']] = [
				'id'       => $row['id'],
				'ident'    => $row['ident'],
				'position' => $row['position'],
			];
		$supplier            = $this->importHelper->getSupplier($import->getSyncOpt('supplier'));
		$syncBy              = $import->getSyncOpt('syncBy');
		$soldOutAvailability = $this->em->getConnection()->fetchColumn("SELECT id FROM eshop_catalog__availability WHERE ident = ?", [Availability::SOLD_OUT]);
		$syncBySupplier      = $syncBy === 'supplierCode';

		$lang               = $this->translator->getDefaultLocale();
		$exists             = [];
		$existsWithSupplier = [];
		$imported           = [];
		// Zjisteni existujicich produktu prirazenych dodavateli
		if ($syncBySupplier) {
			foreach ($this->em->getConnection()->fetchAllAssociative("SELECT p.id, ps.id_supplier, ps.code FROM eshop_catalog__product_supplier ps
				INNER JOIN eshop_catalog__product p on ps.id_product = p.id
				WHERE ps.id_supplier = ?", [$supplier->getId()]) as $v) {
				$exists[$v['code']]             = $v['id'];
				$existsWithSupplier[$v['code']] = $v['id'];
			}
		} else {
			$tmp1 = [];
			$tmp2 = [];
			foreach ($this->em->getRepository(Product::class)->createQueryBuilder('p')
				         ->select('p.id, IDENTITY(ps.supplier) as supplier, ps.code, pt.name, p.code1, p.' . $syncBy)
				         ->leftJoin('p.suppliers', 'ps', 'WITH', 'ps.supplier = :supplier')
				         ->innerJoin('p.productTexts', 'pt', Join::WITH, 'pt.lang = :lang')
				         ->setParameters([
					         'supplier' => $supplier->getId(),
					         'lang'     => $this->translator->getLocale(),
				         ])
				         ->groupBy('p.id')
				         ->getQuery()->getResult() as $v) {
				$syncByV = $v[$syncBy] ?: $v['name'];

				if (isset($tmp1[$syncByV]) || isset($tmp2[$syncByV]))
					$syncByV = $syncByV . '_' . $v['code1'];

				if ($v['supplier']) {
					$tmp1[$syncByV]                 = $v['id'];
					$existsWithSupplier[$v['code']] = $v['id'];
				} else
					$tmp2[$syncByV] = $v['id'];
			}

			$exists = $tmp1 + $tmp2;
		}

		// Srovnani podle synchronizacniho pole a vyfiltrovani zvolenych vyrobcu
		$tmp                  = [];
		$tmpForNotFound       = [];
		$allowedManufacturers = $pairing['allowedManufacturers']['value'] ?? null;
		if (count($allowedManufacturers) === 0)
			$allowedManufacturers = null;

		if (empty($data['data']))
			return $result;

		foreach ($data['data'] as $row) {
			$skipThis = false;

			foreach (['ean', 'code1', 'code2'] as $col) {
				$colVal = $import->getSyncOpt('only' . ucfirst($col) . 'Start');
				if (!$colVal || $colVal == '')
					continue;

				$skip = true;
				foreach (explode(',', $colVal) as $ms) {
					if (Strings::startsWith($row[$pairing[$col]['key']], $ms)) {
						$skip = false;
						break;
					}
				}

				if ($skip)
					$skipThis = true;
			}

			if ($skipThis)
				continue;

			if (!$allowedManufacturers || in_array($row[$pairing['allowedManufacturers']['key']], $allowedManufacturers)) {
				$tmp[]                                                   = $row;
				$tmpForNotFound[(string) $row[$pairing['code1']['key']]] = $row;
			}
		}

		$data['data'] = $tmp;
		$tmp          = null;

		// Nenalezení v importu nastavit počet ks na 0
		$notFound       = array_diff_key($existsWithSupplier, $tmpForNotFound);
		$tmpForNotFound = null;
		if ($notFound) {
			foreach (array_chunk($notFound, 100) as $nfs) {
				$notFoundProds = [];
				foreach ($this->em->getRepository(Product::class)->createQueryBuilder('p')
					         ->join('p.suppliers', 'ps')
					         ->where('p.id IN (:ids)')
					         ->setParameter('ids', $nfs)
					         ->getQuery()->getResult() as $row) {
					$notFoundProds[$row->getId()] = $row;
					$result['notFound']++;
				}

				$inValues = implode(',', $nfs);
				$this->em->getConnection()->executeStatement('UPDATE eshop_catalog__product_supplier SET quantity = 0 WHERE id_product IN (' . $inValues . ') AND id_supplier = ' . $supplier->getId());

				foreach ($notFoundProds as $prod) {
					$this->avService->updateAvailabilityByQuantity($prod);
					$this->em->persist($prod);
				}
				$this->em->flush();
			}

			$notFoundProds = null;
			$notFound      = null;
		}

		$chunks = array_chunk($data['data'], 25, true);
		$data   = null;

		$total            = 0;
		$totalLimit       = (int) Config::load('importTotalLimit', 20000);
		$galleries        = [];
		$entitiesForClean = [];
		$updatedProducts  = [];

		foreach ($chunks as $chunk) {
			$this->em->beginTransaction();
			try {
				$i        = 0;
				$supplier = $this->importHelper->getSupplier($import->getSyncOpt('supplier'));
				foreach ($chunk as $rowKey => $row) {
					$exist = false;

					// Pokud neexistuje parovaci klic, tak se pouzije ean
					if (!$row[$pairing[$syncBy]['key']])
						$row[$pairing[$syncBy]['key']] = $row[$pairing['ean']['key']];

					$syncByKey   = Strings::trim((string) $row[$pairing[$syncBy]['key']]);
					$pairingName = array_values($pairing['name']['key'])[0] ?? '';
					$productId   = null;

					if (isset($existsWithSupplier[$row[$pairing['supplierCode']['key']]]))
						$productId = $existsWithSupplier[$row[$pairing['supplierCode']['key']]];
					else if (isset($exists[$syncByKey . '_' . $row[$pairing['code1']['key']]]))
						$productId = $exists[$syncByKey . '_' . $row[$pairing['code1']['key']]];
					else if (isset($exists[$syncByKey]))
						$productId = $exists[$syncByKey];
					else if (isset($exists[$row[$pairingName]]))
						$productId = $exists[$row[$pairingName]];
					else if (isset($exists[$row[$pairingName] . '_' . $row[$pairing['code1']['key']]]))
						$productId = $exists[$row[$pairingName] . '_' . $row[$pairing['code1']['key']]];

					if ($productId) {
						/** @var Product $product */
						$product      = $this->em->getRepository(Product::class)->find($productId);
						$productTexts = $product->getTexts()->toArray();
						$exist        = true;

						if (isset($updatedProducts[$product->getId()]))
							continue;
						$updatedProducts[$product->getId()] = true;
					}

					if (!$exist) {
						if ($import->getSyncOpt('onlyUpdate') == 1)
							continue;

						$product      = new Product();
						$productTexts = $product->getTexts()->toArray();
					}

					/** @var ProductSupplier $productSupplier */
					$productSupplier = $exist
						? ($this->em->getRepository(ProductSupplier::class)->createQueryBuilder('ps')
							->where('ps.product = :prod')
							->andWhere('ps.supplier = :supp')
							->setParameters([
								'prod' => $product->getId(),
								'supp' => $supplier->getId(),
							])->getQuery()->setMaxResults(1)->getOneOrNullResult())
						: null;

					if (!$productSupplier) {
						$productSupplier       = new ProductSupplier($product, $supplier);
						$productSupplier->code = $this->fixString((string) $row[$pairing['supplierCode']['key']]);
					}

					if ($productSupplier && $pairing['supplierCode']['key'] && (!$exist || $pairing['supplierCode']['applyOn'] != 'new'))
						$productSupplier->code = $this->fixString((string) $row[$pairing['supplierCode']['key']]);

					if ($productSupplier && $productSupplier->code === null)
						continue;

					$this->em->persist($product);
					$this->em->flush($product);

					foreach (['width', 'height', 'depth', 'weight'] as $v) {
						if (!isset($pairing[$v]) || $exist && $pairing[$v]['applyOn'] == 'new')
							continue;

						$modify = $pairing[$v]['modifyInt'];
						$value  = (float) str_replace(',', '.', $this->fixString($row[$pairing[$v]['key']]));

						if ($modify)
							$value = round($value * (float) $modify, 0);

						$product->$v = (int) $value;
					}

					$pVals = ['ean', 'code1', 'code2', 'vatRate', 'price' => 'priceVat', 'purchasePrice',
						'retailPrice', 'recyclingFee'];
					foreach ($pVals as $k => $v) {
						$k = is_string($k) ? $k : $v;

						if (!isset($pairing[$v]) || $exist && $pairing[$v]['applyOn'] == 'new')
							continue;

						if ($v == 'vatRate') {
							$product->$k = $this->importHelper->getVatRate((int) $this->fixString($row[$pairing[$v]['key']]) ?: $pairing[$v]['fixedValue']);
						} else if ($v === 'ean') {
							$product->ean = $this->importHelper->validateEAN((string) ($this->fixString($row[$pairing[$v]['key']]) ?: $pairing[$v]['fixedValue']));
						} else if (in_array($k, ['price', 'retailPrice'])) {
							$modify          = $pairing[$v]['modifyInt'];
							$priceWithoutVat = $pairing[$v]['withoutVat'] ? true : false;
							$price           = (float) $this->fixString($row[$pairing[$v]['key']]);
							if ($modify)
								$price = round($price * (float) $modify, 0);
							if ($priceWithoutVat)
								$price = round($price * (1 + $product->getVateRate()->getModifier()), 0);

							$this->importHelper->setProductPrice($product, $price, $k);
						} else {
							if (isset($pairing[$v]['key'])) {
								$product->$k = $this->fixString($row[$pairing[$v]['key']]);
							} else if (isset($pairing[$v]['fixedValue']))
								$product->$k = $this->fixString($pairing[$v]['fixedValue']);
						}
					}

					$pValsBool = ['isPublished', 'isAssort'];
					foreach ($pValsBool as $k => $v) {
						$k = is_string($k) ? $k : $v;

						if (!isset($pairing[$v]) || $exist && $pairing[$v]['applyOn'] == 'new')
							continue;

						if (isset($row[$pairing[$v]['key']]))
							if (!isset($pairing[$v]['key']))
								$product->$k = 0;
							else {
								$val         = (float) $this->fixString($row[$pairing[$v]['key']]);
								$compare     = (float) $pairing[$v]['value'];
								$product->$k = $this->importHelper->checkValueIs($pairing[$v]['valueIs'], $val, $compare) ? 1 : 0;
							}
						else if (isset($pairing[$v]['fixedValue']))
							$product->$k = (int) $this->fixString($pairing[$v]['fixedValue']);
					}

					$ptVals       = ['name', 'description', 'shortDescription'];
					$textsUpdated = false;
					foreach ($ptVals as $k => $v) {
						$k = is_string($k) ? $k : $v;

						if (!isset($pairing[$v]) || $exist && $pairing[$v]['applyOn'] == 'new')
							continue;

						$lastText = null;
						foreach ($pairing[$v]['key'] as $kLang => $vKey) {
							$text = $productTexts[$kLang];

							if (!$text) {
								$text                 = new ProductTexts($product, $kLang);
								$productTexts[$kLang] = $text;
							}

							$textVal = '';
							if ($vKey)
								$textVal = $this->fixString($row[$vKey]);
							else if (isset($pairing[$v]['fixedValue']))
								$textVal = $this->fixString($pairing[$v]['fixedValue']);

							if (!$textVal && $lastText && $lastText->$k)
								$textVal = $lastText->$k;

							if ($k == 'shortDescription')
								$text->setShortDescription((string) $textVal);
							//							else
							//								$text->$k = (string) $textVal;

							if ($v == 'name')
								$text->setName((string) $textVal);

							if ($v == 'name' && $text->$k) {
								$text->setAlias((string) $text->$k);
							}

							if ($v == 'description' && $text->description && !$text->shortDescription) {
								$text->setShortDescription((string) $productTexts->description);
							}
							$lastText = $text;
						}

						$textsUpdated = true;
					}

					if ($textsUpdated) {
						$this->em->persist($product);
						foreach ($productTexts as $pt) {
							$this->em->persist($pt);
						}
					}

					// TODO dodělat nefixní hodnotu
					//				if (isset($pairing['category']['key']) && $row[$pairing['category']['key']] && false)
					//					$productSupplier->quantity = $this->importHelper->getManufacturer($this->fixString($row[$pairing['quantity']['key']]));

					if (isset($pairing['manufacturer']) && (!$exist || $pairing['manufacturer']['applyOn'] != 'new'))
						if (isset($row[$pairing['manufacturer']['key']])) {
							$man = $this->fixString($row[$pairing['manufacturer']['key']]);
							if ($man && $man != '') {
								$product->setManufacturer($this->importHelper->getManufacturer($man));
							} else
								$product->setManufacturer(null);
						} else if (isset($pairing['manufacturer']['fixedValue']))
							$product->setManufacturer($this->importHelper->getManufacturer($pairing['manufacturer']['fixedValue']));

					//					if (isset($pairing['inStock']) && (!$exist || $pairing['inStock']['applyOn'] != 'new'))
					//						$product->inStock = $product->quantity > 0;

					if (!$exist || $pairing['category'] && $pairing['category']['applyOn'] != 'new') {
						$baseCat = $this->importHelper->findCategory($import->getSyncOpt('baseCategory'));

						if ($baseCat) {
							$this->em->getConnection()->executeQuery("INSERT IGNORE INTO eshop_catalog__product_in_site (product_id, site, is_active, category_id) VALUES (?, ?, ?, ?)", [
								$product->getId(),
								$this->sites->get($baseCat->getRoot()->getCategoryText()->alias)->getIdent(),
								0,
								$baseCat->getId(),
							]);
						}

						if (isset($pairing['category']['fixedValue'])) {
							$cat = $this->importHelper->findCategory($pairing['category']['fixedValue']);
							if ($cat)
								$product->addCategoryProduct($this->importHelper->getCategoryProduct($product, $cat));
						} else {
							$pCat       = $pairing['category'];
							$importCats = null;

							if (isset($row[$pCat['key']])) {
								if ($pCat['separator'])
									$importCats = explode($pCat['separator'], $row[$pCat['key']]);
								else
									$importCats = [$row[$pCat['key']]];

								if ($pCat['deepSeparator']) {
									foreach ($importCats as $icK => $icV) {
										if (is_array($icV)) {
											foreach ($icV as $icVV) {
												$importCats[] = explode($pCat['deepSeparator'], $icVV);
											}
										} else {
											$importCats[$icK] = explode($pCat['deepSeparator'], $icV);
										}
									}
								}
							}

							if ($baseCat && $importCats) {
								foreach ($importCats as $ic) {
									$deepC = $this->importHelper->getDeepCategory($baseCat, $ic, $lang);
									if ($deepC)
										$product->addCategoryProduct($this->importHelper->getCategoryProduct($product, $deepC));
								}
							}
						}
					}

					// Parametry
					if (isset($pairing['params']['key']) && (!$exist || $pairing['params']['applyOn'] != 'new')) {
						$params = [];
						foreach ($pairing['params']['key'] as $ik) {
							if (is_array($row[$ik]))
								foreach ($row[$ik] as $ikk => $ikv)
									$params[$ikk] = $ikv;
							elseif ($row[$ik])
								$params[$ik] = $row[$ik];
						}

						if ($params)
							$this->importHelper->parametersProccess($product, $params, $lang);
					}

					// Dostupnost
					$setQuantity = true;
					$quantity    = $pairing['quantity']['key'] ? 0 : null;

					if (isset($row[$pairing['quantity']['key']]))
						$quantity = $this->fixString($row[$pairing['quantity']['key']]);
					else if (isset($pairing['quantity']['fixedValue']))
						$quantity = $this->fixString($pairing['quantity']['fixedValue']);

					if ($quantity && !is_numeric($quantity)) {
						$quantity = filter_var($quantity, FILTER_SANITIZE_NUMBER_INT);
					}

					// Pokud není nastaven počet skladem, zkontroluje se hodnota
					if ($pairing['inStock']['key'] && (!$exist || $pairing['inStock']['applyOn'] != 'new')) {
						$inStock  = false;
						$iv       = $row[$pairing['inStock']['key']];
						$ic       = $pairing['inStock']['value'] !== '' ? $pairing['inStock']['value'] : $pairing['inStock']['custom'];
						$inStock  = $this->importHelper->checkValueIs($pairing['inStock']['valueIs'], $iv, $ic);
						$quantity = 0;
						if ($inStock)
							$quantity = $pairing['quantity']['fixedValue'] ?? 1;
					} else if ($pairing['idAvailability']['key'] && (!$exist || $pairing['idAvailability']['applyOn'] != 'new')) {
						$inStock  = false;
						$iv       = $row[$pairing['idAvailability']['key']];
						$ic       = $pairing['idAvailability']['value'] !== '' ? $pairing['idAvailability']['value'] : $pairing['idAvailability']['custom'];
						$inStock  = $this->importHelper->checkValueIs($pairing['idAvailability']['valueIs'], $iv, $ic);
						$quantity = 0;
						if ($inStock)
							$quantity = $row[$pairing['idAvailability']['key']] ?? 1;
					}

					if ($productSupplier && $setQuantity && $quantity !== null) {
						$quantity = $this->importHelper->extractInt($quantity);

						$productSupplier->quantity = (int) $quantity;
						$setQuantity               = false;
					}

					if ($productSupplier && isset($pairing['idAvailability']['inStock']) && $pairing['idAvailability']['inStock']) {
						$productSupplier->availabilityAfterSoldOut = $this->em->getReference(Availability::class, (int) $pairing['idAvailability']['inStock']);
					}

					//					// Dostupnost 2
					//					$compare = $pairing['idAvailability']['value'];
					//					if ($pairing['idAvailability']['custom'])
					//						$compare = $pairing['idAvailability']['custom'];
					//					// Pokud je skladem tak se nastavi skladem
					//					if ($pairing['idAvailability']['inStock']
					//						&& $this->importHelper->checkValueIs($pairing['idAvailability']['valueIs'], $row[$pairing['idAvailability']['key']], $compare)) {
					//						// Jeste kontrola - produkt nesmi byt skladem a mit nastaveno skladem
					//						$selAvailability = $availabilities[$pairing['idAvailability']['inStock']] ?? null;
					//						//tady
					//						if ($selAvailability && $product->quantity <= 0 && (!$product->getAvailability() || $product->getAvailability() && $product->getAvailability()->getPosition() > $selAvailability['position']))
					//							$product->setAvailability($this->em->getReference(Availability::class, $pairing['idAvailability']['inStock']));
					//					} else if ($pairing['idAvailability']['soldOut']) {
					//						// Jinak zkontrolovat sklad a nastavit
					//						$maxSupplierStock = (int) $this->em->getConnection()
					//							->fetchColumn("SELECT max(quantity) FROM eshop_catalog__product_supplier WHERE id_product = ?", [$product->getId()]);
					//						if ($maxSupplierStock === 0 && $product->quantity <= 0)
					//							$product->setAvailability($this->em->getReference(Availability::class, $pairing['idAvailability']['soldOut']));
					//					}

					if (isset($pairing['variant']) && isset($row[$pairing['variant']['key']]) && $row[$pairing['variant']['key']] !== '') {
						if ($exist && $pairing['variant']['applyOn'] !== 'new' || !$exist) {
							$this->importHelper->addToVariant($row[$pairing['variant']['key']], $product->getId());
						}
					}

					$this->downloadGallery($row, $lang, $pairing, $exist, $product, $productTexts, $galleries);

					if ($productSupplier) {
						if (!$productSupplier->getSupplier())
							$productSupplier->setSupplier($supplier);
						$this->em->persist($productSupplier);
						$entitiesForClean[] = $productSupplier;
					}

					if (!$product->getAvailability())
						$product->setAvailability($this->em->getReference(Availability::class, $soldOutAvailability));

					$product->validateCreated();
					$this->avService->updateAvailabilityByQuantity($product);

					if ($product) {
						if ($productSupplier) {
							$this->em->persist($productSupplier);
							$entitiesForClean[] = $productSupplier;
						}
						$this->em->persist($product);
						$entitiesForClean[] = $product;
					}
					if ($productTexts) {
						foreach ($productTexts as $pt) {
							$this->em->persist($pt);
							$entitiesForClean[] = $pt;
						}
					}
					if ($exist)
						$result['updated']++;
					else
						$result['created']++;
					$i++;
					$total++;

					if (!$exist)
						$updatedProducts[$product->getId()] = true;

					if ($i == 20) {
						$i = 0;
					}

					if ($total >= $totalLimit) {
						break;
					}

					$this->em->flush();
					$this->importHelper->moveGalleryImage($galleries);
					$galleries = [];
				}

				$this->em->flush();
				if ($this->em->getConnection()->isTransactionActive())
					$this->em->commit();
				$this->em->clear();
				$supplier = $this->importHelper->getSupplier($import->getSyncOpt('supplier'));
				$this->importHelper->moveGalleryImage($galleries);
				$galleries                   = [];
				$this->importHelper->onError = [];
				if ($total >= $totalLimit)
					break;
			} catch (\Exception $e) {
				if ($this->em->getConnection()->isTransactionActive())
					$this->em->rollback();
				Debugger::log($e, 'import');
				$result['message'] = $e->getMessage();
				if (isset($product))
					$result['product'] = $product->getId() . ' / ' . $product->code1;
				$error = $e;

				foreach ($this->importHelper->onError as $onError)
					$onError();
				break;
			}
		}
		Debugger::log($result, 'import');

		return $result;
	}

	/**
	 * TODO udělat globální třídu
	 *
	 * @param $str
	 *
	 * @return string
	 */
	protected function fixString($str)
	{
		if (!is_string($str))
			return $str;
		$str = htmlspecialchars_decode($str, ENT_QUOTES);
		$str = html_entity_decode($str);
		$str = Strings::trim($str);

		return $str;
	}

	public function downloadGallery(&$row, &$lang, &$pairing, &$exist, &$product, &$productTexts, &$galleries)
	{
		if (ImportDebugger::$allowImagesDownload) {
			if (isset($pairing['images']) && (!$exist || $pairing['images']['applyOn'] != 'new')) {
				$gallery = $this->importHelper->getGallery($product);

				foreach ($gallery->getImages()->toArray() as $k => $img) {
					if (!file_exists($img->getFile())) {
						$this->em->remove($img);
						$gallery->getImages()->remove($k);
					}
				}

				$galleryText = $gallery->getText($lang);
				if (!$galleryText) {
					if ($gallery->getId())
						$galleryText = $this->em->getRepository(AlbumText::class)->createQueryBuilder('at')
							->where('at.album = :album')->andWhere('at.lang = :lang')
							->setParameters([
								'album' => $gallery->getId(),
								'lang'  => $lang,
							])->getQuery()->getOneOrNullResult();

					if (!$galleryText)
						$galleryText = new AlbumText($gallery, $lang);
				}

				$galleryText->title = $productTexts->name;
				$this->em->persist($galleryText);

				// Zjištění prefixu pokud je nastaven
				$urlPrefix = (string) ($pairing['imagesUrlPrefix']['key'] ?? $pairing['imagesUrlPrefix']['fixedValue']);

				$product->setGallery($gallery);
				$galleries[] = $gallery;

				// Získání obrázků z importu a přidání do galerie
				foreach ($pairing['images']['key'] as $k) {
					foreach ($this->importHelper->getImages($row[$k], $urlPrefix) as $img) {
						$img = $this->parseImgUrl($img, $row);
						$this->importHelper->addImage($gallery, $img);
					}
				}
			}
		}
	}

	protected function parseImgUrl($img, array $row)
	{
		if (!Strings::startsWith($img, 'http') && !Strings::startsWith($img, 'ftp'))
			$img = 'http://' . $img;

		return $img;
	}
}
