<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Presenters;

use EshopCatalog\AdminModule\Components\Products\IProductsFeatureFormFactory;
use EshopCatalog\AdminModule\Components\Products\IProductFormFactory;
use EshopCatalog\AdminModule\Components\Products\IProductsCategoryFormFactory;
use EshopCatalog\AdminModule\Components\Products\IProductsGridFactory;
use EshopCatalog\AdminModule\Components\Products\IProductsSpeditionsFormFactory;
use EshopCatalog\AdminModule\Components\Products\IProductTagsFormFactory;
use EshopCatalog\AdminModule\Components\Products\IRelatedProductsFormFactory;
use EshopCatalog\AdminModule\Components\Products\ISetAsVariantForFactory;
use EshopCatalog\AdminModule\Components\Products\ProductTagsForm;
use EshopCatalog\AdminModule\Components\Products\RelatedProductsForm;
use EshopCatalog\AdminModule\Components\Products\SetAsVariantsFor;
use EshopCatalog\Model\Entities\Product;
use Core\Model\UI\Form\BaseForm;
use Nette\Application\BadRequestException;

/**
 * Class ProductsPresenter
 * @package EshopCatalog\AdminModule\Presenters
 */
class ProductsPresenter extends BasePresenter
{

	/*******************************************************************************************************************
	 * ==================================   Actions
	 */

	public function actionDefault()
	{
		$this->setTitle($this->translator->translate('eshopCatalog.title.products'));
		$this->setHeader('eshopCatalog.title.products', 'fa fa-list');

		$this->getSession('productForm')->remove();

		$this['navigation']->setData(['header' => [
			[
				'title' => 'eshopCatalog.menu.addProduct',
				'link'  => 'Products:add',
				'ico'   => 'plus',
			],
		]]);
	}

	public function actionAdd()
	{
		$this->setHeader('eshopCatalog.title.newProduct', 'fa fa-list-alt');
	}

	public function actionEdit($id)
	{
		/** @var Product $product */
		$product = $this->em->getRepository(Product::class)->find($id);

		if (!$product)
			throw new BadRequestException();

		$title = 'editProduct';
		if ($product->isVariant()) {
			if ($product->isVariant()->isDefault)
				$title = 'editDefaultProductVariant';
			else
				$title = 'editProductVariant';
		}

		$this->setHeader(
			$this->t('eshopCatalog.title.' . $title) . ': ' . $product->getText($this->translator->getLocale())->name,
			'fa fa-list-alt',
			'(ID: ' . $product->getId() . ')',
		);
	}

	public function actionEditCategories($ids)
	{
		$this->setHeader('eshopCatalog.title.editProductsCategories', 'fa fa-list-alt');
	}

	public function actionEditSpeditions($ids)
	{
		$this->setHeader('eshopCatalog.title.editSpeditions', 'fa fa-list-alt');
	}

	public function actionEditFeatures($ids)
	{
		$this->setHeader('eshopCatalog.title.editProductsFeatures', 'fa fa-list-alt');
	}

	public function actionClose($id)
	{
		$this->redirect('Default:');
	}

	public function actionAssignRelatedProducts($ids)
	{
		$this->setHeader('eshopCatalog.title.assignRelatedProducts', 'fa fa-project-diagram');
	}

	public function actionSetTags($ids)
	{
		$this->setHeader('eshopCatalog.title.setTags', 'fa fa-list-alt');
	}

	public function actionSetAsVariantsFor($ids)
	{
		$this->setHeader('eshopCatalog.title.setAsVariantsFor');
	}

	/*******************************************************************************************************************
	 * ==================================   Components
	 */

	protected function createComponentProductForm(IProductFormFactory $product)
	{
		$control = $product->create();
		if ($this->getParameter('id'))
			$control->setProduct($this->getParameter('id'));

		$control['form']->onSuccessSave[]         = function(BaseForm $form) use ($control) {
			$this->redirect('Products:edit', $form->getCustomData('productId'));
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) use ($control) {
			$this->redirect('Products:default');
		};
		$control['form']->onCancel[]              = function(BaseForm $form) use ($control) {
			$this->redirect('Products:default');
		};

		return $control;
	}

	protected function createComponentProductsGrid(IProductsGridFactory $factory)
	{
		return $factory->create();
	}

	protected function createComponentProductsCategoryForm(IProductsCategoryFormFactory $factory)
	{
		$control = $factory->create();
		if ($this->getParameter('ids'))
			$control->setProducts($this->getParameter('ids'));

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this->redirect('Products:editCategories', $form->getCustomData('productIds'));
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->redirect('Products:default');
		};
		$control['form']->onCancel[]              = function(BaseForm $form) {
			$this->redirect('Products:default');
		};

		return $control;
	}

	protected function createComponentProductsSpeditionsForm(IProductsSpeditionsFormFactory $factory)
	{
		$control = $factory->create();
		if ($this->getParameter('ids'))
			$control->setProducts($this->getParameter('ids'));

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this->redirect('Products:editSpeditions', $form->getCustomData('productIds'));
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->redirect('Products:default');
		};
		$control['form']->onCancel[]              = function(BaseForm $form) {
			$this->redirect('Products:default');
		};

		return $control;
	}

	protected function createComponentProductsFeatureForm(IProductsFeatureFormFactory $factory)
	{
		$control = $factory->create();
		if ($this->getParameter('ids'))
			$control->setProducts($this->getParameter('ids'));

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this->redirect('Products:editFeatures', $form->getCustomData('productIds'));
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->redirect('Products:default');
		};
		$control['form']->onCancel[]              = function(BaseForm $form) {
			$this->redirect('Products:default');
		};

		return $control;
	}

	/**
	 * @param IRelatedProductsFormFactory $factory
	 *
	 * @return RelatedProductsForm
	 */
	protected function createComponentRelatedProductsForm(IRelatedProductsFormFactory $factory): RelatedProductsForm
	{
		$control = $factory->create();
		if ($this->getParameter('ids'))
			$control->setProducts($this->getParameter('ids'));

		$control['form']->onSuccess[] = function(BaseForm $form) {
			$this->redirect('Products:default');
		};

		return $control;
	}

	protected function createComponentProductTagsForm(IProductTagsFormFactory $factory): ProductTagsForm
	{
		$control = $factory->create();

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this->redirect('Products:setTags', $form->getCustomData('productIds'));
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->redirect('Products:default');
		};
		$control['form']->onCancel[]              = function(BaseForm $form) {
			$this->redirect('Products:default');
		};

		return $control;
	}

	protected function createComponentSetAsVariantForForm(ISetAsVariantForFactory $factory): SetAsVariantsFor
	{
		$control = $factory->create();

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this->redirect('Products:setAsVariantsFor', $form->getCustomData('productIds'));
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->redirect('Products:default');
		};
		$control['form']->onCancel[]              = function(BaseForm $form) {
			$this->redirect('Products:default');
		};

		return $control;
	}
}
