<?php declare(strict_types = 1);

namespace EshopCatalog\CronModule\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\Strings;
use Core\Model\Images\ImagePipe;
use Core\Model\Sites;
use Currency\Model\Currencies;
use Currency\Model\Exchange;
use EshopCatalog\FrontModule\Model\AvailabilityService;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\Model\Entities\Availability;
use EshopCatalog\Model\Event\ProductsFeedEvent;
use Nette\Http\Request;
use Nette\Utils\Json;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class AdsProductsFeedSubscriber implements EventSubscriberInterface
{
	protected EntityManagerDecorator $em;

	protected Categories $categories;

	protected Sites $sites;

	protected Request $httpRequest;

	protected AvailabilityService $availabilityService;

	protected Currencies $currencies;

	protected Exchange $exchange;

	protected ImagePipe $imagePipe;

	public function __construct(EntityManagerDecorator $em, Categories $categories, Sites $sites, Request $httpRequest,
	                            AvailabilityService    $availabilityService, Currencies $currencies, Exchange $exchange,
	                            ImagePipe              $imagePipe)
	{
		$this->em                  = $em;
		$this->categories          = $categories;
		$this->sites               = $sites;
		$this->httpRequest         = $httpRequest;
		$this->availabilityService = $availabilityService;
		$this->currencies          = $currencies;
		$this->exchange            = $exchange;
		$this->imagePipe           = $imagePipe;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopCatalog.productsFeed'             => 'productsFeed',
			'eshopCatalog.productsFeedAvailability' => 'productsFeedAvailability',
			'eshopCatalog.categoriesFeed'           => 'categoriesFeed',
		];
	}

	public function productsFeed(ProductsFeedEvent $event): void
	{
		if ($event->customerName === 'ads') {
			$event->outputFormat = 'csv';

			$fp = fopen('php://memory', 'wb+');
			fputcsv($fp, ['Page URL', 'Custom label'], ',');
			foreach ($event->data as $row) {
				if ($row->availabilityIdent !== Availability::IN_STOCK)
					continue;

				$label = [Strings::webalize((string) $row->category[0])];

				fputcsv($fp, [
					$row->link,
					implode('-', $label),
				], ',');
			}

			rewind($fp);
			$event->data = [
				'output' => stream_get_contents($fp),
			];

			fclose($fp);
		} else if ($event->customerName === 'ads-lp') {
			$event->outputFormat = 'csv';

			$fp = fopen('php://memory', 'wb+');

			$storeCode       = $this->httpRequest->getQuery('store');
			$targetCurrency  = $this->httpRequest->getQuery('currency');
			$defaultCurrency = $this->currencies->getDefaultCode();

			$ranks = [];
			$i     = 1;
			foreach ($this->em->getConnection()->fetchAllAssociative("SELECT id FROM eshop_catalog__product WHERE `position` IS NOT NULL ORDER BY `position` ASC") as $row) {
				$ranks[$row['id']] = $i;
				$i++;
			}

			fputcsv($fp, ['Rank', 'ID', 'Title', 'Description', 'Item URL', 'Image URL', 'Price',
				'Store code', 'Sale price'], ',');
			foreach ($event->data as $row) {
				if ($row->available <= 0) {
					continue;
				}

				$av    = $this->availabilityService->getByIdent($row->availabilityIdent);
				$image = $row->imageLink;
				if (!$av || !$image) {
					continue;
				}

				$image = explode('thumbs', $image);
				$image = $image[1];

				$price     = $row->priceOriginalCzk ?: $row->priceCzk;
				$salePrice = $row->priceOriginalCzk ? $row->priceCzk : '';

				if ($targetCurrency && $targetCurrency !== $defaultCurrency) {
					$price = $this->exchange->change((float) $price, $targetCurrency);
					if ($salePrice)
						$salePrice = $this->exchange->change((float) $salePrice, $targetCurrency);
				}

				$outputCurrency = $targetCurrency ?: $defaultCurrency;

				if (isset($ranks[$row->id])) {
					$rank = $ranks[$row->id];
				} else {
					$rank = $i;
					$i++;
				}

				fputcsv($fp, [
					(int) $rank,
					(string) $row->id,
					Strings::substring($row->title, 0, 35),
					Strings::truncate($row->description, 127),
					$row->link,
					$this->imagePipe->request($image, '795x450', 'bestFit'),
					$price . ' ' . $outputCurrency,
					$storeCode,
					$salePrice ? $salePrice . ' ' . $outputCurrency : '',
				], ',');
			}

			rewind($fp);
			$event->data = [
				'output' => stream_get_contents($fp),
			];

			fclose($fp);
		}
	}

	public function productsFeedAvailability(ProductsFeedEvent $event): void
	{
		if ($event->customerName !== 'ads')
			return;

		$event->outputFormat = 'csv';

		$fp = fopen('php://memory', 'wb+');

		$storeCode       = $this->httpRequest->getQuery('store');
		$targetCurrency  = $this->httpRequest->getQuery('currency');
		$defaultCurrency = $this->currencies->getDefaultCode();

		fputcsv($fp, ['store_code', 'id', 'quantity', 'price', 'sale_price', 'availability', 'pickup_method',
			'pickup_sla'], ',');
		foreach ($event->data as $row) {
			if ($row['available'] <= 0)
				continue;

			$av = $this->availabilityService->getByIdent($row['availabilityIdent']);
			if (!$av)
				continue;

			$price     = $row['retailPrice'] ?: $row['price'];
			$salePrice = $row['retailPrice'] ? $row['price'] : '';

			if ($targetCurrency && $targetCurrency !== $defaultCurrency) {
				$price = $this->exchange->change((float) $price, $targetCurrency);
				if ($salePrice)
					$salePrice = $this->exchange->change((float) $salePrice, $targetCurrency);
			}

			$outputCurrency = $targetCurrency ?: $defaultCurrency;

			fputcsv($fp, [
				(string) $storeCode,
				$row['id'],
				$row['available'],
				$price . ' ' . $outputCurrency,
				$salePrice ? $salePrice . ' ' . $outputCurrency : '',
				$av->getOpenGraphText(),
				'buy',
				'same day',
			], ',');
		}

		rewind($fp);
		$event->data = [
			'output' => stream_get_contents($fp),
		];

		fclose($fp);
	}

	public function categoriesFeed(ProductsFeedEvent $event): void
	{
		if ($event->customerName !== 'ads')
			return;

		$event->outputFormat = 'csv';

		$currentSite = $this->sites->getCurrentSite();
		$site        = $currentSite->getIdent();
		$file        = WWW_DIR . '/feed-' . $site . '-' . $currentSite->getCurrentDomain()->getLang() . '.json';
		$hasProducts = [];

		try {
			if (file_exists($file)) {
				foreach ((array) Json::decode(file_get_contents($file)) as $row) {
					foreach ($row->categoryId as $catId)
						$hasProducts[$catId] = true;
				}
			}
		} catch (\Exception $e) {
		}

		$fp = fopen('php://memory', 'wb+');
		fputcsv($fp, ['Page URL', 'Custom label'], ',');
		foreach ($event->data as $row) {
			if (!isset($hasProducts[$row['category_id']]))
				continue;

			fputcsv($fp, [
				$row['link'],
				Strings::webalize($row['title']),
			], ',');
		}

		rewind($fp);
		$event->data = [
			'output' => stream_get_contents($fp),
		];

		fclose($fp);
	}
}
