<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Components;

use Core\Model\UI\BaseControl;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\FrontModule\Model\Tags;
use EshopCatalog\Model\Config;
use Nette\Application\UI\Multiplier;
use Nette\Utils\Paginator;

/**
 * Class ProductsList
 * @package EshopCatalog\FrontModule\Components
 */
class ProductsList extends BaseControl
{
	/** @var Config */
	protected $config;

	protected Products $productsService;

	protected array $categories = [];

	/** @var Product[] */
	protected $products;

	/** @var int */
	protected $allProductsCount;

	/** @var string */
	protected $searchQuery;

	/** @var int */
	public $page = 1;

	/** @var int|null @persistent */
	public ?int $limit = null;

	public ?int $overrideLimit = null;

	/** @var int */
	public $start = 0;

	/** @var bool */
	public $paginatorEnabled = true;

	/** @var BaseControl|ProductsFilter */
	public $filterComponent;

	protected Tags $tagsService;

	protected Paginator $paginator;

	protected IProductPreviewFactory $productPreviewFactory;

	protected ProductsFacade $productsFacade;

	public function __construct(Config $config, Products $products, Tags $tags, IProductPreviewFactory $productPreviewFactory, ProductsFacade $productsFacade)
	{
		$this->config                = $config;
		$this->productsService       = $products;
		$this->tagsService           = $tags;
		$this->productPreviewFactory = $productPreviewFactory;
		$this->productsFacade        = $productsFacade;
	}

	public function render()
	{
		$this->template->paginator = $this->getPaginator();
		$this->template->limit     = $this->getLimit();
		$this->template->start     = $this->start;
		$this->template->products  = $this->getProducts();
		$this->template->count     = $this->getCount();
		$this->template->sortLink  = function($a) {
			if ($this->filterComponent)
				return $this->filterComponent->link('sort!', $a);

			return '#';
		};

		$this->template->currentSort = $this->filterComponent ? $this->filterComponent->sort ?: 'recommended' : null;
		$this->template->sortValues  = $this->filterComponent ? $this->filterComponent->getSortValues() : [];

		$list = [];
		foreach (Config::load('productsList.itemsPerPageList') as $v)
			$list[$v] = $this->link('limit!', $v);
		$this->template->listLimits = $list;

		$this->template->render($this->getTemplateFile(null, 'ProductsList'));
	}

	public function handlePaginator($page)
	{
		$this->page  = $page;
		$this->start = $page > 1 ? $this->getLimit() * ($page - 1) : 0;

		if ($this->getPresenter()->isAjax()) {
			$this->redrawControl('list');
			$this->redrawControl('paginator');
		}
	}

	public function handleLimit(int $limit)
	{
		if ($limit == (int) Config::load('productsList.itemsPerPage') || $this->overrideLimit && $limit == $this->overrideLimit)
			$this->limit = null;
		else
			$this->limit = $limit;
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentProductPreview(): Multiplier
	{
		return new Multiplier(function(string $id): ProductPreview {
			$control = $this->productPreviewFactory->create();
			$id      = (int) $id;

			$product = $this->getProducts()[$id] ?? null;
			if ($product)
				$control->setProduct($product);
			else
				$control->setProductById($id);

			return $control;
		});
	}


	/*******************************************************************************************************************
	 * ============================== Get / Set
	 */

	/**
	 * @return Paginator
	 */
	public function getPaginator(): Paginator
	{
		$itemsPerPage = $this->getLimit();

		if ($this->page === 1) {
			$pathParts      = explode('/', $_SERVER['REQUEST_URI']);
			$paginatorIndex = array_search($this->translator->translate('default.urlPart.page'), $pathParts);

			if ($paginatorIndex !== false) {
				$this->page  = (int) $pathParts[$paginatorIndex + 1];
				$this->start = $this->getLimit() * ($this->page - 1);
			}
		}

		$paginator = new Paginator();
		$paginator->setItemCount($this->getCount());
		$paginator->setItemsPerPage($itemsPerPage);
		$paginator->setPage($this->page);

		return $paginator;
	}

	/**
	 * @return Product[]|null
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 * @throws \Nette\Application\UI\InvalidLinkException
	 * @throws \Throwable
	 */
	public function getProducts(): ?array
	{
		if ($this->products === null) {
			$this->getPaginator();

			$products = $this->filterComponent ? $this->filterComponent->getProductIdsAfterFilter() : [];
			$products = array_slice($products, $this->start, $this->getLimit(), true);

			$products       = $this->productsFacade->getProducts($products);
			$this->products = $products;

			if (empty($this->products)) {
				if (!empty($this->categories)) {
					if (empty($filter))
						$this->template->notFoundText = $this->t('eshopCatalogFront.productsList.productsNotFoundCategory');
					else
						$this->template->notFoundText = $this->t('eshopCatalogFront.productsList.productsNotFoundFilter');
				} else if ($this->searchQuery)
					$this->template->notFoundText = $this->t('eshopCatalogFront.productsList.productsNotFoundSearch');
			}

			if ($this->products)
				$this->productsService->loadQuantity($this->products);
		}

		return $this->products;
	}

	protected function getCount(): int
	{
		return $this->filterComponent ? count($this->filterComponent->getProductIdsAfterFilter()) : 0;
	}

	public function setCategories(array $categories): self
	{
		$this->categories = $categories;

		return $this;
	}

	public function setFilterComponent(BaseControl $component): self
	{
		$this->filterComponent = $component;
		if (!empty($this->categories)) {
			$this->filterComponent->setCategories($this->categories);
		} else if ($this->searchQuery) {
			$this->filterComponent->setSearchQuery($this->searchQuery);
		}

		return $this;
	}

	public function setProductsBySearch(string $q): self
	{
		$this->searchQuery = $q;

		return $this;
	}

	protected function getLimit(): int
	{
		if ($this->overrideLimit)
			return $this->overrideLimit;

		return $this->limit ?: (int) Config::load('productsList.itemsPerPage');
	}
}
