<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Lang\DefaultLang;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Entities\FeatureValue;
use Nette\Caching\Cache;

/**
 * Class Features
 * @package EshopCatalog\FrontModule\Model
 */
class Features extends BaseFrontEntityService
{
	protected $entityClass = Feature::class;

	const CACHE_NAMESPACE = 'eshopCatalogFeatures';

	/** @var DefaultLang */
	protected $defaultLang;

	/** @var CacheService */
	protected $cacheService;

	/** @var array */
	protected $cacheDep = [
		Cache::TAGS   => ['features'],
		Cache::EXPIRE => '2 weeks',
	];

	protected ?array $cFeatures = null;

	protected ?array $cFeatureValues = null;

	/**
	 * Features constructor.
	 *
	 * @param CacheService $cacheService
	 * @param DefaultLang  $defaultLang
	 */
	public function __construct(CacheService $cacheService, DefaultLang $defaultLang)
	{
		$this->cacheService = $cacheService;
		$this->defaultLang  = $defaultLang;
	}

	public function getPublishedFeatures(): array
	{
		$features = [];
		foreach ($this->getEr()->createQueryBuilder('f')
			         ->select('f.id, ft.name')
			         ->join('f.featureTexts', 'ft')
			         ->where('f.isPublished = 1')->andWhere('ft.lang = :lang')
			         ->setParameter('lang', $this->defaultLang->locale)
			         ->orderBy('f.position')
			         ->getQuery()->getScalarResult() as $row)
			$features[$row['id']] = $row['name'];

		return $features;
	}

	/**
	 * @param int $id
	 *
	 * @return array
	 */
	public function getFeatureById($id)
	{
		if (!$this->cFeatures[$id]) {
			$locale = $this->translator->getLocale();
			$key    = 'features_' . $locale;

			if ($this->cFeatures === null) {
				$this->cFeatures = $this->cacheService->defaultCache->load($key, function(&$dep) use ($id, $locale) {
					$dep = $this->cacheDep;

					$select = 'f.id, ft.name, f.position, f.unit, f.type, f.decimals';
					if (Config::load('features.allowDescription'))
						$select .= ', ft.productTabTitle';

					$data = [];
					foreach ($this->em->getRepository(Feature::class)->createQueryBuilder('f')
						         ->select($select)
						         ->andWhere('f.isPublished = 1')
						         ->join('f.featureTexts', 'ft', 'WITH', 'ft.lang = :lang')
						         ->setParameter('lang', $locale)->getQuery()->getArrayResult() as $row)
						$data[$row['id']] = $row;

					return $data;
				});
			}
		}

		return $this->cFeatures[$id];
	}

	/**
	 * @param int $id
	 *
	 * @return mixed
	 */
	public function getValuesByFeatureId(int $id)
	{
		if (!isset($this->cFeatures[$id]['values'])) {
			$key = 'featureIdValues';

			$data = $this->cacheService->defaultCache->load($key, function(&$dep) {
				$dep                = $this->cacheDep;
				$dep[Cache::TAGS][] = 'featureValues';

				$data = [];

				foreach ($this->em->getRepository(FeatureValue::class)->createQueryBuilder('fv')
					         ->select('fv.id as id, IDENTITY(fv.feature) as featureId')
					         ->andWhere('fv.isPublished = 1')->orderBy('fv.position')
					         ->getQuery()->getArrayResult() as $row)
					$data[$row['featureId']][] = $row['id'];

				return $data;
			});

			foreach ($data as $k => $v) {
				foreach ($v as $vId)
					$this->cFeatures[$k]['values'][$vId] = $this->getFeatureValueById($vId);
			}
		}

		return $this->cFeatures[$id]['values'];
	}

	/**
	 * @param int $id
	 *
	 * @return array
	 */
	public function getFeatureValueById($id)
	{
		if (!$this->cFeatureValues[$id]) {
			$locale = $this->defaultLang->locale;
			$key    = 'featureValues_' . $locale;

			if ($this->cFeatureValues === null) {
				$this->cFeatureValues = $this->cacheService->defaultCache->load($key, function(&$dep) use ($id, $locale) {
					$dep                = $this->cacheDep;
					$dep[Cache::TAGS][] = 'featureValues';

					$select = 'fv.id as id, fv.position, fvt.name as name, fvt.rodM, fvt.rodZ, fvt.rodS, IDENTITY(fv.feature) as featureId';
					if (Config::load('features.allowImages'))
						$select .= ', fv.image';
					if (Config::load('features.allowDescription'))
						$select .= ', fv.showAsTag, fv.moreLink, fv.tagTextColor, fv.tagBgColor, fvt.tagText, fvt.shortDescription, fvt.longDescription';

					$data = [];
					foreach ($this->em->getRepository(FeatureValue::class)->createQueryBuilder('fv')
						         ->select($select)
						         ->andWhere('fv.isPublished = 1')
						         ->join('fv.featureValueTexts', 'fvt', 'WITH', 'fvt.lang = :lang ')
						         ->orderBy('fv.position')
						         ->setParameter('lang', $locale)->getQuery()->getArrayResult() as $row)
						$data[$row['id']] = $row;

					return $data;
				});
			}
		}

		return $this->cFeatureValues[$id];
	}
}
