<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Helpers\Arrays;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\CategoryProduct;
use EshopCatalog\Model\Entities\ProductSpedition;
use EshopCatalog\Model\Entities\ProductVariant;
use EshopCatalog\Model\Entities\RelatedProduct;
use Gallery\FrontModule\Model\Dao\Album;
use Nette\Application\LinkGenerator;
use Core\Model\Sites;
use EshopCatalog\FrontModule\Model\Event\ProductsEvent;
use Core\Model\Event\EventDispatcher;
use Gallery\FrontModule\Model\Albums;
use Nette\Caching\Cache;
use Nette\Localization\Translator;
use Users\Model\Http\UserStorage;
use EshopCatalog\FrontModule\Model\Dao;

class ProductsFacade
{
	const MODE_ESHOP    = 'eshop';
	const MODE_CHECKOUT = 'checkout';

	protected static string $mode = self::MODE_ESHOP;

	public static ?string $forceLocale = null;

	public static ?bool $cacheEnabled = true;

	protected Translator $translator;

	protected CacheService $cacheService;

	public Products $productsService;

	protected EventDispatcher $eventDispatcher;

	public LinkGenerator $linkGenerator;

	protected UserStorage $userStorage;

	protected Albums $albumsService;

	protected Manufacturers $manufacturersService;

	protected Tags $tagsService;

	protected Sites $sitesService;

	public Categories $categoriesService;

	protected DynamicFeaturesProducts $dynamicFeaturesProducts;

	/** @var Dao\Product[] */
	protected $cProducts, $cVariants;

	public static array $allLoadedIds = [];

	public static bool $loadFeatures = true;

	public static bool $loadTags = true;

	public function __construct(Translator  $translator, CacheService $cacheService, Products $products, EventDispatcher $eventDispatcher,
	                            UserStorage $userStorage, Albums $albums, Manufacturers $manufacturers, Tags $tags,
	                            Sites       $sites, Categories $categories, DynamicFeaturesProducts $dynamicFeaturesProducts)
	{
		$this->translator              = $translator;
		$this->cacheService            = $cacheService;
		$this->productsService         = $products;
		$this->eventDispatcher         = $eventDispatcher;
		$this->userStorage             = $userStorage;
		$this->albumsService           = $albums;
		$this->manufacturersService    = $manufacturers;
		$this->tagsService             = $tags;
		$this->sitesService            = $sites;
		$this->categoriesService       = $categories;
		$this->dynamicFeaturesProducts = $dynamicFeaturesProducts;
	}

	public function loadBase(array $ids)
	{
		$whereIds = [];
		$loaded   = [];
		$result   = [];
		$locale   = $this->getLocale();

		foreach ($ids as $k => $id) {
			if (isset($this->cProducts[$id])) {
				$loaded[$id] = $this->cProducts[$id];
				unset($ids[$k]);
			}
		}

		if (self::$cacheEnabled) {
			$keys = [];
			foreach ($ids as $id)
				$keys[] = 'product/' . $id . '/' . $locale;

			// Načtení z cache
			if ($keys) {
				foreach ($this->cacheService->productCache->bulkLoad($keys) as $key => $product) {
					$tmp = explode('/', $key);
					$id  = $tmp[1];

					if ($product)
						$result[$id] = $product;
					else
						$whereIds[] = $id;
				}
			}
		} else {
			$whereIds = array_values($ids);
		}

		// Vytvoření získání produktů co nejsou v cache
		if (!empty($whereIds)) {
			$query = (new ProductQuery($locale))
				->withTexts()
				->withVatRate()
				->loadSites()
				->withVariantParent()
				->byId($whereIds);

			$rootCats = $this->categoriesService->getAllRootIds();

			$cats = [];
			foreach ($this->productsService->em->createQueryBuilder()
				         ->select('IDENTITY(cp.product) as product, IDENTITY(cp.category) as category')
				         ->from(CategoryProduct::class, 'cp')
				         ->where('cp.product IN (:ids)')
				         ->setParameter('ids', $whereIds)
				         ->getQuery()->getScalarResult() as $cat) {
				$cats[$cat['product']][] = $cat['category'];
			}

			$qb = $query->getQueryBuilder($this->productsService->getEr());
			$qb->groupBy('p.id');

			foreach ($this->productsService->customQB as $cqb)
				$cqb($qb);

			$disabledSpeds = [];
			foreach ($this->productsService->em->getRepository(ProductSpedition::class)->createQueryBuilder('s')
				         ->where('s.product IN (:ids)')->setParameter('ids', $whereIds)
				         ->andWhere('s.speditionDisabled = 1')
				         ->getQuery()->getScalarResult() as $row) {
				$disabledSpeds[$row['s_product_id']][] = $row['s_spedition_id'];
			}

			$qbResult   = $qb->getQuery()->getArrayResult();
			$idsForLink = [];
			foreach ($qbResult as $row)
				$idsForLink[] = $row[0]['id'];
			self::$allLoadedIds = $idsForLink;
			$featureValuesIds   = $this->productsService->featureProductsService->getFeatureValuesIdsForProducts($idsForLink);
			$dynamicFeatures    = $this->dynamicFeaturesProducts->getDaoByProducts($idsForLink);

			foreach ($qbResult as $row) {
				$row[0]['categories']      = $cats[$row[0]['id']] ?? [];
				$tmp                       = $this->productsService->normalizeHydrateArray($row);
				$tmp['extraFields']        = $this->productsService->getAllExtraFields()[$tmp['id']] ?? [];
				$tmp['disabledSpeditions'] = $disabledSpeds[$row[0]['id']] ?? [];
				$tmp['dynamicFeatures']    = $dynamicFeatures[$tmp['id']] ?? [];
				$dao                       = $this->productsService->fillDao($tmp);
				$dao->featureValuesIds     = $featureValuesIds[$dao->getId()] ?? [];

				foreach ($dao->inSites as $v)
					if (isset($rootCats[$v]))
						array_unshift($dao->categories, $rootCats[$v]);

				if (isset($row[0]['isPublished']) && $row[0]['isPublished'] === 0)
					$dao->isActive = false;

				if (Config::load('canCacheProductsUrl') && $this->linkGenerator) {
					$linkParams = [
						'id'     => $dao->getId(),
						'locale' => $locale,
					];

					if (Config::load('useOldUrls') && $product->getExtraField(Config::load('product.oldUrlEFKey'), null) !== null
						&& $dao->variantOf && $dao->variantOf != $dao->getId()) {
						$linkParams['variant'] = $dao->getId();
					}

					$dao->link = $this->linkGenerator->link('EshopCatalog:Front:Default:product', $linkParams);
				}

				if (self::$cacheEnabled) {
					$cacheDep                = $this->productsService->cacheDep;
					$cacheDep[Cache::TAGS][] = 'product/' . $dao->getId();
					$cacheDep[Cache::EXPIRE] = '1 week';
					$this->cacheService->productCache->save('product/' . $dao->getId() . '/' . $locale, $dao, $cacheDep);
				}

				$result[$tmp['id']] = $dao;
			}
		}

		return [
			'prepared' => $result,
			'loaded'   => $loaded,
		];
	}

	/**
	 * Vrátí DAO produktů z cache nebo získá data z db a uloží do cache
	 *
	 * @param array       $ids
	 * @param string|null $siteIdent
	 *
	 * @return Dao\Product[]
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 * @throws \Nette\Application\UI\InvalidLinkException
	 * @throws \Throwable
	 */
	public function getProducts(array $ids, ?string $siteIdent = null): array
	{
		$baseIds = $ids;
		$data    = $this->loadBase($ids);

		/** @var Dao\Product[] $loaded */
		$loaded = $data['loaded'];

		/** @var Dao\Product[] $result */
		$result = $data['prepared'];

		if (!empty($result)) {
			$variants         = [];
			$variantParentIds = [];
			$variantIds       = [];

			foreach ($result as $product) {
				if ($product->variantId)
					$variantIds[$product->getId()] = $product->variantId;
			}

			if (!empty($variantIds)) {
				$tmp = [];
				foreach ($this->productsService->em->getRepository(ProductVariant::class)->createQueryBuilder('pv')
					         ->select('IDENTITY(pv.product) as product, pv.variantId, pv.isDefault')
					         ->where('pv.variantId IN (:ids)')->setParameter('ids', array_unique($variantIds))
					         ->getQuery()->getArrayResult() as $row) {
					if ($row['isDefault'] == 1)
						$variantParentIds[$row['variantId']] = $row['product'];
					if (isset($variantIds[$row['variantId']]) && !is_array($variantIds[$row['variantId']])) {
						$variantIds[$row['variantId']] = [$variantIds[$row['variantId']]];
					}
					$tmp[$row['variantId']][] = $row['product'];
					$variants[]               = $row['product'];
				}
				$variantIds = $tmp;

				if (!empty($variants)) {
					$tmp    = $this->loadBase(array_unique($variants));
					$loaded += $tmp['loaded'];
					$result += $tmp['prepared'];
				}
			}

			foreach ($result as $product)
				if (isset($variantParentIds[$product->variantId]))
					$product->variantOf = (int) $variantParentIds[$product->variantId];

			if (self::$loadFeatures)
				$this->productsService->loadFeatures($result);
			if (self::$loadTags)
				$this->tagsService->loadTagsToProduct($result);
			$this->productsService->loadFresh($result, $this->userStorage->getIdentity() ? $this->userStorage->getIdentity()->getId() : null);

			foreach ($result as $productId => $product)
				uasort($product->features, fn($a, $b) => $a->position <=> $b->position);

			$galIds = [];
			foreach ($result as $product)
				if ($product->galleryId)
					$galIds[] = $product->galleryId;
			$galleries = $this->albumsService->get($galIds);

			for ($i = 0; $i <= 1; $i++) {
				foreach ($result as $id => $product) {
					if (!$product->isActive) {
						unset($result[$id]);
						continue;
					}

					if ($i === 0 && $product->variantOf && $product->variantOf !== $product->getId())
						continue;
					else if ($i === 1 && !$product->variantOf)
						continue;

					if ($product->galleryId) {
						/** @var Album $gallery */
						$gallery = isset($galleries[$product->galleryId]) ? clone $galleries[$product->galleryId] : null;

						if ($gallery) {
							if ($product->helperData['variantImage']) {
								$img = $gallery->getImages()[$product->helperData['variantImage']];

								foreach ($gallery->getImages() as $image) {
									if ($img->id == $image->id)
										$image->isCover = true;
									else
										$image->isCover = false;
								}
								if ($img)
									$gallery->setCover($img);
							}

							if ($gallery->getCover())
								$product->variantImage = $gallery->getCover();

							$product->setGallery($gallery);
						}
					}

					if ($product->manufacturerId)
						$product->setManufacturer($this->manufacturersService->get((int) $product->manufacturerId));

					if (!$product->defaultCategoryId && $this->cProducts[$product->variantOf])
						$product->defaultCategoryId = $this->cProducts[$product->variantOf]->defaultCategoryId;

					if ($product->defaultCategoryId) {
						$product->defaultCategory = $this->categoriesService->get($product->defaultCategoryId);
						$product->canAddToCart    = $product->defaultCategory->canProductsAddToCart ? true : false;
					}

					if (($product->canAddToCart || !$product->defaultCategoryId) && $product->getAvailability())
						$product->canAddToCart = (bool) $product->getAvailability()->canAddToCart();

					$this->cProducts[$id] = $product;
					$loaded[$id]          = $product;
					$result[$id]          = null;
				}
			}

			$this->eventDispatcher->dispatch(new ProductsEvent($loaded), Products::class . '::afterFillDao');

			$allLoaded          = $loaded;
			self::$allLoadedIds = array_keys($allLoaded);

			foreach ($allLoaded as $id => $product) {
				if (!Config::load('canCacheProductsUrl') && $this->linkGenerator) {
					$linkParams = [
						'id'     => $product->getId(),
						'locale' => $this->getLocale(),
					];

					if (Config::load('useOldUrls') && $product->getExtraField(Config::load('product.oldUrlEFKey'), null) !== null
						&& $product->variantOf && $product->variantOf != $product->getId()) {
						$linkParams['variant'] = $product->getId();
					}

					$product->link = $this->linkGenerator->link('EshopCatalog:Front:Default:product', $linkParams);
				}

				$this->cProducts[$id] = $product;
			}

			foreach (array_diff(array_keys($loaded), $baseIds) as $v)
				unset($loaded[$v]);

			if (self::$mode != self::MODE_CHECKOUT)
				foreach ($loaded as $id => $product) {
					if (!$product->variantId)
						continue;

					foreach ($variantIds[$product->variantId] as $v)
						if ($allLoaded[$v])
							$product->variants[$v] = $allLoaded[$v];

					if (count($product->variants) <= 1) {
						$product->variants  = [];
						$product->variantId = null;
					} else {
						foreach ($product->variants as $k => $v) {
							$product->variants[$k]->variants = &$product->variants;
						}
					}
				}
		}

		$result = [];
		foreach ($ids as $id) {
			if (isset($loaded[$id]))
				$result[$id] = $loaded[$id];
		}

		return $result;
	}

	/**
	 * @param int         $id
	 * @param string|null $siteIdent
	 *
	 * @return Dao\Product|null
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 * @throws \Nette\Application\UI\InvalidLinkException
	 * @throws \Throwable
	 */
	public function getProduct(int $id, ?string $siteIdent = null): ?Dao\Product
	{
		return $this->getProducts([$id], $siteIdent)[$id] ?? null;
	}

	public function loadAlternative(Dao\Product &$product, $limit)
	{
		$data       = $this->productsService->getProductsIdInCategory($product->defaultCategoryId, 0, null, null, [], false);
		$variantIds = array_keys($product->variants);

		foreach ($data as $k => $v) {
			if ($product->getId() == $v || in_array($v, $variantIds))
				unset($data[$k]);
		}

		shuffle($data);
		$products = $this->getProducts(array_slice($data, 0, $limit));

		$product->setAlternatives($products);
	}

	public function loadRelated(Dao\Product &$product, $limit)
	{
		if (!Config::load('allowRelatedProducts') || !empty($product->getRelated()))
			return;

		[$related, $ids, $groupedIds] = $this->cacheService->productCache
			->load('related/' . $product->getId(), function(&$dep) use ($product) {
				$dep = [
					Cache::TAGS => ['related'],
				];

				$related    = [];
				$ids        = [];
				$groupedIds = [];
				foreach ($this->productsService->em->getRepository(RelatedProduct::class)->createQueryBuilder('rp')
					         ->select('IDENTITY(rp.product) as product, g.id as groupId, gt.name as groupName')
					         ->innerJoin('rp.group', 'g', Join::WITH, 'g.isPublished = 1')
					         ->innerJoin('g.texts', 'gt', Join::WITH, 'gt.lang = :lang')
					         ->andWhere('rp.origin = :id')
					         ->orderBy('g.position')
					         ->setParameters([
						         'lang' => $this->getLocale(),
						         'id'   => $product->getId(),
					         ])->getQuery()->getArrayResult() as $row) {
					if (!isset($related[$row['groupId']]))
						$related[$row['groupId']] = new Dao\RelatedGroup($row['groupId'], $row['groupName']);
					$ids[]                         = $row['product'];
					$groupedIds[$row['groupId']][] = $row['product'];
				}

				return [$related, $ids, $groupedIds];
			});

		if ($ids) {
			Products::$loadKey = 'related';
			$products          = $this->getProducts($ids);

			foreach ($groupedIds as $groupId => $ids) {
				shuffle($ids);
				$i = 0;

				foreach ($ids as $id) {
					if (!isset($products[$id]))
						continue;

					$related[$groupId]->addProduct($products[$id]);
					$i++;

					if ($i >= $limit)
						break;
				}
			}

			$product->setRelated($related);
			Products::$loadKey = null;
		}
	}

	public static function setMode(string $mode): void
	{
		self::$mode = $mode;
		Products::setMode(Products::MODE_CHECKOUT);
		Categories::setMode(Categories::MODE_CHECKOUT);
	}

	protected function getLocale(): string { return self::$forceLocale ?: $this->translator->getLocale(); }

	public function updateQuantitiesByIds(array $ids)
	{
		$forUpdate = [];
		foreach ($ids as $id) {
			$product = $this->cProducts[$id] ?? null;

			if (!$product)
				continue;

			$forUpdate[$id] = &$product;
		}

		$this->productsService->loadQuantity($forUpdate, true);
	}

	public function clearTemp()
	{
		$this->cProducts = [];
		$this->cVariants = [];
	}
}
