<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Core\Model\Module;
use Doctrine;
use Doctrine\ORM\Event\LifecycleEventArgs;
use Doctrine\ORM\Event\PreUpdateEventArgs;
use Doctrine\ORM\Mapping as ORM;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\AvailabilityService;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Navigation\Home;
use EshopCatalog\Model\Products;
use Nette\Utils\DateTime;
use Nette\Utils\FileSystem;
use Nettrine\ORM\EntityManagerDecorator;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Nette\Caching\Cache;
use Nette\SmartObject;
use Tracy\Debugger;

class ProductListener implements EventSubscriberInterface
{
	use SmartObject;

	protected static array $cleared = [];

	protected CacheService $cacheService;

	protected AvailabilityService $availabilityService;

	protected Products $productsService;

	protected EntityManagerDecorator $em;

	public static bool $updateVariant = true;

	public function __construct(CacheService           $cacheService, AvailabilityService $availabilityService, Products $products,
	                            EntityManagerDecorator $em)
	{
		$this->cacheService        = $cacheService;
		$this->availabilityService = $availabilityService;
		$this->productsService     = $products;
		$this->em                  = $em;
	}

	public static function getSubscribedEvents(): array
	{
		return [];
	}

	/**
	 * @ORM\PreUpdate
	 *
	 * @param Product            $product
	 * @param PreUpdateEventArgs $args
	 */
	public function preUpdate($product, PreUpdateEventArgs $args)
	{
		if ($product instanceof Product && !$product->unlimitedQuantity) {
			// Aktualizuje dostupnost pokud se změnil počet skladem
			if ($product->getAvailability() && isset($args->getEntityChangeSet()['quantity'])) {
				$oldAvIdent = $product->getAvailability()->getIdent();
				$this->availabilityService->updateAvailabilityByQuantity($product);
				$newAvIdent = $product->getAvailability()->getIdent();

				if ($oldAvIdent != $newAvIdent) {
					$this->em->getConnection()->prepare("UPDATE " . $this->em->getClassMetadata(Product::class)->getTableName() . " SET id_availability = ? WHERE id = ?")
						->execute([$product->getAvailability()->getId(), $product->getId()]);
					//					$args->setNewValue('availability', $product->getAvailability());
				}
			}

			//			// Aktualizuje počet skladem pokud se dostunost změnila na vyprodáno
			//			if (isset($args->getEntityChangeSet()['availability'])) {
			//				$avChangeSet = $args->getEntityChangeSet()['availability'];
			//
			//				if ($avChangeSet[1]->getIdent() == Availability::SOLD_OUT && $product->quantity != 0) {
			//					$product->quantity = 0;
			//				}
			//			}
		}
	}

	/**
	 * @ORM\PreFlush
	 *
	 * @param Product            $product
	 * @param LifecycleEventArgs $args
	 */
	public function onPreFlush($product, $args)
	{
		if (isset(self::$cleared[$product->getId()]) && self::$cleared[$product->getId()] === false) {
			$this->cacheService->clean('product', [
				Cache::TAGS => [
					'product/' . $product->getId(),
				],
			]);

			$this->cacheService->clean('navigation', [
				Cache::TAGS => [
					Home::CACHE_PRODUCT . '/' . $product->getId(),
				],
			]);
		}
		self::$cleared[$product->getId()] = true;
	}

	/**
	 * @ORM\PostUpdate
	 *
	 * @param Product            $product
	 * @param LifecycleEventArgs $event
	 *
	 * @throws Doctrine\DBAL\DBALException
	 * @throws Doctrine\ORM\NonUniqueResultException
	 */
	public function postHandlerUpdate(Product $product, LifecycleEventArgs $event)
	{
		if ($product instanceof Product === false)
			return;

		if (Module::isAdmin() && $product->isVariant() && static::$updateVariant) {
			$this->productsService->prepareUpdateVariant($product);
		}
	}

	/**
	 * @ORM\PostUpdate
	 * @ORM\PostRemove
	 *
	 * @param Product            $product
	 * @param LifecycleEventArgs $event
	 *
	 * @throws Doctrine\ORM\ORMException
	 */
	public function postHandler(Product $product, LifecycleEventArgs $event)
	{
		$em = $event->getEntityManager();

		if (Config::load('enableLoggingQuantityChange' . false)) {
			$changeSet = $em->getUnitOfWork()->getEntityChangeSet($product);
			if (isset($changeSet['quantity'])) {
				$old = $changeSet['quantity'][0];
				$new = $changeSet['quantity'][1];

				try {
					$dir  = '_quantities/' . (new DateTime())->format('Y-m-d');
					$file = $product->getId();
					FileSystem::createDir(LOG_DIR . '/' . $dir);
					$e = new \Exception();
					Debugger::log(print_r($old . ' -> ' . $new, true), $dir . '/' . $file);
					Debugger::log(print_r($e->getTraceAsString(), true), $dir . '/' . $file);
				} catch (\Exception $e) {
				}
			}
		}

		self::$cleared[$product->getId()] = true;
	}

	/**
	 * @ORM\PreRemove
	 *
	 * @param Product            $product
	 * @param LifecycleEventArgs $events
	 */
	public function preRemoveHandler(Product $product, LifecycleEventArgs $events): void
	{
		$this->cacheService->productCache->clean([Cache::TAGS => ['product/' . $product->getId()]]);

		if ($product->getGallery() && !$product->isVariant()) {
			$events->getEntityManager()->remove($product->getGallery());
			$events->getEntityManager()->flush($product->getGallery());
		}
	}

}
