<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model;

use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\Traits\TPosition;
use Core\Model\Helpers\Traits\TPublish;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Entities\FeatureValue;
use Nette\Caching\Cache;
use Nette\Localization\ITranslator;

/**
 * Class FeatureValues
 * @package EshopCatalog\AdminModule\Model
 *
 * @method FeatureValue|null|object getReference($id)
 * @method FeatureValue[]|null getAll()
 * @method FeatureValue|null get($id)
 */
class FeatureValues extends BaseEntityService
{
	use TPublish;
	use TPosition;

	/** @var CacheService */
	protected CacheService $cacheService;

	/** @var ITranslator */
	protected ITranslator $translator;

	protected $entityClass = FeatureValue::class;

	public function __construct(CacheService $cacheService, ITranslator $translator)
	{
		$this->cacheService = $cacheService;
		$this->translator   = $translator;
	}

	/**
	 * @param int $id
	 * @param int $position
	 *
	 * @return bool
	 * @throws \Exception
	 */
	public function setPosition($id, $position)
	{
		if ($item = $this->getReference($id)) {
			$item->setPosition((int) $position);
			$this->em->persist($item);
			$this->em->flush();

			return true;
		}

		return false;
	}

	public function mergeAll(?int $featureId = null)
	{
		$delete = [];
		$this->em->getConnection()->beginTransaction();
		try {
			foreach ($this->getEr()->createQueryBuilder('fv')->select('fv.id')
				         ->leftJoin('fv.featureValueTexts', 'fvt')->where('fvt IS NULL')->getQuery()->getArrayResult() as $row)
				$delete[] = $row['id'];

			foreach (array_chunk($delete, 200) as $chunk)
				$this->em->createQuery('DELETE FROM ' . $this->entityClass . ' fv WHERE fv.id IN (:id)')->setParameter('id', $chunk)->execute();
			$delete = null;

			$fks = [];
			foreach ($this->em->getConnection()->fetchAllAssociative("
				SELECT *
					FROM information_schema.KEY_COLUMN_USAGE
					WHERE
  						REFERENCED_TABLE_NAME = 'eshop_catalog__feature_value'
  						AND REFERENCED_COLUMN_NAME = 'id'
  						AND CONSTRAINT_SCHEMA = '{$this->em->getConnection()->getDatabase()}'") as $tmp) {
				if ($tmp['TABLE_NAME'] == 'eshop_catalog__feature_value_texts')
					continue;

				$fks[] = $tmp;
			}

			foreach ($this->translator->getLocalesWhitelist() as $l) {
				$qb = $this->getEr()->createQueryBuilder('fv')->select('fvt.name, count(fv.id), IDENTITY(fv.feature) as feature')
					->join('fv.featureValueTexts', 'fvt', 'WITH', 'fvt.lang = :lang')
					->setParameter('lang', $this->translator->getLocale())
					->groupBy('feature, fvt.name')->having('count(fv.id) > 1');

				if ($featureId)
					$qb->andWhere('fv.feature = :feature')
						->setParameter('feature', $featureId);

				foreach ($qb->getQuery()->getArrayResult() as $duplicate) {
					$ids = [];
					foreach ($this->getEr()->createQueryBuilder('fv')->select('fv.id')
						         ->where('fvt.name = :name')->setParameter('name', $duplicate['name'])
						         ->andWhere('fv.feature = :feature')->setParameter('feature', $duplicate['feature'])
						         ->join('fv.featureValueTexts', 'fvt', 'WITH', 'fvt.lang = \'cs\'')
						         ->getQuery()->getArrayResult() as $tmp)
						$ids[] = $tmp['id'];

					$first = array_shift($ids);

					foreach ($fks as $fk) {
						$this->em->getConnection()->executeStatement("UPDATE {$fk['TABLE_NAME']} SET {$fk['COLUMN_NAME']} = $first WHERE {$fk['COLUMN_NAME']} IN (" . implode(',', $ids) . ")");
					}

					$this->em->createQuery('DELETE FROM ' . $this->entityClass . ' fv WHERE fv.id IN(:id)')->setParameter('id', $ids)->execute();
				}
			}

			$this->em->getConnection()->commit();
		} catch (\Exception $e) {
			if ($this->em->getConnection()->isTransactionActive())
				$this->em->getConnection()->rollBack();
		}

		$this->cacheService->defaultCache->clean([Cache::TAGS => ['features']]);
	}

	/**
	 * @param int|null $feature
	 *
	 * @return bool
	 * @throws \Doctrine\DBAL\ConnectionException
	 */
	public function repairPositions(?int $feature = null)
	{
		$groups = [];
		$qb     = $this->getEr()->createQueryBuilder('fv')->select('fv.id, fv.position, IDENTITY(fv.feature) as feature')
			->orderBy('feature')->addOrderBy('fv.position')->addOrderBy('fv.id');

		if ($feature)
			$qb->andWhere('fv.feature = :feature')->setParameter('feature', $feature);

		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$groups[$row['feature']][] = $row['id'];
		}

		$lastGroup = null;
		$conn      = $this->em->getConnection();
		$tableName = $this->em->getClassMetadata($this->entityClass)->getTableName();
		$conn->beginTransaction();
		try {
			foreach ($groups as $items) {
				foreach ($items as $k => $id) {
					$conn->update($tableName, [
						'position' => $k,
					], [
						'id' => $id,
					]);
				}
			}

			$conn->commit();
		} catch (\Exception $e) {
			if ($conn->isTransactionActive())
				$conn->rollBack();

			return false;
		}

		return true;
	}

	public function getOptionsForSelect(): array
	{
		$result = [];

		foreach ($this->getEr()->createQueryBuilder('f')->select('f.id, ft.name, IDENTITY(f.feature) as feature')
			         ->innerJoin('f.featureValueTexts', 'ft', Join::WITH, 'ft.lang = :lang')
			         ->innerJoin('f.feature', 'ff')
			         ->innerJoin('ff.featureTexts', 'fft', Join::WITH, 'fft.lang = :lang')
			         ->setParameter('lang', $this->translator->getLocale())
			         ->orderBy('fft.name')->addOrderBy('ft.name')
			         ->getQuery()->getArrayResult() as $row)
			$result[$row['feature']][$row['id']] = $row['name'];

		return $result;
	}

	public function sortAlphabetically(?int $featureId = null): bool
	{
		$groups = [];

		$qb = $this->getEr()->createQueryBuilder('f')->select('f.id, IDENTITY(f.feature) as feature, ft.name')
			->innerJoin('f.featureValueTexts', 'ft', Join::WITH, 'ft.lang = :lang')
			->setParameter('lang', $this->translator->getLocale());

		if ($featureId)
			$qb->andWhere('f.feature = :feature')
				->setParameter('feature', $featureId);

		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$groups[$row['feature']][] = [
				'id'   => $row['id'],
				'name' => $row['name'],
			];
		}

		foreach ($groups as &$vals) {
			usort($vals, fn($a, $b) => $a['name'] <=> $b['name']);
		}

		$conn      = $this->em->getConnection();
		$tableName = $this->em->getClassMetadata($this->entityClass)->getTableName();
		foreach ($groups as $vals) {
			foreach ($vals as $k => $v) {
				$conn->update($tableName, [
					'position' => $k,
				], [
					'id' => $v['id'],
				]);
			}
		}

		return true;
	}
}
