<?php declare(strict_types = 1);

namespace EshopCatalog\CronModule\Presenters;

use Core\Model\Entities\EntityManagerDecorator;
use Currency\Model\Currencies;
use Doctrine\ORM\Query;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\CronModule\Model\CategoriesFeed;
use EshopCatalog\CronModule\Model\ProductsFeed;
use EshopCatalog\CronModule\Model\ProductsVariants;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\AvailabilityService;
use EshopCatalog\Model\Entities\Availability;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Event\ProductsFeedEvent;
use Tracy\Debugger;

class ProductsPresenter extends BasePresenter
{
	/** @var Products */
	protected Products $products;

	protected ProductsFacade $productsFacade;

	protected EntityManagerDecorator $em;

	protected AvailabilityService $availabilityService;

	protected ProductsFeed $productsFeed;

	protected CategoriesFeed $categoriesFeed;

	protected ProductsVariants $productsVariants;

	public function __construct(Products $products, EntityManagerDecorator $em, AvailabilityService $availabilityService,
	                            ProductsFacade $productsFacade, ProductsFeed $productsFeed, CategoriesFeed $categoriesFeed,
	                            ProductsVariants $productsVariants)
	{
		$this->productsFacade      = $productsFacade;
		$this->products            = $products;
		$this->em                  = $em;
		$this->availabilityService = $availabilityService;
		$this->productsFeed        = $productsFeed;
		$this->categoriesFeed      = $categoriesFeed;
		$this->productsVariants    = $productsVariants;
	}

	public function actionFindByTerm(string $term = '', array $excluded = [])
	{
		$output = [];
		foreach ($this->products->getByTerm($term, Query::HYDRATE_ARRAY) as $product) {
			if (!in_array($product['id'], $excluded)) {
				$output[$product['id']] = [
					'id'    => $product['id'],
					'name'  => $product['productTexts'][$this->translator->getLocale()]['name'],
					'code1' => $product['code1'],
					'ean'   => $product['ean'],
				];
			}
		}

		$this->sendJson($output);
	}

	public function actionLoadAll(array $excluded = [])
	{
		$output = [];
		$qb     = $this->products->getEr()->createQueryBuilder('p')
			->select('p.id, p.code1, p.ean, pt.name')
			->innerJoin('p.productTexts', 'pt', 'with', 'pt.lang = :lang')
			->setParameters([
				'lang' => $this->translator->getLocale(),
			]);

		if ($excluded)
			$qb->andWhere('p.id NOT IN (:excluded)')
				->setParameter('excluded', $excluded);

		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$output[$row['id']] = [
				'id'    => (string) $row['id'],
				'name'  => (string) $row['name'],
				'code1' => (string) $row['code1'],
				'ean'   => (string) $row['ean'],
			];
		}

		$this->sendJson($output);
	}

	public function actionCheckAvailabilities()
	{
		set_time_limit(1200);
		$ids = [];

		foreach ($this->products->getEr()->createQueryBuilder('p')
			         ->select('p.id')
			         ->where('p.quantity <= 0 AND p.availability != :av')
			         ->orWhere('p.quantity > 0 AND p.availability != :av2')
			         ->orWhere('p.availability IS NULL')
			         ->setParameters([
				         'av'  => Availability::SOLD_OUT,
				         'av2' => Availability::IN_STOCK,
			         ])
			         ->getQuery()->getArrayResult() as $row) {
			$ids[] = $row['id'];
		}

		//		$supps = [];
		//		foreach ($this->em->getRepository(Supplier::class)->createQueryBuilder('s')
		//			         ->getQuery()->getArrayResult() as $row) {
		//			$supps[$row['name']] = $row['id'];
		//		}
		//
		//		$impIds = [9, 4, 12, 14, 16, 19, 6, 8, 21];
		//		foreach ($this->em->getRepository(Import::class)->createQueryBuilder('i')
		//			         ->where('i.id IN (:ids)')
		//			         ->setParameter('ids', $impIds)
		//			         ->getQuery()->getResult() as $row) {
		//			$avSoldOut = $row->data['idAvailability']['inStock'] ?? null;
		//			$supp      = $row = $row->syncOpts['supplier'];
		//			$suppId    = $supps[$supp] ?? null;
		//
		//			if ($avSoldOut === null || !$suppId)
		//				continue;
		//
		//			dump("UPDATE eshop_catalog__product_supplier SET id_availability_after_sold_out = {$avSoldOut} WHERE id_supplier = {$suppId}");
		//			$this->em->getConnection()->prepare("UPDATE eshop_catalog__product_supplier SET id_availability_after_sold_out = {$avSoldOut} WHERE id_supplier = {$suppId}")->execute();
		//		}
		//		die();

		foreach (array_chunk($ids, 50) as $chunk) {
			$this->em->beginTransaction();
			try {
				foreach ($this->products->getEr()->createQueryBuilder('p')
					         ->addSelect('ps, psa')
					         ->leftJoin('p.suppliers', 'ps')
					         ->leftJoin('ps.availabilityAfterSoldOut', 'psa')
					         ->andWhere('p.id IN (:ids)')
					         ->setParameters([
						         'ids' => $chunk,
					         ])
					         ->getQuery()->getResult() as $row) {
					/** @var Product $row */
					$this->availabilityService->updateAvailabilityByQuantity($row);
					$this->em->persist($row);
				}
				$this->em->flush();
				$this->em->commit();
				$this->em->clear();
			} catch (\Exception $e) {
				if ($this->em->getConnection()->isTransactionActive())
					$this->em->rollback();
			}
		}

		$this->availabilityService->updateProductsWithoutSupplier();

		$this->sendJson([]);
	}

	public function actionFeed(string $lang = 'cs', string $currency = 'CZK')
	{
		ProductsFacade::$forceLocale     = $lang;
		Currencies::$currentCodeOverride = $currency;
		Currencies::clearConvertedProducts();

		$this->productsFeed->baseUrl = $this->getHttpRequest()->getUrl()->getBaseUrl();

		$generate                     = $this->getParameter('generate', null);
		$this->productsFeed->generate = $generate;

		if ($generate)
			$this->sendJson($this->productsFeed->getProducts());

		$data     = $this->productsFeed->loadAndParseProducts();
		$customer = $this->getParameter('customer');

		$event = new ProductsFeedEvent($data, $customer);
		$this->eventDispatcher->dispatch($event, 'eshopCatalog.productsFeed');

		$this->sendJson($data);
	}

	public function actionFeedAvailability()
	{
		ini_set('memory_limit', '2048M');
		set_time_limit(360);

		$this->productsFeed->baseUrl = $this->getHttpRequest()->getUrl()->getBaseUrl();

		$this->sendJson($this->productsFeed->getProductsAvailability());
	}

	public function actionFeedCategories()
	{
		ini_set('memory_limit', '2048M');
		set_time_limit(360);

		$this->categoriesFeed->baseUrl = $this->getHttpRequest()->getUrl()->getBaseUrl();

		$this->sendJson($this->categoriesFeed->getCategories());
	}

	public function actionReSetVariantCategories()
	{
		ini_set('memory_limit', '2048M');
		set_time_limit(360);
		Debugger::$showBar = false;

		try {
			$this->productsVariants->reSetAllCategories();
		} catch (\Exception $e) {
			$this->sendJson(['status' => 'error', 'message' => $e->getMessage()]);
		}

		$this->sendJson(['status' => 'ok']);
	}
}
