<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model;

use Core\Model\Entities\Redirect;
use Core\Model\Entities\Repository\NestedTreeRepository;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\Strings;
use Core\Model\Helpers\Traits\TPublish;
use Core\Model\Http\Session;
use Core\Model\Lang\Langs;
use Core\Model\Sites;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities;
use EshopCatalog\Model\Entities\Category;
use Exception;
use Nette\Application\LinkGenerator;
use Nette\Caching\Cache;
use Nette\Http\SessionSection;
use Nette\Http\Url;
use Contributte\Translation\Translator;

/**
 * @method Category|null getReference($id)
 * @method Category[] getAll()
 * @method NestedTreeRepository getEr()
 */
class Categories extends BaseEntityService
{
	use TPublish;

	public static bool $hidePrefixAndSuffix = false;

	/** @var Session @inject */
	public $session;

	protected $entityClass = Category::class;

	protected CacheService  $cacheService;
	protected Translator    $translator;
	protected Langs         $langs;
	protected LinkGenerator $linkGenerator;
	protected Sites         $sites;

	protected ?array $cForTree     = null;
	protected ?array $cRootIds     = null;
	protected ?array $cSiteRootIds = null;

	public function __construct(
		CacheService  $cacheService,
		Translator    $translator,
		Langs         $langs,
		LinkGenerator $linkGenerator,
		Sites         $sites
	)
	{
		$this->cacheService  = $cacheService;
		$this->translator    = $translator;
		$this->langs         = $langs;
		$this->linkGenerator = $linkGenerator;
		$this->sites         = $sites;
	}

	public function getSessionSection(): SessionSection
	{
		return $this->session->getSection('EshopCatalog/Admin/Categories');
	}

	public function get($id): ?Category
	{
		return $this->getEr()->createQueryBuilder('c')->addSelect('ct')
			->leftJoin('c.categoryTexts', 'ct')
			->where('c.id = :id')->setParameter('id', $id)
			->getQuery()->getOneOrNullResult();
	}

	public function getRootIds(): array
	{
		if ($this->cRootIds === null) {
			$this->cRootIds = [];

			foreach ($this->em->getRepository(Entities\Category::class)->createQueryBuilder('c')
				         ->select('c.id, ct.alias')
				         ->where('c.lvl = 0 AND c.id = c.root')
				         ->innerJoin('c.categoryTexts', 'ct')
				         ->getQuery()->getArrayResult() as $row) {
				if ($row['alias']) {
					$this->cRootIds[$row['id']] = $row['alias'];
				}
			}
		}

		return $this->cRootIds;
	}

	public function getSiteRootId(?string $siteIdent = null): int
	{
		if ($siteIdent) {
			if ($this->cSiteRootIds === null) {
				$this->cSiteRootIds = array_flip($this->getRootIds());
			}

			$rootId = $this->cSiteRootIds[$siteIdent] ?? null;
		} else {
			$rootId = $this->em->getRepository(Entities\Category::class)->createQueryBuilder('c')
				->select('c.id')
				->where('c.lvl = 0 AND c.id = c.root')
				->getQuery()->setMaxResults(1)->getArrayResult()[0]['id'] ?? null;
		}

		if (!$rootId) {
			$category = new Category;
			$this->em->persist($category);

			foreach ($this->langs->getLangs(false) as $lang) {
				$categoryText = new Entities\CategoryTexts($category, $lang->getTag());
				$categoryText->setAlias($siteIdent);
				$this->em->persist($categoryText);
			}

			$this->em->flush();
			$rootId = $category->getId();

			if ($this->cSiteRootIds !== null) {
				$this->cSiteRootIds[$siteIdent] = $rootId;
			}
		}

		return (int) $rootId;
	}

	/**
	 * @param int|string $id
	 *
	 * @throws Exception
	 */
	public function removeCategory($id): bool
	{
		$this->em->beginTransaction();
		try {
			$category = $this->get($id);

			if (!$category) {
				throw new Exception("category '$id' not found");
			}

			if ($category->getParent()) {
				$siteIdent                   = $this->getRootIds()[$category->getRoot()->getId()];
				$site                        = $this->sites->getSites()[$siteIdent];
				Sites::$currentIdentOverride = $siteIdent;

				$existRedirects = [];
				foreach ($this->em->createQueryBuilder()
					         ->select('r.id, r.relationValue, r.siteIdent, r.lang')
					         ->from(Redirect::class, 'r')
					         ->where('r.package = :package')
					         ->andWhere('r.relationKey = :relationKey')
					         ->andWhere('r.relationValue = :relationValue')
					         ->setParameters([
						         'package'       => 'EshopCatalog',
						         'relationKey'   => 'Category',
						         'relationValue' => $id,
					         ])->getQuery()->getArrayResult() as $row) {
					$existRedirects[$row['lang']] = $row['id'];
				}

				foreach ($site->getDomains() as $domain) {
					$lang = $domain->getLang();

					if (isset($existRedirects[$lang])) {
						continue;
					}

					Sites::$currentLangOverride = $lang;

					$oldUrl = $this->linkGenerator->link('EshopCatalog:Front:Default:category', [
						'id'     => $id,
						'locale' => $lang,
					]);
					$oldUrl = new Url($oldUrl);
					$oldUrl = ltrim($oldUrl->getPath(), '/');

					$newUrl = $this->linkGenerator->link('EshopCatalog:Front:Default:category', [
						'id'     => $category->getParent()->getId(),
						'locale' => $lang,
					]);
					$newUrl = new Url($newUrl);
					$newUrl = ltrim($newUrl->getPath(), '/');

					if ($oldUrl !== $newUrl) {
						$redirect                = new Redirect($category->getCategoryText()->name, $oldUrl, $newUrl);
						$redirect->package       = 'EshopCatalog';
						$redirect->relationKey   = 'Category';
						$redirect->relationValue = (string) $category->getId();
						$redirect->siteIdent     = $siteIdent;
						$redirect->lang          = $lang;

						$this->em->persist($redirect);
					}
				}

				$this->em->flush();

				Sites::$currentIdentOverride = null;
				Sites::$currentLangOverride  = null;
			}

			foreach ($this->em->getRepository(Entities\CategoryTexts::class)->findBy(['id' => $id]) as $ct) {
				$this->em->remove($ct);
			}
			$this->em->remove($category);
			$this->em->flush();

			$this->em->commit();

			$cache = new Cache($this->cacheStorage, Redirect::CACHE_NAMESPACE);
			$cache->clean([Cache::All => true]);

			return true;
		} catch (Exception $e) {
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->rollback();
			}
		}

		return false;
	}

	public function cleanCacheDeep(int $categoryId): void
	{
		$cat = $this->get($categoryId);

		if (!$cat) {
			return;
		}

		$tags = ['category/' . $cat->getId(), 'hierarchy'];
		$loop = static function(array $childs) use (&$loop): void {
			foreach ($childs as $child) {
				$tags[] = 'category/' . $child->getId();

				if ($child->children->toArray()) {
					$loop($child->children->toArray());
				}
			}
		};

		$loop($cat->children->toArray());

		$this->cacheService->categoryCache->clean([Cache::TAGS => $tags]);
	}

	protected function prepareDataForTree(?string $siteIdent = null): array
	{
		$rootIds = $this->getRootIds();
		if ($siteIdent) {
			$cKey = $siteIdent;
		} else {
			$cKey = array_values($rootIds)[0];
		}

		if ($this->cForTree === null) {
			$this->cForTree = [];

			$byRootId = [];
			$qb       = $this->getEr()->createQueryBuilder('c')
				->select('c.id, ct.name, c.lvl, IDENTITY(c.parent) as parent, IDENTITY(c.root) as rootId')
				->leftJoin('c.categoryTexts', 'ct', 'WITH', 'ct.lang = :lang')
				->setParameter('lang', $this->translator->getLocale())
				->groupBy('c.id')
				->addOrderBy('c.root')->addOrderBy('c.lft');

			if (Config::load('category.publishedByLang')) {
				$qb->addSelect('ct.isPublished as isPublished');
			} else {
				$qb->addSelect('c.isPublished as isPublished');
			}

			foreach ($qb->getQuery()->getArrayResult() as $row) {
				$byRootId[$row['rootId']][] = $row;
			}

			foreach ($byRootId as $rootId => $rows) {
				$roots      = [];
				$categories = [];
				$flat       = [];

				foreach ($rows as $c) {
					if ($c['lvl'] == 0) {
						$roots[$c['id']] = [
							'id'   => $c['id'],
							'name' => $c['name'],
						];
						continue;
					}

					$name = !$c['isPublished'] && !self::$hidePrefixAndSuffix ? '[x ' . $c['name'] . ' x]' : $c['name'];

					$categories[$c['id']] = $name;
					$flat[]               = [
						'id'     => $c['id'],
						'parent' => $c['parent'],
						'name'   => $name,
					];
				}

				$this->cForTree[$rootIds[$rootId]] = [
					'roots'      => $roots,
					'categories' => $categories,
					'flat'       => $flat,
				];
			}

			$byRootId = null;
		}

		return $this->cForTree[$cKey] ?? [
			'roots'      => [],
			'categories' => [],
			'flat'       => [],
		];
	}

	public function getOptionsForSelect(?string $siteIdent = null): array { return $this->prepareDataForTree($siteIdent)['categories']; }

	public function getOptionsForSelectNested(?int $rootId = null, array $skip = [], bool $onlyPublished = false): array
	{
		$roots = $this->getRootIds();

		$categories = [];
		$qb         = $this->getEr()->createQueryBuilder('c')->select('c.id, c.lvl, ct.name, IDENTITY(c.root) as root')
			->andWhere('c.root IN (:root)')
			->andWhere('c.lvl > 0')
			->join('c.categoryTexts', 'ct', Join::WITH, 'ct.lang = :lang')
			->setParameters([
				'lang' => $this->translator->getLocale(),
				'root' => $rootId ? [$rootId] : array_keys($roots),
			])
			->groupBy('c.id')
			->addOrderBy('c.lft');

		if ($onlyPublished) {
			if (Config::load('category.publishedByLang')) {
				$qb->andWhere('ct.isPublished = 1');
			} else {
				$qb->andWhere('c.isPublished = 1');
			}
		}

		$titles = [];
		$lastLvl = 0;
		foreach ($qb->getQuery()->getArrayResult() as $category) {
			if ($category['lvl'] === 1) {
				$titles = [];
			}

			if ($category['lvl'] <= $lastLvl) {
				$titles = array_slice($titles, 0, $category['lvl'], true);
			}

			$titles[$category['lvl']] = $category['name'];
			$lastLvl = $category['lvl'];

			if (Arrays::contains($skip, $category['id'])) {
				continue;
			}

			$categories[$roots[$category['root']]][$category['id']] = $category['id'] . ' - ' . implode(' > ' , $titles);
		}

		return $categories;
	}

	public function getFlatTree(?string $siteIdent = null): array { return $this->prepareDataForTree($siteIdent)['flat']; }

	public function getTreeForSelect(?string $siteIdent = null): array
	{
		$data = $this->prepareDataForTree($siteIdent);

		$trees = [];
		foreach ($data['roots'] as $id => $root) {
			$trees[$id] = [
				'name'     => $root['name'],
				'children' => Arrays::buildTree($data['flat'], 'parent', 'id', $root['id']),
			];
		}

		return $trees;
	}

	public function invertPublish(int $id, ?string $lang = null): bool
	{
		$category = $this->get($id);

		if (!$category || ($lang && !$category->getCategoryText($lang))) {
			return false;
		}

		if ($lang) {
			$category->getCategoryText($lang)->isPublished = (int) !$category->getCategoryText($lang)->isPublished;
			$this->em->persist($category);
		} else {
			$category->isPublished = (int) !$category->isPublished;
			$this->em->persist($category);

			foreach ($category->getCategoryTexts() as $text) {
				$text->isPublished = $category->isPublished;
				$this->em->persist($text);
			}
		}

		$this->em->flush();

		$this->cacheStorage->clean([Cache::TAGS => ['eshopNavigation', 'navigation']]);
		$this->cacheService->categoryCache->clean([Cache::TAGS => [
			\EshopCatalog\FrontModule\Model\Categories::CACHE_NAMESPACE,
			'categories',
		]]);

		return true;
	}

	public function clearProductsNotDefaultCategoryNested(string $siteIdent, array $ids): bool
	{
		if (!Config::load('category.allowClearProductsNotDefault')) {
			return false;
		}

		$trees      = [];
		$categories = [];

		try {
			foreach (array_chunk($ids, 200) as $chunk) {
				foreach ($this->getEr()->createQueryBuilder('c')
					         ->select('c.id, c.lft, c.gt, IDENTITY(c.root) as root')
					         ->where('c.id IN (' . implode(',', $chunk) . ')')
					         ->getQuery()->getArrayResult() as $row) {
					$trees[] = [
						'root' => $row['root'],
						'lft'  => $row['lft'],
						'rgt'  => $row['gt'],
					];
				}
			}

			foreach (array_chunk($trees, 100) as $chunk) {
				$arr = [];

				foreach ($chunk as $v) {
					$arr[] = "(c.root = {$v['root']} AND c.lft >= {$v['lft']} AND c.gt <= {$v['rgt']})";
				}

				foreach ($this->getEr()->createQueryBuilder('c')
					         ->select('c.id')
					         ->where(implode(' OR ', $arr))
					         ->getQuery()->getArrayResult() as $row) {
					$categories[] = $row['id'];
				}
			}

			if (!empty($categories)) {
				foreach ($this->em->getRepository(Entities\Product::class)->createQueryBuilder('p')
					         ->innerJoin('p.categoryProducts', 'cp', Join::WITH, 'cp.category IN (' . implode(',', $categories) . ')')
					         ->getQuery()->getResult() as $product) {
					/** @var Entities\Product $product */
					/** @var Entities\ProductInSite|null $site */
					$site = $product->sites[$siteIdent] ?? null;
					if ($site && $site->category && Arrays::contains($categories, $site->category->getId())) {
						continue;
					}

					foreach ($categories as $catId) {
						$cat = $product->categoryProducts->get($catId);
						if ($cat) {
							$product->categoryProducts->remove($catId);
							$this->em->remove($cat);
						}
					}

					$this->em->persist($product);
					$this->em->flush();
				}
			}
		} catch (\Exception $e) {
			return false;
		}

		return true;
	}
}
