<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model\FormContainers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\UI\Form\BaseContainer;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductVideo;
use Nette\Forms\Form;
use Contributte\Translation\Translator;
use Nette\Utils\ArrayHash;
use Nette\Utils\Validators;

class VideoContainer
{
	protected Translator             $translator;
	protected EntityManagerDecorator $em;
	protected CacheService           $cacheService;

	public function __construct(
		Translator             $translator,
		EntityManagerDecorator $em,
		CacheService           $cacheService
	)
	{
		$this->translator   = $translator;
		$this->em           = $em;
		$this->cacheService = $cacheService;
	}

	public function getContainer(bool $multilang = false): BaseContainer
	{
		$container = new BaseContainer;
		$container->addCustomData('template', __DIR__ . '/VideoContainer.latte');

		for ($i = 0; $i < Config::load('maxVideosToUpload', 5); $i++) {
			$videoContainer = new BaseContainer;

			$videoContainer->addText('url')
				->setNullable()
				->setPlaceholder($this->translator->translate('eshopCatalog.videos.url'))
				->addCondition(Form::FILLED)
				->addRule(Form::URL);
			$videoContainer->addFilesManager('file', '')->setNullable();
			$videoContainer->addFilesManager('thumbnail', '')->setNullable();
			$videoContainer->addText('title')
				->setPlaceholder($this->translator->translate('eshopCatalog.videos.title'));
			$videoContainer->addLangsSelect('lang', 'eshopCatalog.videos.lang')
				->setDefaultValue(null)
				->setPrompt(null);
			$videoContainer->addHidden('removeLink');
			$videoContainer->addHidden('id');

			$videoContainer->getComponent('lang')->setItems([null => 'eshopCatalog.videos.lang'] + $videoContainer->getComponent('lang')->getItems());

			$container->addComponent($videoContainer, 'video_' . $i);
		}

		return $container;
	}

	/**
	 * @param array|ArrayHash $values
	 */
	public function setDefaults(BaseContainer $container, $values): void
	{
		$i = 0;
		/** @var BaseContainer $videoContainer */
		foreach ($container->getComponents() as $videoContainer) {
			foreach ($videoContainer->getControls() as $c) {
				if (isset($values[$i][$c->getName()])) {
					$c->setDefaultValue($values[$i][$c->getName()]);
				}
			}
			$i++;
		}
	}

	public function saveData(ArrayHash $data, Product $product): void
	{
		foreach ($data as $video) {
			if (Validators::isNone($video['url']) && Validators::isNone($video['file'])) {
				continue;
			}

			if (Validators::isNone($video['id'])) {
				$this->em->persist(new ProductVideo($product, empty($video['lang']) ? null : $video['lang'], $video['url'], $video['title']));
			} else {
				foreach ($product->getVideos() as $v) {
					if ($v->getId() === ((int) $video['id'])) {
						$v->lang      = empty($video['lang']) ? null : $video['lang'];
						$v->url       = Validators::isNone($video['url']) ? ($video['file'] ?? null) : $video['url'];
						$v->title     = $video['title'] ?? null;
						$v->thumbnail = $video['thumbnail'] ?? null;
						$this->em->persist($v);
						break;
					}
				}
			}
		}

		$this->cacheService->removeVideos([$product->getId()]);
	}

}
