<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model;

use Contributte\Translation\Translator;
use Core\AdminModule\Model\Redirects;
use Core\Model\Countries;
use Core\Model\Entities\EntityRepository;
use Core\Model\Entities\ExtraField;
use Core\Model\Entities\QueryBuilder;
use Core\Model\Entities\Redirect;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\Strings;
use Core\Model\Helpers\Traits\TPublish;
use Core\Model\Lang\Langs;
use Core\Model\Sites;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\AbstractQuery;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\FrontModule\Model\ProductQuery;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Availability;
use EshopCatalog\Model\Entities\CategoryProduct;
use EshopCatalog\Model\Entities\DynamicFeatureProduct;
use EshopCatalog\Model\Entities\FeatureProduct;
use EshopCatalog\Model\Entities\Manufacturer;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductDocument;
use EshopCatalog\Model\Entities\ProductInSite;
use EshopCatalog\Model\Entities\ProductPayment;
use EshopCatalog\Model\Entities\ProductPrice;
use EshopCatalog\Model\Entities\ProductPriceLevel;
use EshopCatalog\Model\Entities\ProductPriceLevelCountry;
use EshopCatalog\Model\Entities\ProductSpedition;
use EshopCatalog\Model\Entities\ProductTag;
use EshopCatalog\Model\Entities\ProductTexts;
use EshopCatalog\Model\Entities\ProductVariant;
use EshopCatalog\Model\Entities\ProductVariantText;
use EshopCatalog\Model\Entities\ProductVideo;
use EshopCatalog\Model\Entities\RelatedProduct;
use EshopCatalog\Model\Entities\Tag;
use EshopCatalog\Model\Entities\VatRate;
use EshopCatalog\Model\Products as BaseProducts;
use EshopProductsComparison\Model\Entities\ProductExport;
use Exception;
use Gallery\AdminModule\Model\Albums;
use Gallery\Model\Entities\Album;
use Gallery\Model\Entities\Image;
use Nette\Application\BadRequestException;
use Nette\Application\LinkGenerator;
use Nette\Caching\Cache;
use Nette\Http\Url;
use Nette\Utils\FileSystem;
use Throwable;
use Tracy\Debugger;

/**
 * @method Product getReference($id)
 * @method Product[] getAll()
 * @method Product|null get($id)
 * @method EntityRepository getEr()
 */
class Products extends BaseEntityService
{
	use TPublish;

	public const SESSION_SECTION = 'EshopCatalog/Admin/Products';

	protected                $entityClass = Product::class;
	protected BaseProducts   $baseProductsService;
	protected Albums         $albums;
	protected Translator     $translator;
	protected CacheService   $cacheService;
	protected Redirects      $redirects;
	protected ProductsFacade $productsFacade;
	protected LinkGenerator  $linkGenerator;
	protected Sites          $sites;
	protected Langs          $langs;
	protected Countries      $countries;

	public function __construct(
		BaseProducts   $baseProductsService,
		Translator     $translator,
		Albums         $albums,
		CacheService   $cacheService,
		Redirects      $redirects,
		ProductsFacade $productsFacade,
		LinkGenerator  $linkGenerator,
		Sites          $sites,
		Langs          $langs,
		Countries      $countries
	)
	{
		$this->baseProductsService = $baseProductsService;
		$this->translator          = $translator;
		$this->albums              = $albums;
		$this->cacheService        = $cacheService;
		$this->redirects           = $redirects;
		$this->productsFacade      = $productsFacade;
		$this->linkGenerator       = $linkGenerator;
		$this->sites               = $sites;
		$this->langs               = $langs;
		$this->countries           = $countries;
	}

	/**
	 * @param array|int  $ids
	 * @param int|string $manufacturerId
	 */
	public function setManufacturer($ids, $manufacturerId): bool
	{
		try {
			/** @var Manufacturer $manufacturer */
			$manufacturer = $this->em->getReference(Manufacturer::class, $manufacturerId);

			foreach ($this->getEr()->findBy(['id' => !is_array($ids) ? [$ids] : $ids]) as $entity) {
				/** @var Product $entity */
				$entity->setManufacturer($manufacturer);
				$this->em->persist($entity);
			}

			$this->em->flush();

			foreach (!is_array($ids) ? [$ids] : $ids as $id) {
				$this->clearProductCache((int) $id);
			}

			return true;
		} catch (Exception $e) {
		}

		return false;
	}

	/**
	 * @param int[]|int  $ids
	 * @param int|string $vatRateId
	 */
	public function setVatRate($ids, $vatRateId): bool
	{
		try {
			/** @var VatRate $vatRate */
			$vatRate = $this->em->getReference(VatRate::class, $vatRateId);

			foreach ($this->getEr()->findBy(['id' => !is_array($ids) ? [$ids] : $ids]) as $entity) {
				/** @var Product $entity */
				$entity->setVateRate($vatRate);
				$this->em->persist($entity);
			}

			$this->em->flush();

			foreach (!is_array($ids) ? [$ids] : $ids as $id) {
				$this->clearProductCache((int) $id);
			}

			return true;
		} catch (Exception $e) {
		}

		return false;
	}

	/**
	 * @param int[]|int  $ids
	 * @param int|string $av
	 */
	public function setAvailabilityAfterSoldOut($ids, $av): bool
	{
		try {
			/** @var Availability $availability */
			$availability = $this->em->getReference(Availability::class, $av);

			foreach ($this->getEr()->findBy(['id' => !is_array($ids) ? [$ids] : $ids]) as $entity) {
				/** @var Product $entity */
				$entity->availabilityAfterSoldOut = $availability;
				$this->em->persist($entity);
			}

			$this->em->flush();

			return true;
		} catch (Exception $e) {
		}

		return false;
	}

	public function setAvailability(array $ids, int $av): bool
	{
		try {
			/** @var Availability $availability */
			$availability = $this->em->getRepository(Availability::class)->find($av);

			foreach ($this->getEr()->findBy(['id' => $ids]) as $entity) {
				/** @var Product $entity */
				$entity->setAvailability($availability);
				$this->em->persist($entity);
			}

			$this->em->flush();

			return true;
		} catch (Exception $e) {
		}

		return false;
	}

	/**
	 * Vrati seznam produktu, ktere obsahuji dany retezec
	 *
	 * @param 1|2|3|4|5|string $hydrate
	 *
	 * @return ArrayCollection|int|mixed|string
	 */
	public function getByTerm(?string $term = null, $hydrate = AbstractQuery::HYDRATE_OBJECT)
	{
		if (is_null($term)) {
			return new ArrayCollection;
		}

		$qb = $this->getEr()->createQueryBuilder('p', 'p.id')
			->addSelect('pt')
			->join('p.productTexts', 'pt', 'WITH', 'pt.lang = :lang')
			->join('p.vatRate', 'vr')
			->setParameter('lang', $this->translator->getLocale())
			->orderBy('pt.name', 'ASC');

		if ($term) {
			$qb->orWhere('pt.name LIKE :term')
				->orWhere('p.code1 LIKE :term')
				->orWhere('p.code2 LIKE :term')
				->orWhere('p.ean LIKE :term')
				->orWhere('p.id LIKE :term')
				->setParameter('term', '%' . $term . '%');
		}

		$qb->andWhere('p.isDeleted = 0')
			->andWhere('p.disableListing = 0');

		return $qb->getQuery()->getResult($hydrate);
	}

	/**
	 * TODO REMOVE?
	 */
	public function getByCode(string $codeKey, string $code): ?Product
	{
		return $this->getEr()->createQueryBuilder('p')
			->addSelect('pt')
			->leftJoin('p.productTexts', 'pt')
			->where('p.' . $codeKey . ' = :code')
			->andWhere('p.isDeleted = 0')
			->setParameter('code', $code)
			->setMaxResults(1)
			->getQuery()->getOneOrNullResult();
	}

	public function deleteDuplicity(): bool
	{
		try {
			set_time_limit(600);
			$data = $this->getEr()->createQueryBuilder('p')->select('p.id, pt.name as name, p.code1, p.code2, count(p) as q')
				->leftJoin('p.productTexts', 'pt')
				->andWhere('p.isDeleted = 0')
				->having('count(p) > 1')->groupBy('pt.name, p.code1, p.code2')->getQuery()->getResult();

			foreach ($data as $row) {
				$products = $this->getEr()->createQueryBuilder('p')->where('pt.name = :name AND p.code1 = :code1')
					->join('p.productTexts', 'pt')->orderBy('p.id')
					->andWhere('p.isDeleted = 0')
					->setParameters(['code1' => $row['code1'], 'name' => $row['name']]);

				if ($row['code2'] == null) {
					$products->andWhere('p.code2 IS NULL');
				} else {
					$products->andWhere('p.code2 = :code2')->setParameter('code2', $row['code2']);
				}

				$products = $products->getQuery()->getResult();

				if (count($products) >= 2) {
					array_pop($products);

					foreach ($products as $product) {
						/** @var Product $product */
						if ($product->getGallery()) {
							FileSystem::delete(WWW_DIR . $product->getGallery()->generatePath());

							$this->em->createQuery('DELETE FROM ' . Image::class . ' i WHERE i.album = :album')->execute(['album' => $product->getGallery()->getId()]);
							$this->em->createQuery('DELETE FROM ' . Album::class . ' a WHERE a.id = :album')->execute(['album' => $product->getGallery()->getId()]);
						}

						$this->em->remove($product);
						$this->em->flush();
					}

					$this->em->clear();
				}
			}
		} catch (Exception $e) {
			return false;
		}

		return true;
	}

	public function setDiscountDisabled(array $ids, int $value): bool
	{
		try {
			foreach ($ids as $id) {
				/** @var Product $product */
				$product = $this->getReference($id);

				$product->discountDisabled = $value ? 1 : 0;
				$this->em->persist($product);
			}

			$this->em->flush();
		} catch (Exception $e) {
			return false;
		}

		return true;
	}

	public function setTag(array $ids, int $tagId): bool
	{
		try {
			/** @var Tag $tag */
			$tag        = $this->em->getReference(Tag::class, $tagId);
			$activeTags = [];
			foreach ($this->em->getRepository(ProductTag::class)->createQueryBuilder('pt')
				         ->select('IDENTITY(pt.product) as product, IDENTITY(pt.tag) as tag')
				         ->where('pt.product IN (:ids)')
				         ->andWhere('pt.tag = :tag')
				         ->setParameters([
					         'ids' => $ids,
					         'tag' => $tagId,
				         ])->getQuery()->getArrayResult() as $row) {
				$activeTags[$row['product']] = $row['tag'];
			}

			foreach ($ids as $id) {
				if (isset($activeTags[$id])) {
					continue;
				}

				$product = $this->getReference($id);

				$productTag = new ProductTag($product, $tag);
				$this->em->persist($productTag);
			}

			$this->em->flush();

			$this->cacheService->defaultCache->clean([Cache::TAGS => ['productsByTag']]);

			foreach ($ids as $id) {
				$this->clearProductCache((int) $id);
			}
		} catch (Exception $e) {
			return false;
		}

		return true;
	}

	public function removeTag(array $ids, int $tagId): bool
	{
		try {
			$this->em->getRepository(ProductTag::class)->createQueryBuilder('pt')
				->delete()
				->where('pt.tag = :tag')
				->andWhere('pt.product IN (:ids)')
				->setParameters([
					'tag' => $tagId,
					'ids' => $ids,
				])->getQuery()->execute();
			$this->em->flush();

			$this->cacheService->defaultCache->clean([Cache::TAGS => ['productsByTag']]);

			foreach ($ids as $id) {
				$this->clearProductCache((int) $id);
			}
		} catch (Exception $e) {
			bdump($e->getMessage());

			return false;
		}

		return true;
	}

	public function createVariant(int $id): Product
	{
		$product = $this->get($id);

		if (!$product->isVariant()) {
			$baseProductVariant                 = new ProductVariant($product, (string) ($this->getLastVariantId() + 1));
			$baseProductVariant->isDefault      = 1;
			$baseProductVariant->createdDefault = $product->getCreated();

			$this->em->persist($baseProductVariant);
		} else {
			if ($product->isVariant()->isDefault === 1) {
				$baseProductVariant = $product->isVariant();
			} else {
				foreach ($product->variants->toArray() as $v) {
					if ($v->isDefault === 1) {
						$product            = $v->product;
						$baseProductVariant = $v;
						break;
					}
				}
			}
		}

		if (!isset($baseProductVariant)) {
			throw new BadRequestException;
		}

		$variant = new Product;

		$texts = [];
		foreach ($product->getTexts()->toArray() as $text) {
			$this->em->detach($text);
			/** @var ProductTexts $text */
			$text->setProduct($variant);
			$texts[] = $text;
		}

		foreach ($product->getProductTags()->toArray() as $tag) {
			$this->em->detach($tag);
			$variant->addProductTag($tag);
			$this->em->persist($tag);
		}

		$variant->setProductTexts($texts);

		if (!$product->getGallery()) {
			$album = new Album(UPLOADS_PATH . '/products');
			$this->em->persist($album);
			$product->setGallery($album);
		}

		$variant->setGallery($product->getGallery());
		$variant->setVateRate($product->getVateRate());

		$variant->price        = $product->price;
		$variant->recyclingFee = $product->recyclingFee;
		if (Config::load('enableCountryPrices')) {
			$variant->prices              = new ArrayCollection($product->clonePrices($variant));
			$variant->priceLevelCountries = new ArrayCollection($product->clonePriceLevelCounties($variant));
		}

		if (Config::load('createVariant.copyAllFields')) {
			$variant->code1 = $product->code1;
			$variant->code2 = $product->code2;
			$variant->setManufacturer($product->getManufacturer());
			$variant->condition = $product->condition;

			foreach ($product->getFeatureProducts() as $featureProduct) {
				$this->em->detach($featureProduct);
				$featureProduct->setProduct($variant);
				$this->em->persist($featureProduct);
			}

			foreach ($product->dynamicFeatures as $dynamicFeature) {
				$this->em->detach($dynamicFeature);
				$dynamicFeature->setProduct($variant);
				$this->em->persist($dynamicFeature);
			}

			foreach ($this->em->getRepository(ProductExport::class)->createQueryBuilder('pe')
				         ->where('pe.id = :id')
				         ->setParameter('id', $product->getId())
				         ->getQuery()->getResult() as $row) {
				/** @var ProductExport $row */
				$this->em->detach($row);
				$row->id = $variant;
				$this->em->persist($row);
			}

			foreach ($product->documents as $document) {
				$this->em->detach($document);
				$document->product = $variant;
				$this->em->persist($document);
			}

			foreach ($product->videos as $video) {
				$this->em->detach($video);
				$video->product = $variant;
				$this->em->persist($video);
			}

			foreach ($product->relatedProducts as $relatedProduct) {
				$this->em->detach($relatedProduct);
				$relatedProduct->setOrigin($variant);
				$this->em->persist($relatedProduct);
			}
		}

		$this->em->persist($variant);

		$productVariant                 = new ProductVariant($variant, $baseProductVariant->getVariantId());
		$productVariant->createdDefault = $baseProductVariant->createdDefault;

		foreach (Config::load('createVariant.useFields') as $field) {
			$productVariant->{'use' . Strings::firstUpper($field)} = 1;
		}

		$this->em->persist($productVariant);

		$this->em->flush();

		$conn = $this->em->getConnection();
		foreach ($conn->fetchAllAssociative("SELECT site, is_active, category_id FROM eshop_catalog__product_in_site WHERE product_id = ?", [$id]) as $row) {
			$conn->executeStatement("INSERT INTO eshop_catalog__product_in_site (product_id, site, is_active, category_id) VALUES (?, ?, ?, ?)", [
				$variant->getId(),
				$row['site'],
				$row['is_active'],
				$row['category_id'],
			]);
		}

		foreach ($conn->fetchAllAssociative("SELECT id_category FROM eshop_catalog__category_product WHERE id_product = ?", [$id]) as $row) {
			$conn->executeStatement("INSERT INTO eshop_catalog__category_product (id_product, id_category) VALUES (?, ?)", [
				$variant->getId(),
				$row['id_category'],
			]);
		}

		return $variant;
	}

	public function removeFromVariants(Product $product): bool
	{
		if (!$product->getVariantParent() || !$product->isVariant()) {
			return true;
		}

		$productId       = $product->getId();
		$productParentId = $product->getVariantParent()->getId();

		if ($productParentId && $product->getGallery()
			&& $product->getGallery()->getId() === $product->getVariantParent()->getGallery()->getId()) {
			$newGallery = $this->albums->cloneAlbum($product->getGallery());
			$product->setGallery($newGallery);

			$this->em->persist($product);
			$this->em->flush();
		}

		$variantId = $product->isVariant()->getVariantId();

		$allVariants = $this->em->getRepository(ProductVariant::class)->createQueryBuilder('pv')
			->where('pv.variantId = :variantId')->setParameter('variantId', $variantId)
			->getQuery()->getResult();

		if (count($allVariants) === 2) {
			foreach ($allVariants as $v) {
				$this->em->remove($v);
			}
			$this->em->flush();
		} else {
			$this->em->remove($product->isVariant());
			$this->em->flush();
		}

		$conn = $this->em->getConnection();
		$conn->executeQuery("DELETE FROM eshop_catalog__product_in_site WHERE product_id = :id", ['id' => $productId]);
		$conn->executeQuery("DELETE FROM eshop_catalog__category_product where id_product = :id", ['id' => $productId]);

		foreach ($conn->fetchAllAssociative("SELECT * FROM eshop_catalog__product_in_site WHERE product_id = :id", ['id' => $productParentId]) as $row) {
			$conn->executeQuery("INSERT INTO eshop_catalog__product_in_site (product_id, site, is_active, category_id) VALUES (:id, :site, :active, :category)", [
				'id'       => $productId,
				'site'     => $row['site'],
				'active'   => $row['is_active'],
				'category' => $row['category_id'],
			]);
		}

		foreach ($conn->fetchAllAssociative("SELECT * FROM eshop_catalog__category_product WHERE id_product = :id", ['id' => $productParentId]) as $row) {
			$conn->executeQuery("INSERT INTO eshop_catalog__category_product (id_product, id_category) VALUES (:id, :category)", [
				'id'       => $productId,
				'category' => $row['id_category'],
			]);
		}

		$this->cacheService->productCache->clean([
			Cache::Tags => ['variants'],
		]);

		$this->clearProductCache($productId);
		$this->clearProductCache($productParentId);

		return true;
	}

	public function setAsMainVariant(int $prodId): bool
	{
		try {
			/** @var ProductVariant|null $variant */
			$variant = $this->em->getRepository(ProductVariant::class)->createQueryBuilder('pv')
				->where('pv.product = :id')
				->setParameter('id', $prodId)
				->getQuery()->getOneOrNullResult();

			if (!$variant) {
				return false;
			}

			/** @var ProductVariant|null $parent */
			$parent = $this->em->getRepository(ProductVariant::class)->createQueryBuilder('pv')
				->where('pv.variantId = :variantId')
				->andWhere('pv.isDefault = 1')
				->setParameter('variantId', $variant->getVariantId())
				->getQuery()->getOneOrNullResult();

			if (!$parent) {
				return false;
			}

			$variant->isDefault = 1;
			$parent->isDefault  = 0;

			$this->em->persist($variant);
			$this->em->persist($parent);
			$this->em->flush();

			$parentId = $parent->getProduct()->getId();
			$this->em->createQueryBuilder()->delete(ProductInSite::class, 'pis')
				->where('pis.product = ' . $prodId)->getQuery()->execute();
			$this->em->createQueryBuilder()->update(ProductInSite::class, 'pis')
				->set('pis.product', $prodId)
				->where('pis.product = ' . $parentId)->getQuery()->execute();
			$this->em->createQueryBuilder()->delete(ProductInSite::class, 'pis')
				->where('pis.product = ' . $parentId)->getQuery()->execute();

			$this->em->createQueryBuilder()->delete(CategoryProduct::class, 'cp')
				->where('cp.product = ' . $prodId)->getQuery()->execute();
			$this->em->createQueryBuilder()->update(CategoryProduct::class, ' cp')
				->set('cp.product', $prodId)
				->where('cp.product = ' . $parentId)->getQuery()->execute();
			$this->em->createQueryBuilder()->delete(CategoryProduct::class, 'cp')
				->where('cp.product = ' . $parentId)->getQuery()->execute();

			return true;
		} catch (Exception $e) {
		}

		return false;
	}

	public function setAsVariantFor(int $prodId, int $parentId, array $variantNames = []): bool
	{
		if ($prodId === $parentId) {
			return false;
		}

		$product = null;
		$parent  = null;
		foreach ($this->getEr()->createQueryBuilder('p')->addSelect('isVar')
			         ->where('p.id IN (:ids)')->setParameter('ids', [$parentId, $prodId])
			         ->andWhere('p.isDeleted = 0')
			         ->leftJoin('p.isVariant', 'isVar')
			         ->getQuery()->getResult() as $row) {
			/** @var Product $row */
			if ($row->getId() == $prodId) {
				$product = $row;
			} else {
				$parent = $row;
			}
		}

		if (!$product || !$parent) {
			return false;
		}

		if ($parent->isVariant() && $parent->isVariant()->isDefault === 0) {
			return false;
		}

		if (!$parent->isVariant()) {
			$parentVariant                 = new ProductVariant($parent, (string) ($this->getLastVariantId() + 1));
			$parentVariant->createdDefault = $parent->getCreated();
			$parentVariant->isDefault      = 1;

			$this->em->persist($parentVariant);

			foreach ($variantNames as $k => $v) {
				$vaText       = new ProductVariantText($parentVariant, $k);
				$vaText->name = $v;

				$this->em->persist($vaText);
			}
		} else {
			$parentVariant = $parent->isVariant();
		}

		if (!$product->getGallery()) {
			if (!$parent->getGallery()) {
				$album = new Album(UPLOADS_PATH . '/products');
				$this->em->persist($album);
				$parent->setGallery($album);
				$this->em->persist($parent);
			}

			$product->setGallery($parent->getGallery());
		}

		$product->setVateRate($parent->getVateRate());

		if (!$product->isVariant()) {
			$productVariant = new ProductVariant($product, $parentVariant->getVariantId());
			/** @var ProductTexts $prodText */
			$prodText = $product->getText();

			$productVariant->useName             = $prodText->name ? 1 : 0;
			$productVariant->useName2            = $prodText->name2 ? 1 : 0;
			$productVariant->usePrice            = $product->price ? 1 : 0;
			$productVariant->usePurchasePrice    = $product->purchasePrice ? 1 : 0;
			$productVariant->useShortDescription = $prodText->shortDescription ? 1 : 0;
			$productVariant->useDescription      = $prodText->description ? 1 : 0;
			$productVariant->usePriceLevels      = $product->getPriceLevels()->count() ? 1 : 0;
			$productVariant->createdDefault      = $parentVariant->createdDefault;

			if (Config::load('product.allowRetailPrice')) {
				$productVariant->useRetailPrice = $product->retailPrice ? 1 : 0;
			}
		} else {
			if ($product->isVariant()->isDefault === 1) {
				throw new Exception('eshopCatalog.product.productIsVariantParent');
			}

			$productVariant = $product->isVariant();
			$productVariant->setVariantId($parentVariant->getVariantId());
		}

		$this->em->persist($productVariant);
		$this->em->persist($product);
		$this->em->flush();

		$conn = $this->em->getConnection();
		$conn->executeStatement("DELETE FROM eshop_catalog__product_in_site WHERE product_id = ?", [$prodId]);

		foreach ($conn->fetchAllAssociative("SELECT site, is_active, category_id FROM eshop_catalog__product_in_site WHERE product_id = ?", [$parentId]) as $row) {
			$conn->executeStatement("INSERT INTO eshop_catalog__product_in_site (product_id, site, is_active, category_id) VALUES (?, ?, ?, ?)", [
				$prodId,
				$row['site'],
				$row['is_active'],
				$row['category_id'],
			]);
		}

		$conn->executeStatement("DELETE FROM eshop_catalog__category_product WHERE id_product = ?", [$prodId]);

		foreach ($conn->fetchAllAssociative("SELECT id_category FROM eshop_catalog__category_product WHERE id_product = ?", [$parentId]) as $row) {
			$conn->executeStatement("INSERT INTO eshop_catalog__category_product (id_product, id_category) VALUES (?, ?)", [
				$prodId,
				$row['id_category'],
			]);
		}

		$this->cacheService->productCache->clean([
			Cache::Tags => ['variants'],
		]);

		return true;
	}

	public function getLastVariantId(): int
	{
		return (int) ($this->em->getConnection()->fetchAllAssociative("SELECT max(CONVERT(variant_id, SIGNED INTEGER)) as maxVariant, variant_id REGEXP '^[0-9]+$' AS isNumeric 
				FROM `eshop_catalog__product_variant` 
				GROUP BY isNumeric 
				HAVING isNumeric = 1")[0]['maxVariant'] ?? 0);
	}

	public function duplicateProduct(int $productId): ?int
	{
		set_time_limit(120);

		/** @var Product $originProduct */
		$originProduct = $this->get($productId);
		$originGallery = $originProduct->getGallery();
		$gallery       = null;

		if ($originGallery) {
			$this->em->getConnection()->insert('gallery__album', [
				'base_path'    => $originGallery->basePath,
				'is_published' => 1,
			]);
			$galleryId = $this->em->getConnection()->lastInsertId();
			$gallery   = $this->em->getRepository(Album::class)->find($galleryId);
		}

		$this->em->beginTransaction();
		try {
			$newProduct = clone $originProduct;

			$newProduct->ean = null;

			// Radeji smazani informace o variante
			$newProduct->isVariant = new ArrayCollection();
			$newProduct->variants  = new ArrayCollection();
			$this->em->persist($newProduct);
			$this->em->flush();

			// Texty
			foreach ($originProduct->getTexts()->toArray() as $text) {
				/** @var ProductTexts $text */
				$entity = clone $text;
				$entity->setProduct($newProduct);
				$this->em->persist($entity);
			}
			$this->em->flush();

			// Kategorie
			foreach ($originProduct->getCategoryProducts()->toArray() as $cp) {
				/** @var CategoryProduct $cp */
				$entity = new CategoryProduct($newProduct, $cp->getCategory());
				$this->em->persist($entity);
			}
			$this->em->flush();

			// Features
			foreach ($originProduct->getFeatureProducts()->toArray() as $fp) {
				/** @var FeatureProduct $fp */
				$entity = new FeatureProduct($newProduct, $fp->getFeatureValue());
				$this->em->persist($entity);
			}
			$this->em->flush();

			// Dynamic features
			foreach ($originProduct->dynamicFeatures as $df) {
				$entity = new DynamicFeatureProduct($newProduct, $df->getFeature(), $df->value);
				$this->em->persist($entity);
			}
			$this->em->flush();

			// Tags
			foreach ($originProduct->getProductTags()->toArray() as $tag) {
				/** @var ProductTag $tag */
				$entity            = new ProductTag($newProduct, $tag->getTag());
				$entity->validFrom = $tag->validFrom;
				$entity->validTo   = $tag->validTo;
				$this->em->persist($entity);
			}
			$this->em->flush();

			// Gallery
			if ($originGallery && $gallery) {
				foreach ($originGallery->getImages()->toArray() as $img) {
					/** @var Image $img */
					$image = clone $img;
					$image->setAlbum($gallery);
					$image->setCloneOf($img);
					$image->clear();
					$img->setLastUse();

					$this->em->persist($img);
					$this->em->persist($image);
					$this->em->flush();
				}

				$newProduct->setGallery($gallery);
				$this->em->flush();
			}
			$this->em->flush();

			// Price levels
			foreach ($originProduct->getPriceLevels()->toArray() as $pl) {
				/** @var ProductPriceLevel $pl */
				$entity        = new ProductPriceLevel($newProduct, $pl->getGroupId());
				$entity->price = $pl->price;
				$this->em->persist($entity);
			}
			$this->em->flush();

			// Sites
			foreach ($originProduct->sites->toArray() as $site) {
				/** @var ProductInSite $site */
				$entity = new ProductInSite($newProduct, $site->getSite());
				$entity->setActive($site->isActive() ? 1 : 0);
				$entity->category = $site->category;
				$this->em->persist($entity);
			}
			$this->em->flush();

			// Documents
			foreach ($originProduct->documents->toArray() as $doc) {
				/** @var ProductDocument $doc */
				$entity = new ProductDocument($doc->lang, $doc->name, $doc->file, $newProduct);
				$this->em->persist($entity);
			}
			$this->em->flush();

			// RelatedProducts
			foreach ($originProduct->getRelatedProducts()->toArray() as $rl) {
				/** @var RelatedProduct $rl */
				$entity = new RelatedProduct($newProduct, $rl->getProduct(), $rl->getGroup());
				$this->em->persist($entity);
			}
			$this->em->flush();

			if (Config::load('enableCountryPrices')) {
				$newProduct->prices              = new ArrayCollection($originProduct->clonePrices($newProduct));
				$newProduct->priceLevelCountries = new ArrayCollection($originProduct->clonePriceLevelCounties($newProduct));
			}
			$this->em->flush();

			$this->em->persist($newProduct);
			$this->em->flush();
			$this->em->commit();

			// Znovu smazani informaci o variantach k novemu produktu
			$this->em->getConnection()->delete('eshop_catalog__product_variant', [
				'product_id' => $newProduct->getId(),
			]);

			return $newProduct->getId();
		} catch (Exception $e) {
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->rollback();

				if (isset($galleryId)) {
					$this->em->getConnection()->delete('gallery__album', [
						'id' => $galleryId,
					]);
					$this->em->flush();
				}
			}
		}

		return null;
	}

	/**
	 * @param int[]|int $prodIds
	 *
	 * @throws Throwable
	 */
	public function remove($prodIds): bool
	{
		try {
			$allIds = is_array($prodIds) ? $prodIds : [$prodIds];

			ProductsFacade::setMode(ProductsFacade::MODE_CHECKOUT);
			ProductQuery::$ignoreProductPublish = true;
			$this->productsFacade->clearTemp();

			$soldOutAv = $this->em->getRepository(Availability::class)->findOneBy(['ident' => Availability::SOLD_OUT]);

			foreach (array_chunk($allIds, 250) as $ids) {
				$this->em->beginTransaction();
				$products = $this->productsFacade->getProducts($ids);

				$productsInCategories = [];
				foreach ($this->em->createQueryBuilder()->select('IDENTITY(pis.product) as product, IDENTITY(pis.site) as site, IDENTITY(pis.category) as category')
					         ->from(ProductInSite::class, 'pis')
					         ->where('pis.product IN (:ids)')
					         ->setParameters([
						         'ids' => $ids,
					         ])->getQuery()->getArrayResult() as $row) {
					$productsInCategories[$row['product']][$row['site']] = $row['category'];
				}

				$existRedirects = [];
				foreach ($this->em->createQueryBuilder()
					         ->select('r.id, r.relationValue, r.siteIdent, r.lang')
					         ->from(Redirect::class, 'r')
					         ->where('r.package = :package')
					         ->andWhere('r.relationKey = :relationKey')
					         ->andWhere('r.relationValue IN (:relationValue)')
					         ->setParameters([
						         'package'       => 'EshopCatalog',
						         'relationKey'   => 'Product',
						         'relationValue' => $ids,
					         ])->getQuery()->getArrayResult() as $row) {
					$existRedirects[$row['siteIdent']][$row['lang']][$row['relationValue']] = $row['id'];
				}

				foreach ($products as $product) {
					$urlsByLang = [];

					foreach ($product->inSites as $siteIdent) {
						$site                        = $this->sites->getSites(false)[$siteIdent];
						Sites::$currentIdentOverride = $site->getIdent();

						foreach ($site->getDomains() as $domain) {
							$lang = $domain->getLang();
							if (!$domain->isActive || isset($existRedirects[$siteIdent][$lang][$product->getId()])) {
								continue;
							}

							if (!isset($urlsByLang[$lang])) {
								Sites::$currentLangOverride = $lang;
								$url                        = new Url($this->linkGenerator->link('EshopCatalog:Front:Default:product', [
									'id'     => $product->getId(),
									'locale' => $lang,
								]));

								$urlsByLang[$lang] = ltrim($url->getPath(), '/');
							}

							$catUrl = $this->linkGenerator->link('EshopCatalog:Front:Default:category', [
								'id'     => $productsInCategories[$product->getId()][$siteIdent],
								'locale' => $lang,
							]);
							$catUrl = new Url($catUrl);
							$catUrl = ltrim($catUrl->getPath(), '/');

							if ($urlsByLang[$lang] !== $catUrl) {
								$redirect                = new Redirect($product->name, $urlsByLang[$lang], $catUrl);
								$redirect->package       = 'EshopCatalog';
								$redirect->relationKey   = 'Product';
								$redirect->relationValue = (string) $product->getId();
								$redirect->siteIdent     = $siteIdent;
								$redirect->lang          = $lang;

								$this->em->persist($redirect);
							}
						}
					}

					$this->em->flush();

					Sites::$currentLangOverride  = null;
					Sites::$currentIdentOverride = null;
				}

				foreach ($ids as $id) {
					$entity = $this->get($id);

					if (!$entity) {
						return false;
					}

					$stockDelete = class_exists('EshopStock\DI\EshopStockExtension') && $this->em->getConnection()
							->fetchOne("SELECT id FROM eshop_stock__supply_product WHERE product_id = :id LIMIT 1", ['id' => $id]);

					if ($stockDelete) {
						foreach ([
							         ProductPrice::class,
							         ProductPriceLevelCountry::class,
							         DynamicFeatureProduct::class,
							         ProductTag::class,
							         ProductInSite::class,
							         ProductVariant::class,
							         ProductDocument::class,
							         ProductVideo::class,
							         ProductSpedition::class,
							         ProductPayment::class,
							         CategoryProduct::class,
							         FeatureProduct::class,
						         ] as $class) {
							$this->em->createQueryBuilder()->delete($class, 'e')
								->where('e.product = :product')
								->setParameter('product', $id)
								->getQuery()->execute();
						}

						$this->em->createQueryBuilder()->delete(ProductPriceLevel::class, 'e')
							->where('e.productId = :product')
							->setParameter('product', $id)
							->getQuery()->execute();

						$this->em->createQueryBuilder()->delete(RelatedProduct::class, 'e')
							->where('e.origin = :product')
							->setParameter('product', $id)
							->getQuery()->execute();

						$this->em->createQueryBuilder()->delete(ExtraField::class, 'e')
							->where('e.sectionName = :sectionName')
							->andWhere('e.sectionKey = :sectionKey')
							->setParameters([
								'sectionName' => Product::EXTRA_FIELD_SECTION,
								'sectionKey'  => $id,
							])->getQuery()->execute();

						if (class_exists(ProductExport::class)) {
							$this->em->createQueryBuilder()->delete(ProductExport::class, 'e')
								->where('e.id = :product')
								->setParameter('product', $id)
								->getQuery()->execute();
						}

						$entity->setAvailability($soldOutAv);
						$entity->setMoreData([]);
						$entity->availabilityAfterSoldOut = $soldOutAv;
						$entity->isPublished              = 0;
						$entity->position                 = null;
						$entity->countryOfOrigin          = null;

						$entity->isDeleted = 1;
						$this->em->persist($entity);

						if ($entity->getGallery()) {
							$this->albums->removeAlbum($entity->getGallery());
						}

						$this->em->persist($entity);
					} else {
						$this->em->remove($entity);
					}

					foreach ($entity->suppliers as $supplier) {
						$this->em->getConnection()->insert('eshop_catalog__product_supplier_skip', [
							'code'        => $supplier->code,
							'id_supplier' => $supplier->getSupplier()->getId(),
						]);
					}

					$this->em->flush($entity);
				}

				$this->em->commit();

				$cache = new Cache($this->cacheStorage, Redirect::CACHE_NAMESPACE);
				$cache->clean([Cache::All => true]);
			}
		} catch (Exception $e) {
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->rollback();
			}

			return false;
		}

		return true;
	}

	public function updateVariantCategories(Product $product, string $variantId): void
	{
		$conn     = $this->em->getConnection();
		$variants = [];

		$conn->beginTransaction();
		try {
			$categories        = [];
			$variantCategories = [];

			foreach ($this->em->createQueryBuilder()
				         ->select('IDENTITY(pv.product) as id')
				         ->from(ProductVariant::class, 'pv')
				         ->where('pv.variantId = :variantId')
				         ->andWhere('pv.isDefault = :isDefault')
				         ->setParameters([
					         'variantId' => $variantId,
					         'isDefault' => 0,
				         ])->getQuery()->getArrayResult() as $row) {
				$variants[]                    = $row['id'];
				$variantCategories[$row['id']] = [];
			}

			if (!$variants) {
				return;
			}

			foreach ($this->em->getConnection()
				         ->fetchAllAssociative("SELECT id_product, id_category FROM eshop_catalog__category_product WHERE id_product IN (" . implode(',', [$product->getId()] + $variants) . ")") as $row) {
				if ($row['id_product'] == $product->getId()) {
					$categories[] = $row['id_category'];
				} else {
					$variantCategories[$row['id_product']][] = $row['id_category'];
				}
			}

			$ins = [];
			foreach ($variantCategories as $variantId => $variantCats) {
				foreach (array_diff($categories, $variantCats) as $catId) {
					$ins[] = "('$variantId', $catId)";
				}
				foreach (array_diff($variantCats, $categories) as $catId) {
					$conn->executeQuery(sprintf("DELETE FROM `eshop_catalog__category_product` WHERE `id_product` = '%s' AND `id_category` = %s", $variantId, $catId));
				}
			}

			if ($ins) {
				$this->em->getConnection()->executeStatement("INSERT IGNORE INTO eshop_catalog__category_product (id_product, id_category) VALUES " . implode(', ', $ins));
			}

			/** @var ProductInSite[][] $variantsSites */
			$variantsSites = [];
			/** @var ProductInSite[] $productSites */
			$productSites = [];
			foreach ($this->em->getRepository(ProductInSite::class)->createQueryBuilder('pis')
				         ->where('pis.product IN (' . implode(',', $variants) . ')')
				         ->getQuery()->getResult() as $row) {
				/** @var ProductInSite $row */
				$variantsSites[$row->getProduct()->getId()][$row->getSite()->getIdent()] = $row;
			}

			foreach ($product->sites as $site) {
				$productSites[$site->getSite()->getIdent()] = $site;
			}

			// Add
			foreach ($variants as $variantId) {
				foreach (array_diff_key($productSites, $variantsSites[$variantId] ?? []) as $v) {
					/** @var ProductInSite $v */
					$pis           = new ProductInSite(
						$this->getReference($variantId),
						$v->getSite(),
					);
					$pis->category = $v->category;
					$pis->setActive((int) $v->isActive());

					$this->em->persist($pis);
				}
			}

			$this->em->flush();
			$conn->commit();

			$this->cacheService->productCache->clean([
				Cache::Tags => ['variants'],
			]);
		} catch (Exception $e) {
			if ($conn->isTransactionActive()) {
				$conn->rollBack();
			}
		}
	}

	public function allowImportPrice(array $ids, bool $allow): bool
	{
		if (empty($ids)) {
			return false;
		}

		try {
			/** @var QueryBuilder $qb */
			$qb       = $this->em->getRepository(Product::class)->createQueryBuilder('p');
			$iterator = $qb->where('p.id IN (' . implode(',', $ids) . ')')->getIterator();

			$i = 0;
			while (($product = $iterator->next()) !== false) {
				/** @var Product $product */
				$product = $product[0];
				$product->setMoreDataValue('stopImportPrice', (int) !$allow);
				$this->em->persist($product);
				$i++;

				if ($i % 20 === 0) {
					$this->em->flush();
					$this->em->clear();
					gc_collect_cycles();
				}
			}

			$this->em->flush();

			return true;
		} catch (Exception $e) {
			Debugger::log($e);
		}

		return false;
	}

	public function invertPublish(int $id, ?string $lang = null): bool
	{
		$prod = $this->get($id);

		if (!$prod || ($lang && !$prod->getText($lang))) {
			return false;
		}

		if ($lang) {
			$prod->getText($lang)->isPublished = (int) !$prod->getText($lang)->isPublished;
			$this->em->persist($prod);
		} else {
			$prod->isPublished = (int) !$prod->isPublished;
			$this->em->persist($prod);

			foreach ($prod->getTexts() as $text) {
				$text->isPublished = $prod->isPublished;
				$this->em->persist($text);
			}
		}

		$this->em->flush();

		foreach ($prod->getTexts() as $text) {
			$this->cacheService->productCache->remove('product/' . $id . '/' . $text->getLang());
		}

		return true;
	}

	public function setPublish(int $id, int $state): bool
	{
		if ($item = $this->get($id)) {
			$item->isPublished = (int) $state;

			if (Config::load('product.publishedByLang')) {
				foreach ($item->getTexts() as $text) {
					$text->isPublished = (int) $state;
					$this->em->persist($text);
				}
			}

			$this->em->persist($item);
			$this->em->flush();

			foreach ($item->getTexts() as $text) {
				$this->cacheService->productCache->remove('product/' . $id . '/' . $text->getLang());
			}

			return true;
		}

		return false;
	}

	public function clearProductCache(int $productId): void
	{
		foreach ($this->translator->getLocalesWhitelist() as $lang) {
			$this->cacheService->defaultCache->remove('productsExtraFields_' . $lang);
			$this->cacheService->productCache->remove('product/' . $productId . '/' . $lang);
			$this->cacheService->productCache->remove('link/' . $productId . '/' . $lang);
		}

		$this->cacheService->priceCache->remove('retailPrice/' . $productId);

		foreach ($this->countries->getDao() as $country) {
			$this->cacheService->priceCache->remove('retailPriceCountry/' . Strings::lower($country->getId()) . '/' . $productId);
		}

		$cache = new Cache($this->cacheStorage, \EshopCatalog\FrontModule\Model\Tags::CACHE_NAMESPACE);
		$cache->remove('idsForProducts');

		$this->cacheService->productCache->remove('variantBasicByProduct');
	}

	public function setParentGallery(Product $product): void
	{
		if (!$product->isVariant() || $product->isVariant()->isDefault) {
			return;
		}

		$parent = $product->getVariantParent();
		if (!$parent || !$parent->getGallery()) {
			return;
		}

		if ($product->getGallery()) {
			if ($product->getGallery()->getId() === $parent->getGallery()->getId()) {
				return;
			}

			$this->albums->removeAlbum($product->getGallery());
		}

		$this->em->getConnection()->update('eshop_catalog__product', [
			'gallery_id' => $parent->getGallery()->getId(),
		], [
			'id' => $product->getId(),
		]);
	}

	public function findAndSetEmptyVariantGalleries(): void
	{
		// Vyhledani  hlavnich variant bez galerie a projduti jejich variant jestli nemaji galerii
		foreach ($this->em->getConnection()->fetchAllAssociative("SELECT p.id, pv.variant_id FROM eshop_catalog__product p
			INNER JOIN eshop_catalog__product_variant pv ON pv.product_id = p.id AND pv.is_default = 1
			WHERE p.gallery_id IS NULL") as $row) {
			$hasGallery = $this->em->getConnection()->fetchAssociative("SELECT p.id, p.gallery_id FROM eshop_catalog__product p
				INNER JOIN eshop_catalog__product_variant pv ON p.id = pv.product_id AND pv.is_default = 0 AND pv.variant_id = ?
				WHERE p.gallery_id IS NOT NULL 
				LIMIT 1", [$row['variant_id']]);

			if ($hasGallery) {
				$this->em->getConnection()->executeStatement('UPDATE eshop_catalog__product SET gallery_id = ' . $hasGallery['gallery_id'] . ' 
					WHERE id = ' . $row['id']);
				$this->em->getConnection()->executeStatement('UPDATE eshop_catalog__product SET gallery_id = null 
					WHERE id = ' . $hasGallery['id']);

				foreach ($this->langs->getLangs(false) as $lang) {
					$this->cacheService->productCache->remove('product/' . $hasGallery['id'] . '/' . $lang->getTag());
				}
			}

			foreach ($this->langs->getLangs(false) as $lang) {
				$this->cacheService->productCache->remove('product/' . $row['id'] . '/' . $lang->getTag());
			}
		}

		// Nastaveni galerii pro varianty ktere ji nemaji
		$variantGalleries = [];
		foreach ($this->em->getConnection()->fetchAllAssociative("SELECT p.id, pv.variant_id FROM eshop_catalog__product p
			INNER JOIN eshop_catalog__product_variant pv ON pv.product_id = p.id AND pv.is_default = 0
			WHERE p.gallery_id IS NULL") as $row) {

			if (!array_key_exists($row['variant_id'], $variantGalleries)) {
				$variantGalleries[$row['variant_id']] = $this->em->getConnection()->fetchOne("SELECT gallery_id FROM eshop_catalog__product p
					INNER JOIN eshop_catalog__product_variant pv ON p.id = pv.product_id AND pv.is_default = 1 AND pv.variant_id = ?
					WHERE p.gallery_id IS NOT NULL", [$row['variant_id']]);
			}

			if ($variantGalleries[$row['variant_id']]) {
				$this->em->getConnection()->update('eshop_catalog__product', [
					'gallery_id' => (int) $variantGalleries[$row['variant_id']],
				], [
					'id' => $row['id'],
				]);

				foreach ($this->langs->getLangs(false) as $lang) {
					$this->cacheService->productCache->remove('product/' . $row['id'] . '/' . $lang->getTag());
				}

				Debugger::log("Set gallery " . $variantGalleries[$row['variant_id']] . ' to ' . $row['id'], 'setEmptyVariantGallery');
			}
		}
	}
}
