<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model;

use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\Traits\TPublish;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\Model\AvailabilityService;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\Supplier;

/**
 * @method Supplier|null getReference($id)
 * @method Supplier[] getAll()
 * @method Supplier|null get($id)
 */
class Suppliers extends BaseEntityService
{
	use TPublish;

	protected $entityClass = Supplier::class;

	protected AvailabilityService $availabilityService;

	public function __construct(
		AvailabilityService $availabilityService
	)
	{
		$this->availabilityService = $availabilityService;
	}

	/**
	 * @return array<int, string>
	 */
	public function getOptionsForSelect(): array
	{
		$result = [];

		foreach ($this->getEr()->createQueryBuilder('s')->select('s.id, s.name')
			         ->orderBy('s.name')->getQuery()->getArrayResult() as $row) {
			$result[(int) $row['id']] = (string) $row['name'];
		}

		return $result;
	}

	public function setSale(int $id, int $state): bool
	{
		set_time_limit(120);

		if ($item = $this->get($id)) {
			$this->em->beginTransaction();
			try {
				$item->allowSale = $state;
				$this->em->persist($item);
				$this->em->flush();
				foreach ($this->em->getRepository(Product::class)->createQueryBuilder('p')
					         ->addSelect('ps, psa')
					         ->innerJoin('p.suppliers', 'ps', Join::WITH, 'ps.supplier = :supplier')
					         ->leftJoin('ps.availabilityAfterSoldOut', 'psa')
					         ->setParameters([
						         'supplier' => $item->getId(),
					         ])
					         ->getQuery()->getResult() as $row) {
					/** @var Product $row */
					$this->availabilityService->updateAvailabilityByQuantity($row);
					$this->em->persist($row);
				}

				$this->em->flush();
				$this->em->commit();
				$this->em->clear();
			} catch (\Exception $e) {
				if ($this->em->getConnection()->isTransactionActive()) {
					$this->em->rollback();
				}

				return false;
			}

			return true;
		}

		return false;
	}
}
