<?php declare(strict_types = 1);

namespace EshopCatalog\CronModule\Presenters;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\Strings;
use Core\Model\Http\CsvResponse;
use Core\Model\Sites;
use Currency\Model\Currencies;
use Doctrine\ORM\Query;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\CronModule\Model\CategoriesFeed;
use EshopCatalog\CronModule\Model\ProductsFeed;
use EshopCatalog\CronModule\Model\ProductsVariants;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\AvailabilityService;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Availability;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductTexts;
use EshopCatalog\Model\Entities\VatRate;
use EshopCatalog\Model\Event\ProductsFeedEvent;
use Exception;
use Nette\Security\Passwords;
use Tracy\Debugger;

class ProductsPresenter extends BasePresenter
{
	protected Products               $products;
	protected ProductsFacade         $productsFacade;
	protected EntityManagerDecorator $em;
	protected AvailabilityService    $availabilityService;
	protected ProductsFeed           $productsFeed;
	protected CategoriesFeed         $categoriesFeed;
	protected ProductsVariants       $productsVariants;
	protected Sites                  $sites;
	protected Currencies             $currencies;

	public function __construct(
		Products               $products,
		EntityManagerDecorator $em,
		AvailabilityService    $availabilityService,
		ProductsFacade         $productsFacade,
		ProductsFeed           $productsFeed,
		CategoriesFeed         $categoriesFeed,
		ProductsVariants       $productsVariants,
		Sites                  $sites,
		Currencies             $currencies
	)
	{
		parent::__construct();
		$this->productsFacade      = $productsFacade;
		$this->products            = $products;
		$this->em                  = $em;
		$this->availabilityService = $availabilityService;
		$this->productsFeed        = $productsFeed;
		$this->categoriesFeed      = $categoriesFeed;
		$this->productsVariants    = $productsVariants;
		$this->sites               = $sites;
		$this->currencies          = $currencies;
	}

	public function actionFindByTerm(string $term = '', array $excluded = []): void
	{
		$output = [];
		foreach ($this->products->getByTerm($term, Query::HYDRATE_ARRAY) as $product) {
			if (!in_array($product['id'], $excluded)) {
				$output[$product['id']] = [
					'id'    => $product['id'],
					'name'  => $product['productTexts'][$this->translator->getLocale()]['name'],
					'code1' => $product['code1'],
					'ean'   => $product['ean'],
				];
			}
		}

		$this->sendJson($output);
	}

	public function actionLoadAll(array $excluded = []): void
	{
		$output = [];

		$manufacturers = $this->em->getConnection()->fetchAllKeyValue("SELECT id, name FROM eshop_catalog__manufacturer");

		$select    = ['p.id', 'p.code1', 'p.code2', 'p.ean', 'pt.name', 'p.id_manufacturer'];
		$where     = ['p.is_deleted = 0'];
		$innerJoin = ["INNER JOIN eshop_catalog__product_texts pt ON pt.id = p.id AND pt.lang = '" . $this->translator->getLocale() . "'"];

		if ($excluded) {
			$where[] = 'p.id NOT IN (' . implode(',', $excluded) . ')';
		}

		if ($this->getParameter('excludePackages') === '1') {
			$where[] = 'p.package_id IS NULL';
		}

		$sql = "SELECT " . implode(', ', $select) . " FROM eshop_catalog__product p";
		$sql .= ' ' . implode(' ', $innerJoin);
		$sql .= ' WHERE ' . implode(' AND ', $where);

		foreach ($this->em->getConnection()->fetchAllAssociative($sql) as $row) {
			$output[$row['id']] = [
				'id'           => (string) $row['id'],
				'name'         => (string) $row['name'],
				'code1'        => (string) $row['code1'],
				'code2'        => (string) $row['code2'],
				'ean'          => (string) $row['ean'],
				'manufacturer' => $row['id_manufacturer'] ? $manufacturers[$row['id_manufacturer']] : null,
			];
		}

		$this->sendJson($output);
	}

	public function actionCheckAvailabilities(): void
	{
		set_time_limit(1200);
		$ids = [];

		foreach ($this->products->getEr()->createQueryBuilder('p')
			         ->select('p.id')
			         ->where('p.quantity <= 0 AND p.availability != :av AND p.isDeleted = 0')
			         ->orWhere('p.quantity > 0 AND p.availability != :av2 AND p.isDeleted = 0')
			         ->orWhere('p.availability IS NULL AND p.isDeleted = 0')
			         ->setParameters([
				         'av'  => Availability::SOLD_OUT,
				         'av2' => Availability::IN_STOCK,
			         ])
			         ->getQuery()->getArrayResult() as $row) {
			$ids[] = $row['id'];
		}

		foreach (array_chunk($ids, 50) as $chunk) {
			$this->em->beginTransaction();
			try {
				foreach ($this->products->getEr()->createQueryBuilder('p')
					         ->addSelect('ps, psa')
					         ->leftJoin('p.suppliers', 'ps')
					         ->leftJoin('ps.availabilityAfterSoldOut', 'psa')
					         ->andWhere('p.id IN (:ids)')
					         ->setParameters([
						         'ids' => $chunk,
					         ])
					         ->getQuery()->getResult() as $row) {
					/** @var Product $row */
					$this->availabilityService->updateAvailabilityByQuantity($row);
					$this->em->persist($row);
				}
				$this->em->flush();
				$this->em->commit();
				$this->em->clear();
			} catch (Exception $e) {
				if ($this->em->getConnection()->isTransactionActive()) {
					$this->em->rollback();
				}
			}
		}

		$this->availabilityService->updateProductsWithoutSupplier();

		$this->sendJson([]);
	}

	public function actionFeed(string $lang = 'cs', string $currency = 'CZK', int $onlyStock = 1): void
	{
		ProductsFacade::$forceLocale     = $lang;
		Currencies::$currentCodeOverride = $currency;
		Currencies::clearConvertedProducts($currency);

		$this->productsFeed->baseUrl = $this->getHttpRequest()->getUrl()->getBaseUrl();

		$generate                     = $this->getParameter('generate', null);
		$this->productsFeed->generate = $generate;

		if ($generate) {
			$this->productsFeed->getProducts(false, (bool) $onlyStock);
		}

		$data     = $this->productsFeed->loadAndParseProducts((bool) $onlyStock);
		$customer = $this->getParameter('customer');

		$event            = new ProductsFeedEvent($data, $customer);
		$event->lang      = $lang;
		$event->siteIdent = $this->sites->getCurrentSite()->getIdent();
		$this->eventDispatcher->dispatch($event, 'eshopCatalog.productsFeed');

		if ($event->outputFormat === 'xml') {
			$response = $this->getPresenter()->getHttpResponse();
			$response->setContentType('text/xml');
			echo $data['output'];
			exit;
		}

		if ($event->outputFormat === 'csv') {
			CsvResponse::sendResponse('data.csv', $data['output']);
		}

		$this->sendJson(array_values($data));
	}

	public function actionFeedAvailability(): void
	{
		ini_set('memory_limit', '2g');
		set_time_limit(360);

		$this->productsFeed->baseUrl = $this->getHttpRequest()->getUrl()->getBaseUrl();

		$data     = $this->productsFeed->getProductsAvailability();
		$customer = $this->getParameter('customer');

		$event = new ProductsFeedEvent($data, $customer);
		$this->eventDispatcher->dispatch($event, 'eshopCatalog.productsFeedAvailability');

		if ($event->outputFormat === 'xml') {
			$response = $this->getPresenter()->getHttpResponse();
			$response->setContentType('text/xml');
			echo $data['output'];
			exit;
		}

		if ($event->outputFormat === 'csv') {
			CsvResponse::sendResponse('data.csv', $data['output']);
		}

		$this->sendJson(array_values($data));
	}

	public function actionFeedCategories(): void
	{
		ini_set('memory_limit', '2g');
		set_time_limit(360);

		$this->categoriesFeed->baseUrl = $this->getHttpRequest()->getUrl()->getBaseUrl();

		$data     = $this->categoriesFeed->getCategories();
		$customer = $this->getParameter('customer');

		$event = new ProductsFeedEvent($data, $customer);
		$this->eventDispatcher->dispatch($event, 'eshopCatalog.categoriesFeed');

		if ($event->outputFormat === 'xml') {
			$response = $this->getPresenter()->getHttpResponse();
			$response->setContentType('text/xml');
			echo $data['output'];
			exit;
		}

		if ($event->outputFormat === 'csv') {
			CsvResponse::sendResponse('data.csv', $data['output']);
		}

		$this->sendJson(array_values($data));
	}

	public function actionReSetVariantCategories(): void
	{
		ini_set('memory_limit', '2g');
		set_time_limit(360);
		Debugger::$showBar = false;

		try {
			$this->productsVariants->reSetAllCategories();
		} catch (Exception $e) {
			$this->sendJson(['status' => 'error', 'message' => $e->getMessage()]);
		}

		$this->sendJson(['status' => 'ok']);
	}

	public function actionSetVariantMasters(): void
	{
		ini_set('memory_limit', '2g');
		set_time_limit(360);
		Debugger::$showBar = false;

		try {
			$this->productsVariants->setVariantMasters();
		} catch (Exception $e) {
			$this->sendJson(['status' => 'error', 'message' => $e->getMessage()]);
		}

		$this->sendJson(['status' => 'ok']);
	}

	public function actionSetAliases(): void
	{
		if ($this->getParameter('key') !== 'a5rg64f56f4a') {
			die();
		}

		foreach ($this->em->createQueryBuilder()->select('IDENTITY(pt.id) as id, pt.name, pt.lang')
			         ->from(ProductTexts::class, 'pt')
			         ->getQuery()->getArrayResult() as $row) {
			if (!$row['name']) {
				continue;
			}

			$sql = sprintf(
				"UPDATE eshop_catalog__product_texts SET alias = '%s' WHERE id = %s AND lang = '%s'",
				Strings::webalize($row['name']),
				(string) $row['id'],
				$row['lang'],
			);
			$this->em->getConnection()->executeQuery($sql);
		}

		die('ok');
	}

	public function actionClearDuplicatedProductPriceHistory(): void
	{
		if ($this->getParameter('key') !== 'a5rg64f56f4a') {
			die();
		}

		set_time_limit(900);
		ini_set('memory_limit', '4g');

		$data = [];

		foreach ($this->em->getConnection()->fetchAllAssociative(
			"SELECT id, product, price, created FROM eshop_catalog__product_price_history ORDER BY created ASC LIMIT 3000000"
		) as $row) {
			$exist = $data[$row['product']] ?? null;
			if ($exist && $exist == $row['price']) {
				$this->em->getConnection()->executeQuery(
					"DELETE FROM eshop_catalog__product_price_history WHERE id = " . $row['id']
				);
			}

			$data[$row['product']] = $row['price'];
		}

		foreach (array_chunk(array_keys($data), 200) as $chunk) {
			foreach ($this->em->getConnection()->fetchAllAssociative(
				"SELECT id, price FROM eshop_catalog__product WHERE id IN (" . implode(',', $chunk) . ")"
			) as $row) {
				if ($data[$row['id']] && $data[$row['id']] == $row['price']) {
					$this->em->getConnection()->executeQuery(
						"DELETE FROM eshop_catalog__product_price_history WHERE product = ? AND price = ?", [
							$row['id'],
							$row['price'],
						]
					);
				}
			}
		}

		die('ok');
	}

	public function actionProductsMargin(
		string $siteIdent,
		string $key
	): void
	{
		$password  = new Passwords();
		$keyVerify = $password->verify(Config::load('cronUrlHash') . $siteIdent, $key);

		if (!$keyVerify) {
			$this->error();
		}

		header('Content-Encoding: UTF-8');
		header("content-type:application/csv;charset=UTF-8");
		header("Content-Disposition:attachment;filename=\"products-margin-" . $siteIdent . ".csv\"");
		header('Content-Transfer-Encoding: binary');

		$fp = fopen('php://output', 'wb');
		if ($fp === false) {
			exit;
		}
		fputs($fp, "\xEF\xBB\xBF");

		$vatRates = [];
		foreach ($this->em->getRepository(VatRate::class)->createQueryBuilder('vr')
			         ->select('vr.id, vr.rate')
			         ->getQuery()->getArrayResult() as $row) {
			$vatRates[$row['id']] = $row['rate'];
		}

		foreach ($this->em->getRepository(Product::class)->createQueryBuilder('p')
			         ->select('p.id, p.purchasePrice, p.price, IDENTITY(p.vatRate) as vatRate')
			         ->innerJoin('p.sites', 's', Join::WITH, 's.site = :siteIdent')
			         ->andWhere('p.isPublished = 1')
			         ->setParameters([
				         'siteIdent' => $siteIdent,
			         ])
			         ->getQuery()->getArrayResult() as $row) {
			if (!$row['purchasePrice'] || !$row['price'] || !$row['vatRate']) {
				continue;
			}

			$purchase = ($row['purchasePrice'] * ((float) ((100 + $vatRates[$row['vatRate']]) / 100)));

			if ($purchase === 0.0) {
				continue;
			}

			fputcsv($fp, [
				$row['id'],
				number_format((round((float) $row['price'] / $purchase * 100, 2) - 100), 2, ',', ''),
			], ';');
		}

		fclose($fp);

		exit;
	}

	public function actionProductsMarginValue(
		string $siteIdent,
		string $key
	): void
	{
		$password  = new Passwords();
		$keyVerify = $password->verify(Config::load('cronUrlHash') . $siteIdent, $key);

		if (!$keyVerify) {
			$this->error();
		}

		header('Content-Encoding: UTF-8');
		header("content-type:application/csv;charset=UTF-8");
		header("Content-Disposition:attachment;filename=\"products-margin-value-" . $siteIdent . ".csv\"");
		header('Content-Transfer-Encoding: binary');

		$fp = fopen('php://output', 'wb');
		if ($fp === false) {
			exit;
		}

		fputs($fp, "\xEF\xBB\xBF");

		fputcsv($fp, ['id', 'cost_of_goods_sold'], ';');
		foreach ($this->em->getRepository(Product::class)->createQueryBuilder('p')
			         ->select('p.id, p.purchasePrice')
			         ->innerJoin('p.sites', 's', Join::WITH, 's.site = :siteIdent')
			         ->andWhere('p.isPublished = 1')
			         ->andWhere('p.purchasePrice IS NOT NULL')
			         ->setParameters([
				         'siteIdent' => $siteIdent,
			         ])
			         ->getQuery()->getArrayResult() as $row) {
			if (!$row['purchasePrice'] || $row['purchasePrice'] === '0.00') {
				continue;
			}

			fputcsv($fp, [
				$row['id'],
				number_format((float) $row['purchasePrice'], 2, '.', '') . ' ' . $this->currencies->getDefaultCode(),
			], ';');
		}

		fclose($fp);

		exit;
	}
}
