<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model;

use Core\Model\Entities\EntityManagerDecorator;
use EshopCatalog\FrontModule\Model\Dao;
use EshopCatalog\Model\Entities\PackageItem;
use Nette\Caching\Cache;

class Packages
{
	public const cacheNamespace = 'eshopCatalogPackages';

	protected EntityManagerDecorator $em;
	protected CacheService           $cacheService;

	/** @var Dao\Package[] */
	protected array $cPackages = [];

	/** @var Dao\PackageItem[][] */
	protected array $cPackageItems = [];

	public function __construct(
		EntityManagerDecorator $em,
		CacheService           $cacheService
	)
	{
		$this->em           = $em;
		$this->cacheService = $cacheService;
	}

	/**
	 * @return Dao\Package[]
	 */
	public function getPackages(array $ids): array
	{
		$whereIds = [];
		/** @var Dao\Package[][] $result */
		$result = [];

		foreach ($ids as $id) {
			if (isset($this->cPackages[$id])) {
				$result[$id] = $this->cPackages[$id];
			} else {
				$keys[] = 'package/' . $id;
			}
		}

		if (!empty($keys)) {
			foreach ($this->cacheService->packagesCache->bulkLoad($keys) as $key => $package) {
				$id = explode('/', $key)[1];

				if ($package !== null) {
					$this->cPackages[$id] = $package;
					$result[$id]          = $package;
				} else {
					$whereIds[] = $id;
				}
			}
		}

		if (!empty($whereIds)) {
			$qb = $this->em->getRepository(PackageItem::class)->createQueryBuilder('pi')
				->select('pi.id, IDENTITY(pi.package) as package, IDENTITY(pi.product) as product, pi.quantity')
				->where('pi.package IN (:ids)')
				->setParameters([
					'ids' => $whereIds,
				]);

			foreach ($qb->getQuery()->getArrayResult() as $row) {
				$this->cPackageItems[$row['package']][$row['product']] = $this->fillDaoItem($row);
			}

			foreach ($whereIds as $id) {
				$package        = $this->fillDao(['id' => $id]);
				$package->items = $this->cPackageItems[$id] ?? [];

				$this->cPackages[$id] = $package;
				$result[$id]          = $package;

				$this->cacheService->videosCache->save('package/' . $id, $this->cPackages[$id], [
					Cache::EXPIRATION => '14 days',
				]);
			}
		}

		return $result;
	}

	public function setMaximumQuantity(Dao\Product $product): void
	{
		if (!$product->packageId || $product->unlimitedQuantity) {
			return;
		}

		$quantities = [];
		foreach ($product->package->items as $item) {
			$itemProduct = $item->product;

			if ($itemProduct->unlimitedQuantity) {
				$quantities[$itemProduct->id] = 999;
			} else {
				$quantities[$itemProduct->id] = (int) round($itemProduct->quantity / $item->quantity);
			}
		}

		$product->quantity = min($quantities) ?: 0;
		$product->setMaximumAmount($product->quantity);
	}

	protected function fillDao(array $data): Dao\Package
	{
		return new Dao\Package((int) $data['id']);
	}

	protected function fillDaoItem(array $data): Dao\PackageItem
	{
		$item = new Dao\PackageItem(
			(int) $data['id'],
			(int) $data['product'],
		);

		$item->quantity = (int) $data['quantity'];

		return $item;
	}

}
