<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Core\Model\Entities\TId;
use Core\Model\Entities\TTranslateListener;
use Core\Model\Helpers\Traits\TExtraField;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Nette\Utils\ArrayHash;
use Nette\Utils\DateTime;

/**
 * @Gedmo\Tree(type="nested")
 * @ORM\Table(name="eshop_catalog__category", indexes={
 *     @ORM\Index(name="catalog_search", columns={"id", "is_published"}),
 * })
 * @ORM\Entity(repositoryClass="Core\Model\Entities\Repository\NestedTreeRepository")
 * @ORM\EntityListeners({"Core\Model\Entities\TranslateListener", "CategoryListener"})
 */
class Category
{
	use TId;
	use TTranslateListener;
	use TExtraField;

	public const EXTRA_FIELD_SECTION = 'eshopCatalogCategory';

	/**
	 * @ORM\Column(name="image", type="string", nullable=true, length=255)
	 */
	public ?string $image = null;

	/**
	 * @ORM\Column(type="smallint", options={"default": 0})
	 */
	public int $isPublished = 0;

	/**
	 * @var Collection<Category>
	 * @ORM\OneToMany(targetEntity="Category", mappedBy="parent")
	 * @ORM\OrderBy({"lft" = "ASC"})
	 */
	public Collection $children;

	/**
	 * @Gedmo\TreeParent
	 * @ORM\ManyToOne(targetEntity="Category", inversedBy="childrens", cascade={"persist"})
	 * @ORM\JoinColumn(name="parent_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public ?Category $parent = null;

	/**
	 * @Gedmo\TreeLeft
	 * @ORM\Column(type="integer")
	 */
	private int $lft;

	/**
	 * @Gedmo\TreeLevel
	 * @ORM\Column(type="integer")
	 */
	protected int $lvl;

	/**
	 * @Gedmo\TreeRight
	 * @ORM\Column(name="rgt", type="integer")
	 */
	private int $gt;

	/**
	 * @Gedmo\TreeRoot
	 * @ORM\ManyToOne(targetEntity="Category")
	 * @ORM\JoinColumn(name="tree_root", referencedColumnName="id", onDelete="CASCADE")
	 */
	private ?Category $root = null;

	/**
	 * @var Collection<CategoryTexts>
	 * @ORM\OneToMany(targetEntity="CategoryTexts", mappedBy="id", indexBy="lang", cascade={"persist", "remove"})
	 */
	protected Collection $categoryTexts;

	/**
	 * @var Collection<int, CategoryProduct>
	 * @ORM\OneToMany(targetEntity="CategoryProduct", mappedBy="category", indexBy="category", cascade={"all"})
	 */
	protected Collection $categoryProducts;

	/**
	 * @var Collection<CategoryFilter>
	 * @ORM\OneToMany(targetEntity="CategoryFilter", mappedBy="category", indexBy="feature_id")
	 * @ORM\OrderBy({"position" = "ASC"})
	 */
	public Collection $filters;

	/**
	 * @var int|bool
	 * @ORM\Column(type="smallint", options={"default": 1})
	 */
	public $filtersFromParent;

	/**
	 * @ORM\Column(type="smallint", options={"default": 0}, nullable=false)
	 */
	public int $ajaxFilterLoad = 0;

	/**
	 * @var DateTime
	 * @Gedmo\Timestampable(on="create")
	 * @ORM\Column(type="datetime", options={"default":"CURRENT_TIMESTAMP"})
	 */
	private $created;

	/**
	 * @var DateTime|null
	 * @Gedmo\Timestampable(on="update")
	 * @ORM\Column(type="datetime", nullable=true)
	 */
	private $modified;

	/**
	 * Vlastní hodnoty třeba z rozšířeních
	 * @var ArrayHash|array|null
	 * @ORM\Column(type="array", nullable=true)
	 */
	protected $attrs;

	/**
	 * @ORM\Column(type="string", nullable=true, length=1)
	 */
	public ?string $rod = null;

	/**
	 * @var int|bool
	 * @ORM\Column(type="smallint", length=1, options={"default": 1})
	 */
	public $canProductsAddToCart;

	/**
	 * @var Collection<CategoryRelated>
	 * @ORM\OneToMany(targetEntity="CategoryRelated", mappedBy="category")
	 */
	protected Collection $related;

	/**
	 * @ORM\Column(type="smallint", options={"default": 0})
	 */
	public int $disablePickupPoints = 0;

	/**
	 * @var Collection<FeatureCategory>
	 * @ORM\OneToMany(targetEntity="FeatureCategory", mappedBy="category", indexBy="id_feature_value", cascade={"all"})
	 */
	public Collection $features;

	/**
	 * @var Collection<DynamicFeatureCategory>
	 * @ORM\OneToMany(targetEntity="DynamicFeatureCategory", mappedBy="category", indexBy="id", cascade={"all"})
	 */
	public Collection $dynamicFeatures;

	/**
	 * @ORM\Column(type="string", nullable=true, length=255)
	 */
	public ?string $customerGroupRestriction = null;

	public function __construct()
	{
		$this->categoryTexts        = new ArrayCollection;
		$this->categoryProducts     = new ArrayCollection;
		$this->children             = new ArrayCollection;
		$this->filters              = new ArrayCollection;
		$this->isPublished          = 0;
		$this->created              = new DateTime;
		$this->filtersFromParent    = 1;
		$this->attrs                = [];
		$this->canProductsAddToCart = 1;
		$this->extraFields          = new ArrayCollection;
		$this->related              = new ArrayCollection;
		$this->features             = new ArrayCollection();
		$this->dynamicFeatures      = new ArrayCollection();
	}

	public function getLvl(): int { return $this->lvl; }

	public function getRoot(): ?Category { return $this->root; }

	public function addChildren(Category $child): void
	{
		$this->children->add($child);
	}

	public function addCategoryText(string $lang): void
	{
		$this->categoryTexts[$lang] = new CategoryTexts($this, $lang);
	}

	public function getCategoryText(?string $lang = null): ?CategoryTexts
	{
		return $this->categoryTexts[$lang ?? $this->locale];
	}

	/**
	 * @return Collection<string, CategoryTexts>
	 */
	public function getCategoryTexts(): Collection { return $this->categoryTexts; }

	/**
	 * @param CategoryTexts[] $categoryTexts
	 */
	public function setCategoryTexts(array $categoryTexts): self
	{
		$this->categoryTexts = new ArrayCollection($categoryTexts);

		return $this;
	}

	public function setCategoryText(CategoryTexts $categoryTexts): void
	{
		$this->categoryTexts->set($categoryTexts->getLang(), $categoryTexts);
	}

	public function getParent(): ?Category
	{
		return $this->parent;
	}

	public function setParent(?Category $parent): void
	{
		$this->parent = $parent;
	}

	public function addProduct(CategoryProduct $categoryProduct): void
	{
		$this->categoryProducts->add($categoryProduct);
	}

	/**
	 * @return DateTime
	 */
	public function getCreated() { return $this->created; }

	/**
	 * @return DateTime|null
	 */
	public function getModified() { return $this->modified ?: $this->getCreated(); }

	public function getAttrs(): array { return $this->attrs ? (array) $this->attrs : []; }

	/**
	 * @return mixed|null
	 */
	public function getAttr(string $key) { return $this->attrs[$key] ?? null; }

	/**
	 * @param mixed $value
	 */
	public function setAttr(string $key, $value): self
	{
		$this->attrs[$key] = $value;

		return $this;
	}

	public function removeAttr(string $key): self
	{
		unset($this->attrs[$key]);

		return $this;
	}

	public function getLft(): int { return $this->lft; }

	public function getRgt(): int { return $this->gt; }

	public function getAllowedCustomerGroup(): array
	{
		$groups = [];
		foreach (explode(',', (string) $this->customerGroupRestriction) as $group) {
			$groups[] = (int) $group;
		}

		return $groups;
	}

	public function setAllowedCustomerGroup(array $groups): void
	{
		if (empty($groups)) {
			$this->customerGroupRestriction = null;

			return;
		}

		$this->customerGroupRestriction = implode(',', $groups);
	}
}
