<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Doctrine;
use Doctrine\DBAL\Exception;
use Doctrine\ORM\Event\LifecycleEventArgs;
use Doctrine\ORM\Mapping as ORM;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Helpers\VariantsHelper;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Nette\SmartObject;

class ProductInSiteListener implements EventSubscriberInterface
{
	use SmartObject;

	protected static array   $created        = [];
	protected static array   $variantsInSite = [];
	protected CacheService   $cacheService;
	protected VariantsHelper $variantsHelper;

	public function __construct(
		CacheService   $cacheService,
		VariantsHelper $variantsHelper
	)
	{
		$this->cacheService   = $cacheService;
		$this->variantsHelper = $variantsHelper;
	}

	public static function getSubscribedEvents(): array
	{
		return [];
	}

	/**
	 * @ORM\PostUpdate
	 */
	public function postUpdate(ProductInSite $productInSite, LifecycleEventArgs $event): void
	{
		$em  = $event->getEntityManager();
		$key = $productInSite->getProduct()->getId() . '-' . $productInSite->getSite()->getIdent();
		if (isset(self::$created[$key])) {
			return;
		}
		self::$created[$key] = true;

		$variantsIds = $this->variantsHelper->getVariantsOfProduct($productInSite->getProduct()->getId());

		if (!empty($variantsIds)) {
			$variantsIds = array_flip($variantsIds);
			foreach ($em->getRepository(ProductInSite::class)->createQueryBuilder('pis')
				         ->select('IDENTITY(pis.product) as product, IDENTITY(pis.category) as category, pis.isActive')
				         ->where('pis.product IN (' . implode(',', array_keys($variantsIds)) . ')')
				         ->andWhere('pis.site = :site')
				         ->setParameter('site', $productInSite->getSite()->getIdent())
				         ->getQuery()->getArrayResult() as $row) {
				unset($variantsIds[$row['product']]);

				$update = [];

				if ($row['isActive'] != $productInSite->isActive()) {
					$update['is_active'] = $productInSite->isActive();
				}

				if ($row['category'] != $productInSite->category->getId()) {
					$update['category_id'] = $productInSite->category->getId();
				}

				if (!empty($update)) {
					$em->getConnection()->update('eshop_catalog__product_in_site', $update, [
						'product_id' => $row['product'],
						'site'       => $productInSite->getSite()->getIdent(),
					]);
				}
			}

			foreach (array_keys($variantsIds) as $prodId) {
				$em->getConnection()->insert('eshop_catalog__product_in_site', [
					'product_id'  => $prodId,
					'site'        => $productInSite->getSite()->getIdent(),
					'is_active'   => $productInSite->isActive() ? 1 : 0,
					'category_id' => $productInSite->category ? $productInSite->category->getId() : null,
				]);
			}
		}
	}

	/**
	 * @ORM\PostRemove
	 */
	public function postRemove(ProductInSite $productInSite, LifecycleEventArgs $event): void
	{
		$em  = $event->getEntityManager();
		$key = $productInSite->getProduct()->getId() . '-' . $productInSite->getSite()->getIdent();
		if (isset(self::$created[$key])) {
			return;
		}
		self::$created[$key] = true;

		foreach ($this->variantsHelper->getVariantsOfProduct($productInSite->getProduct()->getId()) as $prodId) {
			$em->getConnection()->delete('eshop_catalog__product_in_site', [
				'product_id' => $prodId,
				'site'       => $productInSite->getSite()->getIdent(),
			]);
		}
	}

	/**
	 * @ORM\PreFlush
	 *
	 * @param ProductInSite $productInSite
	 *
	 * @throws Exception
	 */
	public function onPreFlush($productInSite, Doctrine\ORM\Event\PreFlushEventArgs $args): void
	{
		$em = $args->getEntityManager();
		foreach ($em->getUnitOfWork()->getScheduledEntityInsertions() as $entity) {
			if ($entity instanceof ProductInSite === false || $entity->getProduct()->getId() === null || ($entity->getProduct()->isVariant() && $entity->getProduct()->isVariant()->isDefault !== 1)) {
				continue;
			}

			$key = $entity->getProduct()->getId() . '-' . $entity->getSite()->getIdent();
			if (isset(self::$created[$key])) {
				continue;
			}
			self::$created[$key] = true;

			$variantIds = $this->variantsHelper->getVariantsOfProduct($entity->getProduct()->getId());
			if ($variantIds) {
				$existInSite = [];
				$key         = implode('-', $variantIds);
				if (self::$variantsInSite[$key] === null) {
					self::$variantsInSite[$key] = $em->getConnection()->fetchAllAssociative("SELECT product_id, site FROM eshop_catalog__product_in_site WHERE product_id IN (" . implode(',', $variantIds) . ")") ?: [];
				}

				foreach (self::$variantsInSite[$key] ?? [] as $row) {
					$existInSite[$row['product_id']][$row['site']] = true;
				}

				foreach ($variantIds as $prodId) {
					if (isset($existInSite[$prodId][$entity->getSite()->getIdent()])) {
						$em->getConnection()->update('eshop_catalog__product_in_site', [
							'is_active'   => $entity->isActive() ? 1 : 0,
							'category_id' => $entity->category ? $entity->category->getId() : null,
						], [
							'product_id' => $prodId,
							'site'       => $entity->getSite()->getIdent(),
						]);
					} else {
						$em->getConnection()->insert('eshop_catalog__product_in_site', [
							'product_id'  => $prodId,
							'site'        => $entity->getSite()->getIdent(),
							'is_active'   => $entity->isActive() ? 1 : 0,
							'category_id' => $entity->category ? $entity->category->getId() : null,
						]);
					}
				}
			}
		}
	}

}
