<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Core\Model\Entities\TSeo;
use Core\Model\Helpers\Traits\TExtraField;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Nette\Utils\Strings;

/**
 * @ORM\Table(name="eshop_catalog__product_texts", indexes={
 *     @ORM\Index(name="lang", columns={"lang"})
 * })
 * @ORM\Entity
 * @ORM\EntityListeners({"ProductTextListener"})
 */
class ProductTexts
{
	use TSeo;
	use TExtraField;

	public const EXTRA_FIELD_SECTION = 'eshopProductText';

	/**
	 * @ORM\Id
	 * @ORM\ManyToOne(targetEntity="Product", inversedBy="ProductTexts", cascade={"persist", "remove"})
	 * @ORM\JoinColumn(name="id", referencedColumnName="id", onDelete="CASCADE")
	 */
	protected ?Product $id = null;

	/**
	 * @ORM\Id
	 * @ORM\Column(type="string", length=2)
	 */
	private string $lang;

	/**
	 * @ORM\Column(name="is_published", type="smallint", length=1, options={"default": 0})
	 */
	public int $isPublished = 0;

	/**
	 * @ORM\Column(type="string", length=255)
	 */
	public string $name = '';

	/**
	 * @ORM\Column(name="name2", type="string", length=255, nullable=true)
	 */
	public ?string $name2 = null;

	/**
	 * @ORM\Column(type="string", length=255)
	 */
	public string $alias = '';

	/**
	 * @ORM\Column(type="text", nullable=true)
	 */
	public ?string $shortDescription = null;

	/**
	 * @ORM\Column(type="text", nullable=true)
	 */
	public ?string $description = null;

	/**
	 * @ORM\Column(name="condition_description", type="text", nullable=true)
	 */
	public ?string $conditionDescription = null;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public ?string $preorderText = null;

	public function __construct(
		Product $id,
		string  $lang
	)
	{
		$this->id          = $id;
		$this->lang        = $lang;
		$this->seo         = [];
		$this->extraFields = new ArrayCollection;
	}

	public function __clone()
	{
		$this->id          = null;
		$this->extraFields = new ArrayCollection;
	}

	public function setProduct(Product $product): self
	{
		$this->id = $product;

		return $this;
	}

	public function setName(string $name): ProductTexts
	{
		$this->name = $name;
		if (!$this->alias) {
			$this->alias = Strings::webalize($name);
		}

		return $this;
	}

	public function setName2(?string $name): self
	{
		$this->name2 = $name;

		return $this;
	}

	public function setAlias(string $alias): self
	{
		$this->alias = $alias ? Strings::webalize($alias) : Strings::webalize($this->name);

		return $this;
	}

	public function getLang(): string
	{
		return $this->lang;
	}

	public function getProduct(): ?Product { return $this->id; }

	public function setDescription(string $text): self
	{
		$this->description = $text;

		return $this;
	}

	public function setShortDescription(string $text): self
	{
		$this->shortDescription = $text;

		return $this;
	}

}
