<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Helpers;

use Core\Model\Entities\EntityManagerDecorator;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\Model\Entities\ProductVariant;
use Nette\Caching\Cache;

class VariantsHelper
{
	protected EntityManagerDecorator $em;
	protected CacheService           $cacheService;

	protected array $cLoaded   = [];
	protected bool  $allLoaded = false;

	public function __construct(
		EntityManagerDecorator $em,
		CacheService           $cacheService
	)
	{
		$this->em           = $em;
		$this->cacheService = $cacheService;
	}

	public function getAllVariants(): array
	{
		if (!$this->allLoaded) {
			$this->allLoaded = true;
			$map             = [];

			foreach ($this->em->createQueryBuilder()->from(ProductVariant::class, 'pv')
				         ->select('IDENTITY(pv.product) as id, pv.isDefault, pv.variantId')
				         ->orderBy('pv.isDefault', 'DESC')
				         ->getQuery()->getScalarResult() as $row) {
				if ($row['isDefault'] === 1) {
					$this->cLoaded[$row['id']] = [];
					$map[$row['variantId']]    = $row['id'];
				} else {
					$this->cLoaded[$map[$row['variantId']]][] = $row['id'];
				}
			}
		}

		return $this->cLoaded;
	}

	/**
	 * @param int[] $ids
	 *
	 * @return int[][]
	 */
	public function getProductVariants(array $ids): array
	{
		$loaded = [];
		$where  = [];

		foreach ($ids as $id) {
			if (isset($this->cLoaded[$id])) {
				$loaded[$id] = $this->cLoaded[$id];
			} else {
				$where[]            = $id;
				$this->cLoaded[$id] = [];
			}
		}

		if ($where) {
			$variantIds = [];
			foreach ($this->em->createQueryBuilder()
				         ->select('IDENTITY(pv.product) as prod, pv.variantId')
				         ->from(ProductVariant::class, 'pv')
				         ->where('pv.product IN (:prod)')
				         ->setParameter('prod', $where)
				         ->getQuery()->getArrayResult() as $row) {
				$variantIds[$row['variantId']] = $row['prod'];
			}

			if ($variantIds) {
				foreach ($this->em->createQueryBuilder()->from(ProductVariant::class, 'pv')
					         ->select('IDENTITY(pv.product) as id, pv.variantId')
					         ->where('pv.variantId IN (:varId)')
					         ->andWhere('pv.isDefault = 0')
					         ->setParameter('varId', array_keys($variantIds))
					         ->getQuery()->getScalarResult() as $row) {
					$key                   = $variantIds[$row['variantId']];
					$this->cLoaded[$key][] = $row['id'];
					$loaded[$key][]        = $row['id'];
				}
			}
		}

		return $loaded;
	}

	public function getVariantsOfProduct(int $productId): array
	{
		return $this->getProductVariants([$productId])[$productId] ?: [];
	}

	public function getBasicByProduct(): array
	{
		return $this->cacheService->productCache->load('variantBasicByProduct', function(&$dep) {
			$dep = [
				Cache::Tags       => ['variants', Products::CACHE_NAMESPACE],
				Cache::EXPIRATION => '1 week',
			];

			$arr = [];
			foreach ($this->em->createQueryBuilder()
				         ->select('IDENTITY(pv.product) as prod, pv.variantId, pv.isDefault')
				         ->from(ProductVariant::class, 'pv')
				         ->getQuery()->getScalarResult() as $row) {
				$arr[$row['prod']] = $row;
			}

			return $arr;
		});
	}

}
