<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Products;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Model\ProductVariantCombinations;
use EshopCatalog\AdminModule\Model\ProductVariants;
use EshopCatalog\Model\Entities\ProductVariant;
use EshopCatalog\Model\Entities\ProductVariantCombination;
use EshopCatalog\Model\Entities\Variant;
use EshopCatalog\Model\Entities\VariantValue;
use Nette\Application\UI\Form;
use Nette\Http\IResponse;
use Nette\Http\Request;
use Nette\Utils\ArrayHash;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductTexts;
use EshopCatalog\AdminModule\Model\Products;
use Nette\Utils\Json;

/**
 * Class ProductsVariantForm
 * @package EshopCatalog\AdminModule\Components\Products
 */
class ProductsVariantForm extends BaseControl
{
	/** @var array */
	public $products;

	/** @var string */
	private $ids;

	/** @var Products */
	protected $productServices;

	/** @var ProductVariantCombinations */
	protected $productVariantCombinationsService;

	/** @var ProductVariants */
	protected $variantsService;

	/** @var array */
	protected $variants;

	public function __construct(Products $products, ProductVariantCombinations $productVariantCombinations, ProductVariants $productVariants)
	{
		$this->productServices                   = $products;
		$this->productVariantCombinationsService = $productVariantCombinations;
		$this->variantsService                   = $productVariants;
	}

	public function render()
	{
		$this->template->variantsJson = Json::encode($this->getVariants());
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ========================  HANDLE
	 */

	/*******************************************************************************************************************
	 * ========================  COMPONENTS
	 */

	protected function createComponentForm()
	{
		$form = $this->createForm();

		$form->addHidden('produtcIds', $this->ids);

		$form->addSelect('variant', 'eshopCatalog.productVariant.variant', $this->getVariants()['variants']);
		$form->addSelect('value', 'eshopCatalog.productVariant.value', $this->getVariants()['values'][key($this->getVariants()['variants'])]);

		$form->addSaveCancelControl('saveControl');
		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			$variantId = $values->variant;
			$valueId   = $this->getPresenter()->getHttpRequest()->getPost('value');

			$tmp = $this->products;
			foreach ($this->products as $product) {
				/** @var ProductVariantCombination[] $exists */
				$exists = $this->em->getRepository(ProductVariantCombination::class)->createQueryBuilder('pac')
					->join('pac.productVariant', 'pa')
					->where('pa.product = :productId')->setParameter('productId', $product->getId())
					->andWhere('pac.variant = :variantId')->setParameter('variantId', $variantId)
					->getQuery()->getResult();

				foreach ($exists as $exist) {
					unset($tmp[$product->getId()]);
					$exist->variant = $this->em->getReference(Variant::class, $variantId);
					$exist->value   = $this->em->getReference(VariantValue::class, $valueId);
					$this->em->persist($exist);
				}
			}

			foreach ($tmp as $product) {
				$productVariant            = new ProductVariant($product);
				$productVariantCombination = new ProductVariantCombination(
					$productVariant,
					$this->em->getReference(VariantValue::class, $valueId),
					$this->em->getReference(Variant::class, $variantId)
				);

				$this->em->persist($productVariant)->persist($productVariantCombination);
			}

			$this->em->flush();
			$form->addCustomData('productIds', $values->produtcIds);
			$this->getPresenter()->flashMessageSuccess('eshopCatalog.productForm.productSaved');
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}
	}

	/*******************************************************************************************************************
	 * ========================  GET / SET
	 */

	public function setProducts($ids)
	{
		$this->ids  = $ids;
		$productIds = explode('-', $ids);
		foreach ($productIds as $id) {
			$this->products[$id] = $this->em->getRepository(Product::class)->find($id);
		}

		if (!$this->products)
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}

	protected function getVariants()
	{
		if (!$this->variants) {
			$variants = [];
			foreach ($this->variantsService->em->getRepository(Variant::class)->createQueryBuilder('v')->select('v.id, vt.name')
				         ->join('v.texts', 'vt', 'WITH', 'vt.lang = :lang')->where('vt.lang = :lang')->setParameter('lang', $this->translator->getLocale())
				         ->where('v.isPublished = 1')
				         ->getQuery()->getResult() as $v)
				$variants[$v['id']] = $v['name'];

			$values = [];
			foreach ($this->variantsService->em->getRepository(VariantValue::class)->createQueryBuilder('vv')
				         ->select('vv.id, vvt.name, IDENTITY(vv.variant) as variantId')
				         ->join('vv.texts', 'vvt', 'WITH', 'vvt.lang = :lang')->setParameter('lang', $this->translator->getLocale())
				         ->where('vv.isPublished = 1')
				         ->getQuery()->getResult() as $v)
				$values[$v['variantId']][$v['id']] = $v['name'];

			$this->variants = ['variants' => $variants, 'values' => $values];
		}

		return $this->variants;
	}
}
