<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Components;

use Core\Model\UI\BaseControl;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\Dao\FilterGroup;
use EshopCatalog\FrontModule\Model\Event\FilterEvent;
use EshopCatalog\FrontModule\Model\FeatureProducts;
use EshopCatalog\FrontModule\Model\FilterService;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\FrontModule\Model\ProductVariants;
use EshopCatalog\Model\Entities\FeatureProduct;
use EshopCatalog\Model\Entities\Product;
use Slides\FrontModule\Model\Dao\Group;

/**
 * TODO filtr správně nedělat přes handle ale pomocí jednotného parametru
 *
 * Class ProductsFilter
 * @package EshopCatalog\FrontModule\Components
 */
class ProductsFilter extends BaseControl
{
	/** @var int[] */
	public $productIds;

	/**
	 * ID produktů po vyfiltrování
	 *
	 * @var int[]
	 */
	protected $productIdsAfterFilter = [];

	/**
	 * Vyfiltrované produkty ale bez filtru cena
	 *
	 * @var int[]
	 */
	public $productIdsNoPriceFilter;

	/** @var int */
	protected $categoryId;

	/** @var string */
	public $searchQuery;

	/** @var FeatureProducts */
	protected $featureProducts;

	/** @var Products */
	protected $productsService;

	/** @var FilterService */
	protected $filterService;

	/** @var Categories */
	protected $categoriesService;

	/** @var ProductsFacade */
	protected $productsFacade;

	/** @var callable */
	public $onFilter = [];

	/** @var array */
	protected $cFeatures, $cMinMax, $cVariants, $cManu;

	/** @persistent */
	public $filter = [];

	/** @persistent */
	public $min;

	/** @persistent */
	public $max;

	/** @persistent */
	public $sort;

	public function __construct(FeatureProducts $featureProducts, Products $products, FilterService $filterService, Categories $categories,
	                            ProductsFacade $productsFacade)
	{
		$this->featureProducts   = $featureProducts;
		$this->productsService   = $products;
		$this->filterService     = $filterService;
		$this->categoriesService = $categories;
		$this->productsFacade    = $productsFacade;
	}

	protected function attached($presenter)
	{
		parent::attached($presenter);
		$signal = $this->getPresenter()->getSignal();

		if ($this->getPresenter()->isAjax() && !(is_array($signal) && in_array('formSubmit', $signal))) {
			$this->eventDispatcher->dispatch(ProductsFilter::class . '::onFilter', new FilterEvent($this->getFilterValues()));
			$this->getPresenter()->payload->url = $this->link('this');
			$this->redrawControl('filters');
		}
	}

	public function render()
	{
		$ac           = $this->getFilterValues();
		$features     = $this->getFeatures();
		$minMax       = $this->getMinMax();
		$manu         = $this->getManufacturers();
		$filterActive = false;

		$inActiveGroups = [];
		foreach ($features as $group) {
			$hasActive = false;
			foreach ($group->items as $itemK => $item) {
				if ($item->productsCountInit === 0) {
					unset($group->items[$itemK]);
					continue;
				}
				if (isset($ac['features'][$group->id]) && in_array($item->id, $ac['features'][$group->id])) {
					$item->isActive = true;
					$hasActive      = true;
					$filterActive   = true;
				}
			}

			if (!$hasActive)
				$inActiveGroups['feature'][] = $group->id;
		}

		$hasActive = false;
		foreach ($manu->items as $itemK => $item) {
			if ($item->productsCountInit === 0) {
				unset($manu->items[$itemK]);
				continue;
			}
			if (isset($ac['manu']) && in_array($item->id, $ac['manu'])) {
				$item->isActive = true;
				$hasActive      = true;
				$filterActive   = true;
			}
		}

		if (!$hasActive)
			$inActiveGroups['manu'] = $manu->id;

		$this->template->minMax         = $minMax;
		$this->template->currentMinMax  = ['min' => $this->min ?: $minMax['min'],
		                                   'max' => $this->max ?: $minMax['max']];
		$this->template->inActiveGroups = $inActiveGroups;
		$this->template->variants       = $variants;
		$this->template->features       = $features;
		$this->template->manu           = $manu;
		$this->template->sort           = [
			'recommended',
			'p.created DESC', 'p.created ASC',
			'p.price DESC', 'p.price ASC',
			'pt.name ASC', 'pt.name DESC',
		];

		$this->template->currentSort  = $this->sort;
		$this->template->filterActive = $filterActive || $this->sort || $this->min || $this->max;

		$presenter = $this->getPresenter();
		$args      = $presenter->getParameters();
		unset($args['activeNavigation']);
		unset($args['action']);
		unset($args['do']);
		$this->template->resetLink = $presenter->link(':' . $presenter->getName() . ':' . $presenter->getAction(), $args);

		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ============================== Handle
	 */

	public function handleSet($t, $k, $v)
	{
		$this->filter['f' . $t][$k][] = $v;
		if ($this->getPresenter()->isAjax())
			$this->onFilter($this->getFilterValues());
		else
			$this->redirect('this');
	}

	public function handlePrice($min, $max)
	{
		$this->min = $this->getMinMax()['min'] == $min ? null : $min;
		$this->max = $this->getMinMax()['max'] == $max ? null : $max;

		if ($this->getPresenter()->isAjax())
			$this->onFilter($this->getFilterValues());
		else
			$this->redirect('this');
	}

	public function handleSort($sort)
	{
		if ($sort == 'recommended')
			$sort = null;
		$this->sort = $sort;

		if ($this->getPresenter()->isAjax())
			$this->onFilter($this->getFilterValues());
		else
			$this->redirect('this');
	}

	public function handleFormSubmit()
	{
		$presenter   = $this->getPresenter();
		$httpRequest = $presenter->getHttpRequest();
		$this->resetFilter();

		$variants = $httpRequest->getPost('variant');
		if ($variants) {
			foreach ($variants as $g => $vals) {
				if (is_string($vals) && $vals === 'on')
					continue;
				$this->filter['fv'][$g] = implode('|', $vals);
			}
		}
		$features = $httpRequest->getPost('feature');
		if ($features) {
			foreach ($features as $g => $vals) {
				if (is_string($vals) && $vals === 'on')
					continue;
				$this->filter['ff'][$g] = implode('|', $vals);
			}
		}
		$manufacturers = $httpRequest->getPost('manu');
		if ($manufacturers) {
			foreach ($manufacturers as $g => $vals) {
				if (is_string($vals) && $vals === 'on')
					continue;
				$this->filter['fm'][$g] = implode('|', $vals);
			}
		}

		$priceRange = $httpRequest->getPost('priceRange');
		if ($priceRange) {
			if (isset($priceRange['min']) && $this->getMinMax()['min'] != $priceRange['min'])
				$this->min = $priceRange['min'];
			if (isset($priceRange['max']) && $this->getMinMax()['max'] != $priceRange['max'])
				$this->max = $priceRange['max'];
		}

		$sort = $httpRequest->getPost('sort');
		if ($sort)
			$this->sort = $sort;

		if ($presenter->isAjax()) {
			$this->eventDispatcher->dispatch(ProductsFilter::class . '::onFilter', new FilterEvent($this->getFilterValues()));
			$this->getPresenter()->payload->url = $this->link('this');
			$this->redrawControl('filters');
		} else {
			$this->redirect('this');
		}
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */


	/*******************************************************************************************************************
	 * ============================== Get / Set
	 */

	public function setCategoryId(int $id): self
	{
		$this->categoryId = $id;
		$this->productIds = $this->productsService->getProductsIdInCategory($id, 0, null, $this->sort);
		$this->loadFilters();

		return $this;
	}

	public function setSearchQuery(string $search): self
	{
		$this->searchQuery = $search;
		$this->productIds  = $this->productsService->getProductsIdBySearch($search);
		$this->loadFilters();

		return $this;
	}

	protected function loadFilters(): void
	{
		$filter = $this->getFilterValues();
		unset($filter['priceRange']);
		if ($this->categoryId)
			$this->productIdsNoPriceFilter = $this->productsService->getProductsIdInCategory($this->categoryId, null, null, null, $filter);
		else if ($this->searchQuery)
			$this->productIdsNoPriceFilter = $this->productsService->getProductsIdBySearch($this->searchQuery, null, null, null, $filter);;

		$this->getFeatures();
		$this->getMinMax();
		$this->getManufacturers();
	}

	public function getFeatures()
	{
		if ($this->cFeatures === null) {
			$ac = $this->getFilterValues();
			if ($this->categoryId) {
				$cat             = $this->categoriesService->get($this->categoryId);
				$this->cFeatures = $this->filterService->getCategoryFilters((int) $this->categoryId, $cat->filtersFromParent ? true : false);
			} else {
				$this->cFeatures = $this->filterService->getFeatures($this->productIds);
			}

			$counts  = $this->getProductsCountInFilter()['f'];
			$countsP = $this->getProductsCountInFilter()['p'];

			foreach ($this->cFeatures as $kf => $feature) {
				foreach ($feature->items as $ki => $item) {
					if (isset($counts[$ki])) {
						// Nastavení init hodnoty. Jestli se má vůbec zobrazit na frontu
						$this->cFeatures[$kf]->items[$ki]->productsCountInit = count($counts[$ki]);

						$count = 0;
						// Projdutí všech produktů a zjištění spolčných hodnot
						foreach ($counts[$ki] as $prodId) {
							$vals = $countsP[$prodId];

							// Kontrola jestli produkt obsahuje vlastnosti a výrobce odpovídá pokud je vyfiltrován a odpovídá ceně
							if ($ac['priceRange'] && ($vals['price'] < $ac['priceRange']['min'] || $vals['price'] > $ac['priceRange']['max']))
								continue;

							// Pokud produkt neobsahuje výrobce tak pokračovat
							if (!$vals['m'] || ($ac['manu'] && !in_array($vals['m'], $ac['manu'])))
								continue;

							// Pokud produkt vlastnost obsahuje tak odstraníme z $tmp a pokud je prázdný tak přičteme
							$tmp = $ac['features'];
							foreach ($ac['features'] as $fk => $fv) {
								if ($fk == $kf)
									$count++;

								foreach ($fv as $v2) {
									if (in_array($v2, $vals['f'])) {
										unset($tmp[$fk]);
										break;
									}
								}
							}

							if (empty($tmp)) {
								$this->productIdsAfterFilter[] = $prodId;
								$count++;
							}
						}

						$this->cFeatures[$kf]->items[$ki]->productsCount = $count;
					}
				}
			}
		}

		return $this->cFeatures;
	}

	public function getMinMax()
	{
		if (!$this->cMinMax && $this->productIdsNoPriceFilter)
			$this->cMinMax = $this->filterService->getMinMaxPrice($this->productIdsNoPriceFilter);

		return $this->cMinMax;
	}

	public function getManufacturers()
	{
		if (!$this->cManu) {
			// Načteme vlastnosti pro porovnání jestli hodnoty obsahují některé produkty
			$features = $this->getFeatures();
			$ac       = $this->getFilterValues();
			$items    = $this->filterService->getManufacturers($this->productIds);
			$counts   = $this->getProductsCountInFilter()['m'];
			$countsP  = $this->getProductsCountInFilter()['p'];

			foreach ($items as $k => $v) {
				if (isset($counts[$k]))
					$items[$k]->productsCountInit = count($counts[$k]);

				$count = 0;
				// Projdutí všech produktů výrobce
				foreach ($counts[$k] as $prodId) {
					$vals = $countsP[$prodId];

					// Kontrola jestli produkt odpovídá ceně
					if ($ac['priceRange'] && ($vals['price'] < $ac['priceRange']['min'] || $vals['price'] > $ac['priceRange']['max']))
						continue;

					// Pokud není aktivní filtr vlastnoti tak přidat a pokračovat
					if (!$ac['features']) {
						if ($ac['manu'] && in_array($vals['m'], $ac['manu']) || !$ac['manu']) {
							$this->productIdsAfterFilter[] = $prodId;
						}
						$count++;
						continue;
					}

					// Pokud produkt nemá vlastnosti tak ignorovat
					if (!$vals['f'])
						continue;

					// Pokud produkt vlastnost obsahuje tak odstraníme z $tmp a pokud je prázdný tak přičteme
					$tmp = $ac['features'];
					foreach ($ac['features'] as $fk => $fv) {
						foreach ($fv as $v2) {
							if (in_array($v2, $vals['f'])) {
								unset($tmp[$fk]);
								break;
							}
						}
					}

					if (empty($tmp)) {
						$count++;
						$this->productIdsAfterFilter[] = $prodId;
					}
				}
				$items[$k]->productsCount = $count;
			}

			$this->cManu = (new FilterGroup())
				->setType('manu')
				->setId(0)
				->setName($this->t('eshopCatalogFront.filter.manufacturer'))
				->setItems($items);
		}

		return $this->cManu;
	}

	/**
	 * Vrátí ID produktů které odpovídají filtru
	 *
	 * @return array
	 */
	public function getProductIdsAfterFilter(): array
	{
		return $this->productIdsAfterFilter ? array_intersect($this->productIds, $this->productIdsAfterFilter) : $this->productIds;
	}

	public function getFilterValues()
	{
		$return = [];

		if (isset($this->filter['fv']))
			foreach ($this->filter['fv'] as $group => $vals)
				$return['variants'][$group] = explode('|', $this->filter['fv'][$group]);
		if (isset($this->filter['ff']))
			foreach ($this->filter['ff'] as $group => $vals)
				$return['features'][$group] = explode('|', $this->filter['ff'][$group]);
		if (isset($this->filter['fm']))
			foreach ($this->filter['fm'] as $group => $vals)
				$return['manu'] = explode('|', $this->filter['fm'][$group]);
		if ($this->min)
			$return['priceRange']['min'] = $this->min;
		if ($this->max)
			$return['priceRange']['max'] = $this->max;

		if ($this->sort)
			$return['sort'] = $this->sort;

		return $return;
	}

	protected function resetFilter()
	{
		$this->filter = null;
		$this->min    = null;
		$this->max    = null;
		$this->sort   = null;
	}

	protected function getProductsCountInFilter()
	{
		$key = $this->categoryId ? 'c' . $this->categoryId : $this->searchQuery;

		return $this->filterService->getProductsData($this->productIds, $key);
	}
}
