<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Tags;

use Core\Components\Flashes\Flashes;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\Model\Entities\Tag;
use EshopCatalog\Model\Entities\TagText;
use Nette\Caching\Cache;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use EshopCatalog\AdminModule\Model\Tags;

class TagForm extends BaseControl
{
	/** @var Tag */
	public $tag;

	/** @var int @persistent */
	public $tagId;

	/** @var Tags */
	protected $tagServices;

	public function __construct(Tags $tags)
	{
		$this->tagServices = $tags;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addText('type', 'eshopCatalog.tagForm.type')
			->setMaxLength(255);
		$form->addText('name', 'eshopCatalog.tagForm.name')
			->setIsMultilanguage()
			->setMaxLength(255);
		$form->addFilesManager('image', 'default.image');
		$form->addColorInput('color', 'default.textColor');
		$form->addColorInput('bgColor', 'default.bgColor');

		$form->addSaveCancelControl('saveControl');

		$form->onValidate[] = [$this, 'formValidate'];
		$form->onSuccess[]  = [$this, 'formSuccess'];

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values): void
	{
		$nameFilled = false;
		foreach ($values->name as $v)
			if ($v)
				$nameFilled = true;

		if (!$nameFilled)
			$form->addError('eshopCatalog.tagForm.nameMissing');
		if ($form->hasErrors())
			$this->redrawControl('form');
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var TagText[] $tagTexts */
			$tagTexts = [];

			if ($this->tagId) {
				$tag      = $this->tagServices->get($this->tagId);
				$tagTexts = $tag->getTexts()->toArray();
			} else {
				$tag = new Tag();
			}

			$tag->type    = $values->type;
			$tag->image   = $values->image;
			$tag->color   = $values->color;
			$tag->bgColor = $values->bgColor;
			$this->em->persist($tag);

			foreach ($langValues as $l => $v) {
				if ($v['name'] === '')
					continue;

				if (!$tagTexts[$l])
					$tagTexts[$l] = new TagText($tag, $l);

				$tagTexts[$l]->name = $v['name'];
				$this->em->persist($tagTexts[$l]);
			}
			$tag->setTexts($tagTexts);

			$this->em->persist($tag);
			$this->em->flush();
			$form->addCustomData('tagId', $tag->getId());
			$form->addCustomData('tagType', $tag->type);
			$this->getPresenter()->flashMessageSuccess('eshopCatalog.tagForm.saved');

			$cache = new Cache($this->cacheStorage, \EshopCatalog\FrontModule\Model\Tags::CACHE_NAMESPACE);
			$cache->clean([Cache::TAGS => [\EshopCatalog\FrontModule\Model\Tags::CACHE_NAMESPACE]]);
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}

		return true;
	}

	/**
	 * @param int|string $type
	 *
	 * @throws \Nette\Application\BadRequestException
	 */
	public function setTag($type)
	{
		if (is_int($type))
			$this->tag = $this->tagServices->get($type);
		else
			$this->tag = $this->tagServices->getByType($type);

		if ($this->tag) {
			$this->tagId = $this->tag->getId();
			$form        = $this['form'];
			$defaults    = [
				'type'    => $this->tag->type,
				'image'   => $this->tag->image,
				'color'   => $this->tag->color,
				'bgColor' => $this->tag->bgColor,
			];

			foreach ($this->langsService->getLangs(false) as $lang) {
				foreach (['name'] as $name) {
					$defaults[$name][$lang->getTag()] = $this->tag->getText($lang->getTag())->$name;
				}
			}

			$form->setDefaults($defaults);
		} else
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}
}
