<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\VatRates;

use Core\Model\Exceptions\EntityHasRelations;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\Model\Entities\VatRate;
use EshopCatalog\AdminModule\Model\VatRates;

class VatRatesGrid extends BaseControl
{
	/** @var VatRates */
	protected $vatRateServices;

	/** @var IVatRateFormFactory */
	protected $vatRateFormFactory;

	public function __construct(VatRates $vatRate, IVatRateFormFactory $vatRateFormFactory)
	{
		$this->vatRateServices    = $vatRate;
		$this->vatRateFormFactory = $vatRateFormFactory;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleAdd()
	{
		$this->template->modalTitle = $this->t('eshopCatalog.title.newVatRate');
		$this->template->modal      = 'form';
		$this->redrawControl('modal');
	}

	public function handleEdit(int $id): void
	{
		$this['form']->id = $id;
		$this['form']->setVatRate($id);
		$this->template->modalTitle = $this->t('eshopCatalog.title.editVatRate',
			['title' => $this['form']->vatRate->name]);
		$this->template->modal      = 'form';
		$this->redrawControl('modal');
	}

	public function handleDelete(int $id): void
	{
		$presenter = $this->getPresenter();
		try {
			if ($this->vatRateServices->remove($id))
				$presenter->flashMessageSuccess('eshopCatalog.defaultGrid.removed');
			else
				$presenter->flashMessageDanger('eshopCatalog.defaultGrid.removeFailed');
		} catch (EntityHasRelations $e) {
			$presenter->flashMessageDanger($e->getMessage());
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setDefaultPerPage(50);

		$qb = $this->vatRateServices->getEr()->createQueryBuilder('vr');

		$grid->setDataSource($qb);

		// Columns
		$grid->addColumnText('name', 'eshopCatalog.defaultGrid.name');
		$grid->addColumnNumber('rate', 'eshopCatalog.defaultGrid.value');

		// Actions
		$grid->addAction('edit', '', 'edit!')->setIcon('edit')->setBsType('primary')
			->addClass('ajax');
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')
			->addClass('ajax')
			->setConfirm('default.reallyDelete');

		// Columns prototype

		return $grid;
	}

	protected function createComponentForm(): VatRateForm
	{
		$control = $this->vatRateFormFactory->create();
		if ($this->getParameter('id'))
			$control->setVatRate((int) $this->getParameter('id'));

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this->getPresenter()->redrawControl('flashes');
			$this['grid']->reload();
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->getPresenter()->payload->hideModal = true;
			$this->getPresenter()->redrawControl('flashes');
			$this['grid']->reload();
		};
		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}
}
