<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model\FormContainers;

use Core\Model\Countries;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\UI\Form\BaseContainer;
use EshopCatalog\AdminModule\Model\Helpers\PricesHelper;
use EshopCatalog\AdminModule\Model\ProductPrices;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\AdminModule\Model\VatRates;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductPrice;
use Nette\Localization\Translator;

class ProductPricesContainer
{
	protected Translator $translator;

	protected EntityManagerDecorator $em;

	protected Products $products;

	protected ProductPrices $productPrices;

	protected Countries $countries;

	protected VatRates $vatRates;

	protected PricesHelper $pricesHelper;

	public function __construct(Translator $translator, EntityManagerDecorator $em, Products $products,
	                            ProductPrices $productPrices, Countries $countries, VatRates $vatRates,
	                            PricesHelper $pricesHelper)
	{
		$this->translator    = $translator;
		$this->em            = $em;
		$this->products      = $products;
		$this->productPrices = $productPrices;
		$this->countries     = $countries;
		$this->vatRates      = $vatRates;
		$this->pricesHelper  = $pricesHelper;
	}

	public function getContainer(): BaseContainer
	{
		$container = new BaseContainer();
		$container->addCustomData('template', __DIR__ . '/ProductPricesContainer.latte');

		$labels = [
			'price'       => 'eshopCatalog.productPricesContainer.price' . (Config::load('product.priceIsWithoutVat', false) ? 'WithoutVat' : ''),
			'retailPrice' => 'eshopCatalog.productPricesContainer.retailPrice' . (Config::load('product.priceIsWithoutVat', false) ? 'WithoutVat' : ''),
			'currency'    => 'eshopCatalog.productPricesContainer.currency',
			'vatRate'     => 'eshopCatalog.productPricesContainer.vatRate',
		];
		$container->addCustomData('labels', $labels);

		$currencies = [];
		foreach (\Currency\Model\Config::load('whitelist') as $v)
			$currencies[$v] = $v;

		$vatRates = [];
		foreach ($this->vatRates->getAll() as $vrs)
			$vatRates[$vrs->getId()] = $vrs->name;

		foreach ($this->countries->getAllNameColumn() as $countryId => $countryName) {
			$row = $container->addContainer($countryId, $countryName);

			$row->addText('price', $labels['price'])
				->setHtmlType('number')
				->setHtmlAttribute('step', .01)
				->setHtmlAttribute('min', 0);
			$row->addText('retailPrice', $labels['retailPrice'])
				->setHtmlType('number')
				->setHtmlAttribute('step', .01)
				->setHtmlAttribute('min', 0);
			$row->addSelect('currency', $labels['currency'], $currencies);
			$row->addSelect('vatRate', $labels['vatRate'], $vatRates)
				->setRequired();
		}

		return $container;
	}

	public function saveData(Product $product, array $data): void
	{
		$this->pricesHelper->savePricesToProduct($product, $data);
	}

	public function setDefaults(BaseContainer &$container, Product $product): void
	{
		$d = [];
		foreach ($product->prices->toArray() as $countryId => $price) {
			/** @var ProductPrice $price */
			$d[$countryId] = [
				'price'       => $price->price,
				'retailPrice' => $price->retailPrice,
				'currency'    => $price->currency,
				'vatRate'     => $price->vatRate->getId(),
			];
		}

		$container->setDefaults($d);
	}
}
