<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Core\Model\Entities\TTranslateListener;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Doctrine\Common\Collections\ArrayCollection;
use Core\Model\Entities\TId;

/**
 * @ORM\Table(name="eshop_catalog__feature", indexes={@ORM\Index(name="published", columns={"id", "is_published"})})
 * @ORM\Entity
 * @ORM\EntityListeners({"FeatureListener", "Core\Model\Entities\TranslateListener"})
 */
class Feature
{

	use TId;
	use TTranslateListener;

	/**
	 * @var int
	 * @Gedmo\SortablePosition
	 * @ORM\Column(name="position", type="integer")
	 */
	protected $position;

	/**
	 * @var int
	 * @ORM\Column(name="is_published", type="smallint", length=1, options={"default":1})
	 */
	public $isPublished;

	/**
	 * @var int
	 * @ORM\Column(name="use_filter", type="smallint", length=1, options={"default":0})
	 */
	public $useAsFilter;

	/**
	 * @var FeatureTexts[]
	 * @ORM\OneToMany(targetEntity="FeatureTexts", mappedBy="id", indexBy="lang", cascade={"all"})
	 */
	protected $featureTexts;

	/**
	 * @var FeatureValue[]
	 * @ORM\OneToMany(targetEntity="FeatureValue", mappedBy="feature", indexBy="id", cascade={"all"})
	 */
	protected $featureValues;

	/**
	 * @var boolean
	 * @ORM\Column(name="beauty_url_active", type="boolean", options={"default": 0})
	 */
	public $beautyUrlActive;

	/**
	 * @var string
	 * @ORM\Column(name="bea_url_position", type="string", length=10, nullable=true)
	 */
	public $beaUrlPosition;

	/**
	 * @var int
	 * @ORM\Column(name="bea_url_priority", type="integer", length=10, nullable=false, options={"default":10})
	 */
	public $beaUrlPriority;

	/**
	 * @var string
	 * @ORM\Column(name="bea_url_pattern", type="string", nullable=false, options={"default":"%v%"})
	 */
	public $beaUrlPattern;

	/**
	 * @var string
	 * @ORM\Column(name="bea_url_position_multi", type="string", length=10, nullable=true)
	 */
	public $beaUrlPositionMulti;

	/**
	 * @var int
	 * @ORM\Column(name="bea_url_priority_multi", type="integer", length=10, nullable=false, options={"default":10})
	 */
	public $beaUrlPriorityMulti;

	/**
	 * @ORM\Column(name="export_heureka", type="smallint", length=1, options={"default":1})
	 */
	public int $exportHeureka = 1;

	/**
	 * @ORM\Column(name="export_zbozi", type="smallint", length=1, options={"default":1})
	 */
	public int $exportZbozi = 1;

	/**
	 * Feature constructor.
	 */
	public function __construct()
	{
		$this->featureTexts  = new ArrayCollection();
		$this->featureValues = new ArrayCollection();
		$this->setIsPublished(1);
		$this->setUseAsFilter(0);
		$this->setPosition(-1);

		$this->beautyUrlActive     = 0;
		$this->beaUrlPattern       = '%v%';
		$this->beaUrlPriority      = 10;
		$this->beaUrlPriorityMulti = 10;
	}

	public function addFeatureText($lang)
	{
		$this->featureTexts->set($lang, new FeatureTexts($this, $lang));
	}

	public function setFeatureText(FeatureTexts $featureTexts)
	{
		$this->featureTexts->set($featureTexts->getLang(), $featureTexts);
	}

	public function setFeatureTexts(array $texts): void
	{
		$this->featureTexts = new ArrayCollection($texts);
	}

	/**
	 * @param string|null $lang
	 *
	 * @return FeatureTexts|mixed|null
	 */
	public function getFeatureText(?string $lang = null) { return $this->featureTexts[$lang ?: $this->locale] ?? null; }

	public function getFeatureTexts() { return $this->featureTexts; }

	public function getFeatureValues()
	{
		return $this->featureValues;
	}


	/******
	 * === Position
	 */

	/**
	 * @param int $position
	 *
	 * @return $this
	 */
	public function setPosition(int $position)
	{
		$this->position = $position;

		return $this;
	}

	/**
	 * @return int
	 */
	public function getPosition() { return $this->position; }

	/******
	 * === Use filter
	 */

	/**
	 * @return int
	 */
	public function getUseAsFilter() { return $this->useAsFilter; }

	/**
	 * @param int $val
	 *
	 * @return $this
	 */
	public function setUseAsFilter($val)
	{
		if ($val && $val == 1)
			$this->useAsFilter = 1;
		else
			$this->useAsFilter = 0;

		return $this;
	}

	/******
	 * === Publish
	 */

	/**
	 * @return int
	 */
	public function getIsPublished() { return $this->isPublished; }

	/**
	 * @param int $val
	 *
	 * @return $this
	 */
	public function setIsPublished($val)
	{
		if ($val && $val == 1)
			$this->isPublished = 1;
		else
			$this->isPublished = 0;

		return $this;
	}
}

