<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Core\Model\Module;
use Doctrine;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Event\LifecycleEventArgs;
use Doctrine\ORM\Event\PreUpdateEventArgs;
use Doctrine\ORM\Mapping as ORM;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\AvailabilityService;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Navigation\Home;
use EshopCatalog\Model\Products;
use Nette\Utils\DateTime;
use Nette\Utils\FileSystem;
use Nettrine\ORM\EntityManagerDecorator;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Nette\Caching\Cache;
use Nette\SmartObject;
use Tracy\Debugger;

class ProductListener implements EventSubscriberInterface
{
	use SmartObject;

	protected static array $cleared = [];

	protected CacheService $cacheService;

	protected AvailabilityService $availabilityService;

	protected Products $productsService;

	protected static array $postUpdateDone = [];

	protected static array $parentVariants = [];

	protected EntityManagerDecorator $em;

	public static bool $updateVariant = true;

	public function __construct(CacheService $cacheService, AvailabilityService $availabilityService, Products $products, EntityManagerDecorator $em)
	{
		$this->cacheService        = $cacheService;
		$this->availabilityService = $availabilityService;
		$this->productsService     = $products;
		$this->em                  = $em;
	}

	public static function getSubscribedEvents(): array
	{
		return [];
	}

	/**
	 * @ORM\PreUpdate
	 *
	 * @param Product            $product
	 * @param PreUpdateEventArgs $args
	 */
	public function preUpdate($product, PreUpdateEventArgs $args)
	{
		if ($product instanceof Product && !$product->unlimitedQuantity) {
			// Aktualizuje dostupnost pokud se změnil počet skladem
			if ($product->getAvailability() && isset($args->getEntityChangeSet()['quantity'])) {
				$oldAvIdent = $product->getAvailability()->getIdent();
				$this->availabilityService->updateAvailabilityByQuantity($product);
				$newAvIdent = $product->getAvailability()->getIdent();

				if ($oldAvIdent != $newAvIdent) {
					$this->em->getConnection()->prepare("UPDATE " . $this->em->getClassMetadata(Product::class)->getTableName() . " SET id_availability = ? WHERE id = ?")
						->execute([$product->getAvailability()->getId(), $product->getId()]);
					//					$args->setNewValue('availability', $product->getAvailability());
				}
			}

			//			// Aktualizuje počet skladem pokud se dostunost změnila na vyprodáno
			//			if (isset($args->getEntityChangeSet()['availability'])) {
			//				$avChangeSet = $args->getEntityChangeSet()['availability'];
			//
			//				if ($avChangeSet[1]->getIdent() == Availability::SOLD_OUT && $product->quantity != 0) {
			//					$product->quantity = 0;
			//				}
			//			}
		}
	}

	/**
	 * @ORM\PreFlush
	 *
	 * @param Product            $product
	 * @param LifecycleEventArgs $args
	 */
	public function onPreFlush($product, $args)
	{
		if (!self::$cleared[$product->getId()]) {
			$this->cacheService->clean('product', [
				Cache::TAGS => [
					'product/' . $product->getId(),
				],
			]);

			$this->cacheService->clean('navigation', [
				Cache::TAGS => [
					Home::CACHE_PRODUCT . '/' . $product->getId(),
				],
			]);
		}
		self::$cleared[$product->getId()] = true;
	}

	/**
	 * @ORM\PostUpdate
	 *
	 * @param Product            $product
	 * @param LifecycleEventArgs $event
	 *
	 * @throws Doctrine\DBAL\DBALException
	 * @throws Doctrine\ORM\NonUniqueResultException
	 */
	public function postHandlerUpdate(Product $product, LifecycleEventArgs $event)
	{
		if ($product instanceof Product === false)
			return;

		if (Module::isAdmin() && $product->isVariant() && static::$updateVariant) {
			if ($product->isVariant()->isDefault === 1) {
				$variants = $product->variants;
				if (!$variants) {
					$table = $event->getEntityManager()->getClassMetadata(get_class($product->isVariant()))->getTableName();

					$varIds = [];
					foreach ($event->getEntityManager()->getConnection()
						         ->fetchAllAssociative("SELECT product_id FROM $table WHERE variant_id = ?", [$product->isVariant()->getVariantId()]) as $row)
						$varIds[] = $row['product_id'];

					$variants = $event->getEntityManager()->getRepository(ProductVariant::class)->createQueryBuilder('pv')
						->where('pv.product IN (:ids)')->setParameter('ids', $varIds)
						->getQuery()->getResult();
				} else {
					$variants = $variants->toArray();
				}

				foreach ($variants as $va) {
					/** @var ProductVariant $va */
					$vaProd = $va->getProduct();

					if ($vaProd->getId() === $product->getId() || $va->isDefault === 1) {
						continue;
					}

					$this->productsService->updateVariant($va, $product);
				}
			} else if (!in_array($product->getId(), self::$postUpdateDone)) {
				self::$postUpdateDone[] = $product->getId();

				$va = $product->isVariant();

				if (!isset(self::$parentVariants[$va->getVariantId()]))
					self::$parentVariants[$va->getVariantId()] = $event->getEntityManager()->getRepository(Product::class)->createQueryBuilder('p')
						->innerJoin('p.isVariant', 'va')
						->where('va.variantId = :variantId')->setParameter('variantId', $va->getVariantId())
						->andWhere('va.isDefault = 1')->getQuery()->getOneOrNullResult();
				$parent = self::$parentVariants[$va->getVariantId()];

				$this->productsService->updateVariant($va, $parent);
			}
		}
	}

	/**
	 * @ORM\PostUpdate
	 * @ORM\PostRemove
	 *
	 * @param Product            $product
	 * @param LifecycleEventArgs $event
	 *
	 * @throws Doctrine\ORM\ORMException
	 */
	public function postHandler(Product $product, LifecycleEventArgs $event)
	{
		$em = $event->getEntityManager();

		if (Config::load('enableLoggingQuantityChange' . false)) {
			$changeSet = $em->getUnitOfWork()->getEntityChangeSet($product);
			if (isset($changeSet['quantity'])) {
				$old = $changeSet['quantity'][0];
				$new = $changeSet['quantity'][1];

				try {
					$dir  = '_quantities/' . (new DateTime())->format('Y-m-d');
					$file = $product->getId();
					FileSystem::createDir(LOG_DIR . '/' . $dir);
					$e = new \Exception();
					Debugger::log(print_r($old . ' -> ' . $new, true), $dir . '/' . $file);
					Debugger::log(print_r($e->getTraceAsString(), true), $dir . '/' . $file);
				} catch (\Exception $e) {
				}
			}
		}

		self::$cleared[$product->getId()] = true;
	}

	/**
	 * @ORM\PreRemove
	 *
	 * @param Product            $product
	 * @param LifecycleEventArgs $events
	 */
	public function preRemoveHandler(Product $product, LifecycleEventArgs $events): void
	{
		if ($product->getGallery() && !$product->isVariant()) {
			$events->getEntityManager()->remove($product->getGallery());
			$events->getEntityManager()->flush($product->getGallery());
		}
	}

}
