<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\ImportExport;

use Core\Model\Countries;
use Core\Model\Entities\ExtraField;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\Strings;
use Core\Model\Http\CsvResponse;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Model\Manufacturers;
use EshopCatalog\AdminModule\Model\ProductPrices;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductPriceHistory;
use EshopCatalog\Model\Entities\ProductPriceLevel;
use EshopCatalog\Model\Entities\ProductTexts;
use EshopOrders\Model\Entities\GroupCustomers;
use EshopStock\DI\EshopStockExtension;
use EshopStock\Model\Entities\SupplyProduct;
use Nette\Http\FileUpload;
use Nette\Utils\ArrayHash;
use Tracy\Debugger;

class ProductsByManufacturer extends BaseControl
{
	protected Products      $productsService;
	protected Manufacturers $manufacturers;
	protected CacheService  $cacheService;
	protected Countries     $countries;
	protected ProductPrices $productPrices;

	protected array $customFields = [];

	public function __construct(
		Products      $productsService,
		Manufacturers $manufacturers,
		CacheService  $cacheService,
		Countries     $countries,
		ProductPrices $productPrices
	)
	{
		$this->productsService = $productsService;
		$this->manufacturers   = $manufacturers;
		$this->cacheService    = $cacheService;
		$this->countries       = $countries;
		$this->productPrices   = $productPrices;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setShowLangSwitcher(false);

		$form->addSelect('manufacturer', 'eshopCatalog.product.manufacturer', [null => 'default.all'] + $this->manufacturers->getOptionsForSelect())
			->setTranslator(null);
		$form->addSubmit('export', 'eshopCatalog.importExport.exportData')
			->setValidationScope(null);

		$form->addUpload('upload', 'eshopCatalog.importExport.dataForImport');

		$baseFields = ['name', 'code1', 'code2', 'ean'];

		if (Config::load('product.allowRetailPrice')) {
			$baseFields[] = 'retailPrice';
		}

		$baseFields[] = 'price';

		$allowedCountryPrices = Config::load('importExport.countryPrices');
		if (Config::load('enableCountryPrices')) {
			$tmp = [];
			foreach ($this->countries->getAllNameColumn() as $countryId => $countryName) {
				if (!Arrays::contains($allowedCountryPrices, Strings::upper($countryId))) {
					continue;
				}

				$tmp[] = [
					'price_' . $countryId,
				];
			}

			$baseFields = array_merge($baseFields, ...$tmp);
		}

		$baseFields    = array_merge($baseFields, ['quantity', 'width', 'height', 'depth', 'weight']);
		$fields        = [];
		$defaultFields = [];

		if (Config::load('product.allowCountryOfOrigin')) {
			$baseFields[] = 'countryOfOrigin';
		}

		if (Config::load('product.allowHSCustomsCode')) {
			$baseFields[] = 'hsCustomsCode';
		}

		foreach ($baseFields as $v) {
			if (Strings::contains($v, '_')) {
				$tmp        = explode('_', $v);
				$fields[$v] = $this->t('eshopCatalog.importExport.fields.' . $tmp[0]) . ' ' . ($tmp[1] ?? '');
			} else {
				$fields[$v] = $this->t('eshopCatalog.importExport.fields.' . $v);
			}

			if ($v !== 'quantity') {
				$defaultFields[] = $v;
			}
		}

		$tmp            = $this->em->getRepository(GroupCustomers::class)->createQueryBuilder('gc', 'gc.id')
			->select('gc.id, gc.short, gc.name')
			->getQuery()->getArrayResult();
		$customerGroups = $this->sortGroups($tmp);

		foreach ($customerGroups as $row) {
			$fields['cg' . $row['id']] = $this->t('eshopCatalog.importExport.fields.priceLevel') . ' - ' . $row['name'];
			$defaultFields[]           = 'cg' . $row['id'];
		}

		foreach ($this->customFields as $row) {
			$fields[$row['name']] = $this->t('eshopCatalog.importExport.fields.' . $row['name']);
		}

		$form->addCheckboxList('useFields', $this->t('eshopCatalog.importExport.useFields'), $fields)
			->setDefaultValue($defaultFields)
			->setTranslator(null);

		$form->addSubmit('import', 'eshopCatalog.importExport.importData');

		$form->onSuccess[] = [$this, 'formOnSuccess'];

		return $form;
	}

	protected function sortGroups(array $data): array
	{
		$result      = [];
		$groupsOrder = Config::load('importExport.groupsOrder', []);
		foreach ($groupsOrder as $id) {
			if (isset($data[$id])) {
				$result[$id] = $data[$id];
				unset($data[$id]);
			}
		}

		return $result + $data;
	}

	public function formOnSuccess(BaseForm $form, ArrayHash $values): bool
	{
		set_time_limit(300);
		$allowedCountryPrices = Config::load('importExport.countryPrices');

		$tmp            = $this->em->getRepository(GroupCustomers::class)
			->createQueryBuilder('gc', 'gc.id')
			->select('gc.id, gc.short, gc.name')
			->getQuery()->getArrayResult();
		$customerGroups = $this->sortGroups($tmp);

		try {
			// Pokud se data exportují tak načteme data a odešleme do CSV response
			if ($form->isSubmitted()->name === 'export') {
				$filename = 'import-export-' . date('Y-m-d') . '.csv';

				// Základní data
				$head             = [];
				$disableTranslate = [];
				$headFields       = ['id', 'lang', 'name', 'code1', 'code2', 'ean'];

				// Skupiny zákazníků
				foreach ($customerGroups as $row) {
					$disableTranslate[] = $row['name'];
					$headFields[]       = $row['name'];
				}

				if (Config::load('product.allowRetailPrice')) {
					$headFields[] = 'retailPrice';
				}

				$headFields[] = 'price';

				if (Config::load('enableCountryPrices')) {
					$tmp = [];
					foreach ($this->countries->getAllNameColumn() as $countryId => $countryName) {
						if (!Arrays::contains($allowedCountryPrices, Strings::upper($countryId))) {
							continue;
						}

						$tmp[] = [
							'price_' . $countryId,
						];
					}

					$headFields = array_merge($headFields, ...$tmp);
				}

				$headFields = array_merge($headFields, ['width', 'height', 'depth', 'weight', 'quantity']);

				// Sklad
				$stockData = [];
				if (class_exists(EshopStockExtension::class)) {
					$headFields[] = 'stock';

					foreach ($this->em->getRepository(SupplyProduct::class)->createQueryBuilder('sp')
						         ->select('IDENTITY(sp.product) as product, IDENTITY(sp.order) as order, sp.writtenOffDate')
						         ->getQuery()->getScalarResult() as $row) {
						if (!isset($stockData[$row['product']]))
							$stockData[$row['product']] = 0;
						$stockData[$row['product']]++;

						if ($row['order'] || $row['writtenOffDate']) {
							$stockData[$row['product']]--;
						}
					}
				}

				if (Config::load('product.allowCountryOfOrigin')) {
					$headFields[] = 'countryOfOrigin';
				}

				if (Config::load('product.allowHSCustomsCode')) {
					$headFields[] = 'hsCustomsCode';
				}

				foreach ($headFields as $v) {
					if (Arrays::contains($disableTranslate, $v)) {
						$head[] = $v;
						continue;
					}

					if (Strings::contains($v, '_')) {
						$tmp    = explode('_', $v);
						$head[] = $this->t('eshopCatalog.importExport.head.' . $tmp[0]) . ' ' . ($tmp[1] ?? '');
					} else {
						$head[] = $this->t('eshopCatalog.importExport.head.' . $v);
					}
				}

				foreach ($this->customFields as $cs) {
					array_splice($head, $cs['position'] - 1, 0, [$this->t('eshopCatalog.importExport.head.' . $cs['name'])]);
				}

				$csv = '"' . implode('";"', $head) . '"';
				$csv .= PHP_EOL;

				$qb = $this->productsService->getEr()->createQueryBuilder('p')
					->select('p.id, p.code1, pt.lang, pt.name, p.code2, p.ean, p.quantity, p.retailPrice, p.price, p.width, p.height, p.depth, p.weight')
					->andWhere('p.isDeleted = 0')
					->addSelect('GROUP_CONCAT(CONCAT(IDENTITY(pl.groupId), \'|\' , pl.price)) as priceLevels')
					->addSelect('IDENTITY(p.countryOfOrigin) as countryOfOrigin')
					->innerJoin('p.productTexts', 'pt')
					->leftJoin('p.priceLevels', 'pl')
					->groupBy('p.id');

				if ($values->manufacturer) {
					$qb->andWhere('p.manufacturer = :manufacturer')->setParameter('manufacturer', $values->manufacturer);
				}

				$data = [];
				foreach ($qb->getQuery()->getArrayResult() as $row) {
					$data[$row['id']] = $row;
				}

				$extraFields = [];
				foreach ($this->em->getRepository(ExtraField::class)->createQueryBuilder('ef')
					         ->select('ef.sectionKey, ef.key, ef.value')
					         ->where('ef.sectionName = :sectionName')
					         ->andWhere('ef.sectionKey IN (' . implode(',', array_keys($data)) . ')')
					         ->setParameters([
						         'sectionName' => Product::EXTRA_FIELD_SECTION,
					         ])->getQuery()->getArrayResult() as $row) {
					$extraFields[$row['sectionKey']][$row['key']] = $row['value'];
				}

				$productPrices = [];
				if (Config::load('enableCountryPrices')) {
					foreach (array_chunk(array_keys($data), 500) as $chunk) {
						foreach ($this->productPrices->getEr()->createQueryBuilder('pp')
							         ->select('IDENTITY(pp.product) as product, IDENTITY(pp.country) as country, pp.currency, pp.retailPrice, pp.price')
							         ->where('pp.product IN (' . implode(',', $chunk) . ')')
							         ->getQuery()->getArrayResult() as $row) {
							if (!Arrays::contains($allowedCountryPrices, Strings::upper($row['country']))) {
								continue;
							}

							$productPrices[$row['product']][Strings::upper($row['country'])] = [
								'price' => $row['price'],
							];
						}
					}
				}

				foreach ($data as $row) {
					$isVar = $row['isDefault'] === 0;

					// Základní data
					$rowData = [
						$row['id'],
						$row['lang'],
						str_replace('"', '""', (string) $row['name']),
						$row['code1'],
						$row['code2'],
						'="' . $row['ean'] . '"',
					];

					// Skupiny zákazníků
					$priceLevels = [];
					if ($row['priceLevels']) {
						foreach (explode(',', $row['priceLevels']) as $v) {
							$v                  = explode('|', $v);
							$priceLevels[$v[0]] = $v[1] ?? '';
						}
					}

					foreach ($customerGroups as $k => $v) {
						$rowData[] = str_replace('.', ',', ($priceLevels[$k] ?? ''));
					}

					if (Config::load('product.allowRetailPrice')) {
						$rowData[] = str_replace('.', ',', $row['retailPrice']);
					}

					$rowData[] = str_replace('.', ',', $row['price']);

					if (Config::load('enableCountryPrices')) {
						foreach ($this->countries->getAllNameColumn() as $countryId => $countryName) {
							if (!Arrays::contains($allowedCountryPrices, Strings::upper($countryId))) {
								continue;
							}

							$tmp = $productPrices[$row['id']][Strings::upper($countryId)] ?? [];

							$rowData[] = $tmp['price'] ? str_replace('.', ',', (string) $tmp['price']) : '';
						}
					}

					$rowData[] = $row['width'];
					$rowData[] = $row['height'];
					$rowData[] = $row['depth'];
					$rowData[] = $row['weight'];
					$rowData[] = $row['quantity'];

					// Sklad
					if (class_exists(EshopStockExtension::class)) {
						$rowData[] = $stockData[$row['id']] ?? 0;
					}

					if (Config::load('product.allowCountryOfOrigin')) {
						$rowData[] = $row['countryOfOrigin'];
					}

					if (Config::load('product.allowHSCustomsCode')) {
						$rowData[] = $row['hsCustomsCode'];
					}

					foreach ($this->customFields as $cs) {
						$csValue = '';
						if ($cs['source'] === 'ef') {
							$csValue = $extraFields[$row['id']][$cs['name']];
						}

						if ($cs['type'] === 'price') {
							$csValue = str_replace('.', ',', $csValue);
						}

						array_splice($rowData, $cs['position'] - 1, 0, [$csValue]);
					}

					foreach ($rowData as $k => $v) {
						if (Strings::startsWith((string) $v, '=')) {
							continue;
						}

						$rowData[$k] = '"' . $v . '"';
					}

					$csv .= implode(';', $rowData);
					$csv .= PHP_EOL;
				}

				CsvResponse::sendResponse($filename, $csv);
			} elseif ($values->upload && $values->upload->hasFile()) {
				// Při importu připravíte názvy tabulek a další hodnoty
				$productTextsTableName = $this->em->getClassMetadata(ProductTexts::class)->getTableName();
				$priceLevelsTableName  = $this->em->getClassMetadata(ProductPriceLevel::class)->getTableName();

				/** @var FileUpload $upload */
				$upload = $values->upload;
				$fr     = fopen($upload->getTemporaryFile(), 'r');
				$this->em->beginTransaction();
				$csvData           = [];
				$priceLevels       = [];
				$allowedProductsId = [];
				$keysForClean      = [];
				$canUse            = $values->useFields;

				// Zjistíme produkty patřící k výrobci nebo všechny
				$qb = $this->productsService->getEr()->createQueryBuilder('p')->select('p.id')
					->andWhere('p.isDeleted = 0');

				if ($values->manufacturer) {
					$qb->where('p.manufacturer = :manu')->setParameter('manu', $values->manufacturer);
				}

				foreach ($qb->getQuery()->getArrayResult() as $row) {
					$allowedProductsId[] = $row['id'];
				}

				$extraFields = [];
				foreach ($this->em->getRepository(ExtraField::class)->createQueryBuilder('ef')
					         ->select('ef.sectionKey, ef.key, ef.value')
					         ->where('ef.sectionName = :sectionName')
					         ->andWhere('ef.sectionKey IN (' . implode(',', $allowedProductsId) . ')')
					         ->setParameters([
						         'sectionName' => Product::EXTRA_FIELD_SECTION,
					         ])->getQuery()->getArrayResult() as $row) {
					$extraFields[$row['sectionKey']][$row['key']] = $row['value'];
				}

				// Přečteme nahraný csv soubor
				while (!feof($fr) && $line = fgetcsv($fr, null, ';')) {
					// V prvním řádku načteme dynamické sloupečky
					if (Strings::toAscii($line[0]) == 'ID') {
						foreach ($line as $k => $v) {
							$tmp = explode('|', $v);
						}

						continue;
					}

					if (!$line[0]) {
						continue;
					}

					$productId = (int) $line[0];

					// Zkontrolujeme existenci produktu
					if (!in_array($productId, $allowedProductsId)) {
						continue;
					}

					$csvData[$productId] = array_map('utf8_encode', $line);
				}

				// Načteme cenové hladiny
				foreach ($this->em->getRepository(ProductPriceLevel::class)->createQueryBuilder('pl')
					         ->select('IDENTITY(pl.productId) as productId, IDENTITY(pl.groupId) as groupId, pl.price')
					         ->innerJoin('pl.productId', 'p')
					         ->getQuery()->getArrayResult() as $row) {
					$priceLevels[$row['groupId'] . '-' . $row['productId']] = $row['price'];
				}

				foreach ($csvData as $productId => $line) {
					$customFieldsData = [];
					foreach ($this->customFields as $k => $cs) {
						$pos                  = $cs['position'] - 1;
						$customFieldsData[$k] = $line[$pos];
						unset($line[$pos]);
					}
					$line = array_values($line);

					$lang = trim($line[1]);
					$name = trim($line[2]);

					if ($lang) {
						$keysForClean[] = 'product/' . $productId . '/' . $lang;
					}

					// Aktualizace názvu
					if (in_array('name', $canUse) && $lang && $name) {
						$name = preg_replace("/u([0-9a-fA-F]{4})/", "&#x\\1;", $name);
						$name = iconv("UTF-8", "ISO-8859-1//TRANSLIT", $name);
						$this->em->getConnection()->update($productTextsTableName, [
							'name' => $name,
						], [
							'id'   => $productId,
							'lang' => $lang,
						]);
					}

					$product = $this->em->getReference(Product::class, $productId);

					$code1 = trim($line[3]);
					$code2 = trim($line[4]);
					$ean   = trim($line[5]);
					$ean   = (int) trim($ean, '"=');

					$lineK                = 5;
					$customerGroupsPrices = [];
					foreach ($customerGroups as $groupId => $groupName) {
						$lineK++;
						$customerGroupsPrices[$groupId] = strtr(trim($line[$lineK]), [' ' => '', ',' => '.']);
					}

					if (Config::load('product.allowRetailPrice')) {
						$lineK++;
						$retailPrice = strtr(trim($line[$lineK]), [' ' => '', ',' => '.']);
					} else {
						$retailPrice = null;
					}

					$lineK++;
					$price = strtr(trim($line[$lineK]), [' ' => '', ',' => '.']);

					if (Config::load('enableCountryPrices')) {
						foreach ($this->countries->getAllNameColumn() as $countryId => $countryName) {
							if (!Arrays::contains($allowedCountryPrices, Strings::upper($countryId))) {
								continue;
							}

							$update = [];

							$lineK++; // price
							if (Arrays::contains($canUse, 'price_' . $countryId)) {
								$ppPrice         = strtr(trim($line[$lineK]), [' ' => '', ',' => '.']);
								$update['price'] = is_numeric($ppPrice) ? $ppPrice : 'null';
							}

							if (!empty($update)) {
								$onDuplicate = [];
								foreach ($update as $uK => $uV) {
									$onDuplicate[] = $uK . ' = ' . $uV;
								}

								$prefilled = Config::load('importExport.prefilledCountryPrices');
								if (isset($prefilled['currency_' . Strings::lower($countryId)])) {
									$update['currency_code'] = "'" . $prefilled['currency_' . Strings::lower($countryId)] . "'";
									$onDuplicate[]           = "currency_code = '" . $prefilled['currency_' . Strings::lower($countryId)] . "'";
								}

								$this->em->getConnection()->executeQuery("INSERT INTO eshop_catalog__product_price 
    								(product, country, " . implode(', ', array_keys($update)) . ") 
    								VALUES ({$productId}, '{$countryId}', " . implode(', ', $update) . ")
    								ON DUPLICATE KEY UPDATE " . implode(', ', $onDuplicate));
							}
						}
					}

					$lineK++;
					$width = $line[$lineK] !== '' && $line[$lineK] !== null ? (int) $line[$lineK] : null;

					$lineK++;
					$height = $line[$lineK] !== '' && $line[$lineK] !== null ? (int) $line[$lineK] : null;

					$lineK++;
					$depth = $line[$lineK] !== '' && $line[$lineK] !== null ? (int) $line[$lineK] : null;

					$lineK++;
					$weight = $line[$lineK] !== '' && $line[$lineK] !== null ? (int) $line[$lineK] : null;

					$lineK++;
					$quantity = $line[$lineK] !== '' && $line[$lineK] !== null ? (int) $line[$lineK] : null;

					// skip  pocet ks ve skladu
					$lineK++;

					// next
					if (Config::load('product.allowCountryOfOrigin')) {
						$lineK++;
						if (in_array('countryOfOrigin', $canUse)) {
							$countryOfOrigin = $line[$lineK] !== '' && $line[$lineK] !== null ? (string) $line[$lineK] : null;

							if ($countryOfOrigin !== null && !isset($this->countries->getAllNameColumn()[$countryOfOrigin])) {
								throw new \Exception($this->translator->translate('eshopCatalog.importExport.err.countryOfOriginNotFound', ['country' => $countryOfOrigin]));
							}

							$product->countryOfOrigin = $countryOfOrigin ? $this->countries->getReference($countryOfOrigin) : null;
						}
					}

					if (Config::load('product.allowHSCustomsCode')) {
						$lineK++;
						if (in_array('hsCustomsCode', $canUse)) {
							$hsCustomsCode          = $line[$lineK] !== '' && $line[$lineK] !== null ? (int) $line[$lineK] : null;
							$product->hsCustomsCode = $hsCustomsCode;
						}
					}

					// Aktulizace hodnot pokud jsou validní
					if ($code1 && Arrays::contains($canUse, 'code1')) {
						$product->code1 = $code1;
					}

					if ($code2 && Arrays::contains($canUse, 'code2')) {
						$product->code2 = $code2;
					}

					if ($ean && $ean !== '=""' && Arrays::contains($canUse, 'ean')) {
						$product->ean = (string) $ean;
					}

					if ($quantity !== null && is_numeric($quantity) && Arrays::contains($canUse, 'quantity')) {
						$product->quantity;
					}

					if ($retailPrice !== null && is_numeric($retailPrice) && Arrays::contains($canUse, 'retailPrice')) {
						$product->retailPrice = $retailPrice;
					}

					if ($price !== null && is_numeric($price) && Arrays::contains($canUse, 'price')) {
						if (Config::load('enablePriceHistory') && $product && $product->price !== $price) {
							$priceHistory = new ProductPriceHistory(
								$product,
								(float) $product->price,
								(float) $price,
								'importExport',
							);
							$this->em->persist($priceHistory);
						}

						$product->price = $price;
					}

					foreach (['width', 'height', 'depth', 'weight'] as $c) {
						if (${$c} !== null && Arrays::contains($canUse, $c)) {
							$product->{$c} = ${$c};
						}
					}

					$this->em->persist($product);
					$this->em->flush($product);

					// Aktualizace cenových hladin
					foreach ($customerGroupsPrices as $groupId => $price) {
						$exist = isset($priceLevels[$groupId . '-' . $productId]);

						if (is_numeric($price)) {
							// Aktualizace nebo vytvoření
							if ($exist) {
								$this->em->getConnection()->update($priceLevelsTableName, [
									'price' => $price,
								], [
									'product_id' => $productId,
									'group_id'   => $groupId,
								]);
							} else {
								$this->em->getConnection()->insert($priceLevelsTableName, [
									'product_id' => $productId,
									'group_id'   => $groupId,
									'price'      => $price,
								]);
							}
						} else if ($exist) {
							// Smazání
							$this->em->getConnection()->delete($priceLevelsTableName, [
								'product_id' => $productId,
								'group_id'   => $groupId,
							]);
						}
					}

					foreach ($this->customFields as $k => $cs) {
						$value = $customFieldsData[$k] ?? null;

						if ($cs['type'] === 'price') {
							$value = str_replace(',', '.', $value);
						}

						if ($cs['source'] === 'ef') {
							$ef = $extraFields[$productId][$cs['name']] ?? null;

							if ($ef) {
								if ($value === '' || $value === null) {
									$this->em->getConnection()->delete('core__extra_field', [
										'section_name' => Product::EXTRA_FIELD_SECTION,
										'section_key'  => $productId,
										'`key`'        => $cs['name'],
									]);
								} else {
									$this->em->getConnection()->executeStatement("UPDATE core__extra_field SET `value` = ? WHERE section_name = ? AND section_key = ? AND `key` = ?", [
										$value, Product::EXTRA_FIELD_SECTION, $productId, $cs['name'],
									]);
								}
							} else if ($value) {
								$this->em->getConnection()->insert('core__extra_field', [
									'section_name' => Product::EXTRA_FIELD_SECTION,
									'section_key'  => $productId,
									'`key`'        => $cs['name'],
									'`value`'      => $value,
									'`lang`'       => null,
								]);
							}
						}
					}
				}

				$this->em->commit();

				if ($keysForClean) {
					$this->cacheService->productCache->remove($keysForClean);
				}

				$this->getPresenter()->flashMessageSuccess('default.saved');
				$this->getPresenter()->redrawControl('flashes');
				$this->template->messageOk = $this->t('default.saved');
				$this->redrawControl('formErrors');
			}
		} catch (\Exception $e) {
			$this->getPresenter()->flashMessageDanger($e->getMessage());
			$this->getPresenter()->redrawControl('flashes');

			$this->template->messageError = $e->getMessage();
			$this->redrawControl('formErrors');

			if ($this->em->getConnection()->isTransactionActive())
				$this->em->rollback();

			return false;
		}

		return true;
	}
}
