<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\ProductAssign;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\AdminModule\Model\ProductsAssignedToCustomers;
use EshopCatalog\Model\Entities\ProductAssignCustomer;
use EshopOrders\AdminModule\Model\Customers;
use Nette\Application\LinkGenerator;
use Nette\Utils\ArrayHash;

class ProductsToCustomersForm extends BaseControl
{
	protected ProductsAssignedToCustomers $productsAssignedToCustomers;
	protected Products                    $products;
	protected Customers                   $customers;
	protected LinkGenerator               $linkGenerator;

	public function __construct(
		ProductsAssignedToCustomers $productsAssignedToCustomers,
		Products                    $products,
		Customers                   $customers,
		LinkGenerator               $linkGenerator
	)
	{
		$this->productsAssignedToCustomers = $productsAssignedToCustomers;
		$this->products                    = $products;
		$this->customers                   = $customers;
		$this->linkGenerator               = $linkGenerator;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ======================== Handler
	 */

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();
		$form->setShowLangSwitcher(false);
		$form->getElementPrototype()
			->setAttribute('id', 'eshopCatalog_productsToCustomersForm');
		$form->addHidden('productId');
		$form->addHidden('customerId');

		$form->addBool('hideProductInListing', 'eshopCatalog.productAssignCustomer.hideProductInListing')
			->setDescription('eshopCatalog.productAssignCustomer.hideProductInListingDescription')
			->setDefaultValue(1);

		$form->addText('product', 'eshopCatalog.product.searchProduct')
			->setHtmlAttribute('data-autocomplete-name', 'productsSetAsVariantsFor')
			->setHtmlAttribute('data-autocomplete-target', $form['productId']->getHtmlId())
			->setHtmlAttribute('data-autocomplete-keys', 'id,code1,ean,name')
			->setHtmlAttribute('data-autocomplete-url', $this->linkGenerator->link('EshopCatalog:Cron:Products:loadAll'));

		$form->addText('customer', 'eshopOrders.customer.searchCustomer')
			->setHtmlAttribute('data-autocomplete-name', 'productsSetAsVariantsFor')
			->setHtmlAttribute('data-autocomplete-target', $form['customerId']->getHtmlId())
			->setHtmlAttribute('data-autocomplete-keys', 'id,name')
			->setHtmlAttribute('data-autocomplete-url', $this->linkGenerator->link('EshopOrders:Admin:customers:jsonList'));

		$form->addSaveCancelControl();
		$form['saveControl']->removeComponent($form['saveControl']['save']);

		$form->onValidate[] = [$this, 'formValidate'];
		$form->onSuccess[]  = [$this, 'formSuccess'];

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values): void
	{
		if (!$values->productId || !$this->products->get($values->productId)) {
			$form['product']->addError('eshopCatalog.productAssignCustomer.missingProduct');
		}
		if (!$values->customerId || !$this->customers->get($values->customerId)) {
			$form['customer']->addError('eshopCatalog.productAssignCustomer.missingCustomer');
		}

		if ($form->hasErrors()) {
			$this->redrawControl('form');
		}
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$product = $this->products->get($values->productId);
			$entity  = new ProductAssignCustomer(
				$product,
				$this->customers->getReference($values->customerId),
			);

			$product->disableListing = (int) $values->hideProductInListing;

			$this->em->persist($product);
			$this->em->persist($entity);

			$this->em->flush();
			$this->getPresenter()->flashMessageSuccess('default.saved');
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}
}
