<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model\Dao;

use Core\Model\Dao\Traits\ExtraFields;
use Core\Model\Entities\TSeo;
use Nette\Utils\DateTime;

/**
 * Class Category
 * @package EshopCatalog\FrontModule\Model\Dao
 */
class Category
{
	use TSeo;
	use ExtraFields;

	/** @var int */
	public $id;

	/** @var string */
	public $name;

	/** @var string */
	public $nameH1;

	/** @var string */
	public $alias;

	public ?string $shortDescription = null;

	public ?string $description = null;

	public ?string $emptyText = null;

	/** @var string */
	public $image;

	/** @var int */
	public $lvl;

	/** @var self[] */
	public $child = [];

	/** @var int */
	protected $parentId;

	/** @var self */
	protected $parent;

	/** @var string */
	public $link = '#';

	/** @var DateTime */
	public $created;

	/** @var DateTime */
	public $modified;

	/** @var int */
	public $filtersFromParent;

	/** @var array */
	public $attrs;

	/** @var string */
	public $rod;

	/** @var bool */
	public bool $canProductsAddToCart = true;

	/** @var self[] */
	public array $related = [];

	public function getId(): int { return $this->id; }

	public function getNameH1(): string { return $this->nameH1 ?: $this->name; }

	public function setParent(Category &$category): self
	{
		$this->parent = &$category;

		return $this;
	}

	public function getParent(): ?Category { return $this->parent; }

	/**
	 * @return self[]
	 */
	public function getParentPath()
	{
		$path = [];

		$parent = $this->getParent();
		while ($parent) {
			$path[$parent->getId()] = $parent;
			$parent                 = $parent->getParent();
		}

		return $path;
	}

	public function getParentPathString(): string
	{
		return implode(' > ', array_map(fn($cat) => $cat->name, $this->getParentPath()));
	}

	public function getParentPathStringFlipped(): string
	{
		return implode(' > ', array_map(fn($cat) => $cat->name, array_reverse($this->getParentPath())));
	}

	public function setParentId(int $id): self
	{
		$this->parentId = $id;

		return $this;
	}

	public function getParentId(): ?int { return $this->parentId; }

	public function setChild(array $arr): self
	{
		$this->child = $arr;

		return $this;
	}

	public function getRoot(): ?Category
	{
		$r = $this->getParent();

		while ($r && $r->getParent())
			$r = $r->getParent();

		return $r;
	}

	public function getRootId()
	{
		$cat = $this;
		while ($cat) {
			if (!$cat->getParent())
				return $cat->getParentId();
			$cat = $cat->getParent();
		}

		return $this->getParentId();
	}

	/**
	 * @return Category[]
	 */
	public function getChild(): array { return $this->child ?: []; }

	public function addAttr(string $key, $value): self
	{
		$this->attrs[$key] = $value;

		return $this;
	}

	public function setAttrs(array $attrs): self
	{
		$this->attrs = $attrs;

		return $this;
	}

	public function getAttr(string $key) { return $this->attrs[$key] ?? null; }

	public function getAttrs(): array { return $this->attrs ?: []; }
}
