<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Core\Model\Entities\TTranslateListener;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Core\Model\Entities\TId;

/**
 * @ORM\Entity
 * @ORM\Table(name="eshop_catalog__availability")
 * @ORM\EntityListeners({"Core\Model\Entities\TranslateListener", "AvailabilityListener"})
 */
class Availability
{
	use TId;
	use TTranslateListener;

	const IN_STOCK = 'inStock';
	const SOLD_OUT = 'soldOut';
	const PREORDER = 'preorder';

	/**
	 * @var string
	 * @ORM\Column(name="ident", type="string", length=255, nullable=false)
	 */
	protected $ident;

	/**
	 * @var int
	 * @ORM\Column(name="delay", type="smallint", options={"unsigned":true, "default":0}, nullable=false)
	 */
	public $delay;

	/**
	 * @var int
	 * @ORM\Column(name="can_show_on_list", type="smallint", options={"default": 1}, nullable=false)
	 */
	public $canShowOnList;

	/**
	 * @var int
	 * @ORM\Column(name="can_show_on_search", type="smallint", options={"default": 1}, nullable=false)
	 */
	public $canShowOnSearch;

	/**
	 * @var int
	 * @ORM\Column(name="can_add_to_cart", type="smallint", options={"default": 1}, nullable=false)
	 */
	public $canAddToCart;

	/**
	 * @var string
	 * @ORM\Column(name="color", type="string", nullable=true)
	 */
	public $color;
	/**
	 * @var string
	 * @ORM\Column(name="bg_color", type="string", nullable=true)
	 */
	public $bgColor;

	/**
	 * @var AvailabilityText[]
	 * @ORM\OneToMany(targetEntity="AvailabilityText", mappedBy="availability", indexBy="lang")
	 */
	protected $texts;

	/**
	 * @var int
	 * @ORM\Column(name="can_delete", type="smallint", options={"default": 1}, nullable=false)
	 */
	protected $canDelete;

	/**
	 * @var string|null
	 * @ORM\Column(name="open_graph_text", type="string", length=255, nullable=true)
	 */
	public ?string $openGraphText = null;

	/**
	 * @var int
	 * @Gedmo\SortablePosition
	 * @ORM\Column(name="position", type="integer")
	 */
	protected int $position;

	/**
	 * @ORM\Column(name="warehouse_overdraft", type="smallint", options={"unsigned": true, "default": 0}, nullable=false)
	 */
	public int $warehouseOverdraft = 0;

	public function __construct(string $ident, int $delay)
	{
		$this->setIdent($ident);
		$this->delay     = $delay;
		$this->texts     = new ArrayCollection();
		$this->canDelete = 1;
		$this->setPosition(-1);
	}

	public function getIdent(): string { return $this->ident; }

	public function setIdent(string $ident): self
	{
		$this->ident = $ident;

		return $this;
	}

	/**
	 * @param string $lang
	 *
	 * @return AvailabilityText|mixed
	 */
	public function getText(string $lang = '') { return $this->texts[$lang ?: $this->locale]; }

	public function getTexts() { return $this->texts; }

	public function setTexts(array $texts): self
	{
		$this->texts = new ArrayCollection($texts);

		return $this;
	}

	public function canDelete(): bool { return (bool) $this->canDelete; }

	public function getPosition(): int { return $this->position; }

	/**
	 * @param int $position
	 *
	 * @return $this
	 */
	public function setPosition(int $position): self
	{
		$this->position = $position;

		return $this;
	}
}
