<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\ImportExport;

use Core\Model\Http\CsvResponse;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\AdminModule\Model\Manufacturers;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductPriceLevel;
use EshopCatalog\Model\Entities\ProductTexts;
use EshopOrders\Model\Entities\GroupCustomers;
use Nette\Http\FileUpload;
use Nette\Utils\ArrayHash;

class ProductsByManufacturer extends BaseControl
{
	protected Products $productsService;

	protected Manufacturers $manufacturers;

	public function __construct(Products $productsService, Manufacturers $manufacturers)
	{
		$this->productsService = $productsService;
		$this->manufacturers   = $manufacturers;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setShowLangSwitcher(false);

		$form->addSelect('manufacturer', 'eshopCatalog.product.manufacturer', [null => 'default.all'] + $this->manufacturers->getOptionsForSelect());
		$form->addSubmit('export', 'eshopCatalog.importExport.exportData')
			->setValidationScope(null);

		$form->addUpload('upload', 'eshopCatalog.importExport.dataForImport');
		$form->addSubmit('import', 'eshopCatalog.importExport.importData');

		$form->onSuccess[] = [$this, 'formOnSuccess'];

		return $form;
	}

	public function formOnSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			// Pokud se data exportují tak načteme data a odešleme do CSV response
			if ($form->isSubmitted()->name === 'export') {
				$filename = 'import-export-' . date('Y-m-d') . '.csv';
				$csv      = '"' . implode('";"', ['ID', 'Lang', 'Name', 'Code 1', 'Code 2', 'EAN', 'Retail price',
						'Price', 'Is variant']) . '"';

				$customerGroups = $this->em->getRepository(GroupCustomers::class)->createQueryBuilder('gc', 'gc.id')
					->select('gc.id, gc.name')->getQuery()->getArrayResult();

				foreach ($customerGroups as $row) {
					$csv .= ';"cg|' . $row['id'] . '|' . $row['name'] . '"';
				}

				$csv .= PHP_EOL;

				$qb = $this->productsService->getEr()->createQueryBuilder('p')
					->select('p.id, p.code1, pt.lang, pt.name, p.code2, p.ean, p.retailPrice, p.price, var.isDefault, var.useName, var.useRetailPrice, var.usePrice')
					->addSelect('GROUP_CONCAT(CONCAT(IDENTITY(pl.groupId), \'|\' , pl.price)) as priceLevels')
					->innerJoin('p.productTexts', 'pt')
					->leftJoin('p.priceLevels', 'pl')
					->leftJoin('p.isVariant', 'var')
					->groupBy('p.id');

				if ($values->manufacturer)
					$qb->where('p.manufacturer = :manufacturer')->setParameter('manufacturer', $values->manufacturer);

				foreach ($qb->getQuery()->getArrayResult() as $row) {
					$isVar = $row['isDefault'] === 0;

					$csv .= '"' . $row['id'] . '";';
					$csv .= '"' . $row['lang'] . '";';
					$csv .= '"' . $row['name'] . '";';
					$csv .= '"' . $row['code1'] . '";';
					$csv .= '"' . $row['code2'] . '";';
					$csv .= '"' . $row['ean'] . '";';
					$csv .= '"' . $row['retailPrice'] . '";';
					$csv .= '"' . $row['price'] . '";';
					$csv .= '"' . ($isVar ? 1 : 0) . '";';

					$priceLevels = [];
					if ($row['priceLevels'])
						foreach (explode(',', $row['priceLevels']) as $v) {
							$v                  = explode('|', $v);
							$priceLevels[$v[0]] = $v[1] ?? '';
						}

					foreach ($customerGroups as $k => $v)
						$csv .= '"' . ($priceLevels[$k] ?? '') . '";';

					$csv .= PHP_EOL;
				}

				CsvResponse::sendResponse($filename, $csv);
			} elseif ($values->upload && $values->upload->hasFile()) {
				// Při importu připravíte názvy tabulek a další hodnoty
				$productsTableName     = $this->em->getClassMetadata(Product::class)->getTableName();
				$productTextsTableName = $this->em->getClassMetadata(ProductTexts::class)->getTableName();
				$priceLevelsTableName  = $this->em->getClassMetadata(ProductPriceLevel::class)->getTableName();

				/** @var FileUpload $upload */
				$upload = $values->upload;
				$fr     = fopen($upload->getTemporaryFile(), 'r');
				$this->em->beginTransaction();
				$customerGroups    = [];
				$csvData           = [];
				$priceLevels       = [];
				$allowedProductsId = [];

				// Zjistíme produkty patřící k výrobci nebo všechny
				$qb = $this->productsService->getEr()->createQueryBuilder('p')->select('p.id');

				if ($values->manufacturer)
					$qb->where('p.manufacturer = :manu')->setParameter('manu', $values->manufacturer);

				foreach ($qb->getQuery()->getArrayResult() as $row)
					$allowedProductsId[] = $row['id'];

				// Přečteme nahraný csv soubor
				while (!feof($fr) && $line = fgetcsv($fr, null, ';')) {
					// V prvním řádku načteme dynamické sloupečky
					if ($line[0] == 'ID') {
						foreach ($line as $k => $v) {
							$tmp = explode('|', $v);

							if ($tmp[0] == 'cg')
								$customerGroups[$k] = $tmp[1];
						}

						continue;
					}

					if (!$line[0])
						continue;

					$productId = (int) $line[0];

					// Zkontrolujeme existenci produktu
					if (!in_array($productId, $allowedProductsId))
						continue;

					$csvData[$productId] = $line;
				}

				// Načteme cenové hladiny
				foreach ($this->em->getRepository(ProductPriceLevel::class)->createQueryBuilder('pl')->select('IDENTITY(pl.productId), IDENTITY(pl.groupId), pl.price')
					         ->innerJoin('pl.productId', 'p', Join::WITH, 'p.manufacturer = :manu')->setParameter('manu', $values->manufacturer)
					         ->getQuery()->getArrayResult() as $row)
					$priceLevels[$row['groupId'] . '-' . $row['productId']] = $row['price'];

				foreach ($csvData as $productId => $line) {
					$lang = trim($line[1]);
					$name = trim($line[2]);
					// Aktualizace názvu
					if ($lang && $name) {
						$this->em->getConnection()->update($productTextsTableName, [
							'name' => $name,
						], [
							'id'   => $productId,
							'lang' => $lang,
						]);
					}

					$updates     = [];
					$code1       = trim($line[3]);
					$code2       = trim($line[4]);
					$ean         = trim($line[5]);
					$retailPrice = strtr(trim($line[6]), [' ' => '', ',' => '.']);
					$price       = strtr(trim($line[7]), [' ' => '', ',' => '.']);

					// Aktulizace hodnot pokud jsou validní
					if ($code1)
						$updates['code1'] = $code1;
					if ($code2)
						$updates['code2'] = $code2;
					if ($ean)
						$updates['ean'] = $ean;
					if (is_numeric($retailPrice))
						$updates['retail_price'] = $retailPrice;
					if (is_numeric($price))
						$updates['price'] = $price;

					if ($updates) {
						$this->em->getConnection()->update($productsTableName, $updates, [
							'id' => $productId,
						]);
					}

					// Aktualizace cenových hladin
					foreach ($customerGroups as $lineK => $customerGroupId) {
						$price = strtr(trim($line[$lineK]), [' ' => '', ',' => '.']);
						$exist = isset($priceLevels[$customerGroupId . '-' . $productId]);

						if (is_numeric($price)) {
							// Aktualizace nebo vytvoření
							if ($exist)
								$this->em->getConnection()->update($priceLevelsTableName, [
									'price' => $price,
								], [
									'product_id' => $productId,
									'group_id'   => $customerGroupId,
								]);
							else
								$this->em->getConnection()->insert($priceLevelsTableName, [
									'product_id' => $productId,
									'group_id'   => $customerGroupId,
									'price'      => $price,
								]);
						} else if ($exist) {
							// Smazání
							$this->em->getConnection()->delete($priceLevelsTableName, [
								'product_id' => $productId,
								'group_id'   => $customerGroupId,
							]);
						}
					}
				}

				$this->em->commit();
				$this->getPresenter()->flashMessageSuccess('default.saved');
				$this->redirect('this');
			}
		} catch (\Exception $e) {
			$this->getPresenter()->flashMessageDanger($e->getMessage());
			$this->getPresenter()->redrawControl('flashes');

			if ($this->em->getConnection()->isTransactionActive())
				$this->em->rollback();

			return false;
		}

		return true;
	}
}
