<?php declare(strict_types=1);

namespace EshopCatalog\AdminModule\Model\FormContainers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\UI\Form\BaseContainer;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\Model\Entities\Product;
use Nette\Localization\ITranslator;

class RelatedProductContainer
{
	/** @var ITranslator */
	protected $translator;

	/** @var EntityManagerDecorator */
	protected $em;

	/** @var Products */
	protected $products;

	/**
	 * RelatedProductContainer constructor.
	 * @param ITranslator $translator
	 * @param EntityManagerDecorator $em
	 * @param Products $products
	 */
	public function __construct(ITranslator $translator, EntityManagerDecorator $em, Products $products)
	{
		$this->translator = $translator;
		$this->em = $em;
		$this->products = $products;
	}

	/**
	 * @param bool $multilang
	 * @return BaseContainer
	 */
	public function getContainer($multilang = false): BaseContainer
	{
		$container = new BaseContainer;
		$container->addText('product', 'eshopCatalog.productForm.findProducts')
				  ->setHtmlAttribute('class', ['productAutocomplete', 'typeahead'])
				  ->setHtmlAttribute('autocomplete', 'off')
				  ->setHtmlId('product-finder');
		$container->addCustomData('template', __DIR__ . '/RelatedProductContainer.latte');

		return $container;
	}

	/**
	 * @param $container
	 * @param Product[] $relatedProducts
	 */
	public function setDefaults(BaseContainer &$container, array $relatedProducts): void
	{
	}

	/**
	 * @param array $data
	 * @param Product $product
	 */
	public function saveData(array $data, Product $product): void
	{
		$relatedProductIds = $data['relatedProductIds'];

		foreach ($relatedProductIds as $relatedProductId) {
			$relatedProduct = $this->products->get((int) $relatedProductId);
			$product->addRelatedProduct($relatedProduct);
		}
	}

	/**
	 * @param string $term
	 * @param array $excludeProducts
	 * @return array
	 */
	public function findProducts(string $term, array $excludeProducts = []): array
	{
		$products = $this->products->getByTerm($term);
		$output = [];
		foreach ($products as $product) {
			if (!in_array($product->getId(), $excludeProducts)) {
				$output[$product->getId()] = array(
					'id'        => $product->getId(),
					'name'      => $product->getProductText($this->translator->getLocale())->name,
					'code1'		=> $product->code1,
					'ean'		=> $product->ean,
				);
			}
		}

		return $output;
	}

	/**
	 * @param int $productId
	 * @param int $relatedProductId
	 */
	public function removeRelatedProduct(int $productId, int $relatedProductId): void
	{
		$product = $this->products->get($productId);

		if (!$product) {
			return;
		}

		foreach ($product->getRelatedProducts() as $relatedProduct) {
			if ($relatedProduct->getId() === $relatedProductId) {
				$product->relatedProducts->removeElement($relatedProduct);
			}
		}

		$this->em->flush();
	}

}