<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Model;

use Doctrine\Common\Collections\ArrayCollection;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\Traits\TPublish;
use EshopCatalog\Model\Entities\Manufacturer;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductTexts;
use EshopCatalog\Model\Entities\ProductVariant;
use EshopCatalog\Model\Entities\VatRate;
use Gallery\Model\Entities\Album;
use Gallery\Model\Entities\Image;
use Nette\Application\BadRequestException;
use Nette\Utils\FileSystem;
use EshopCatalog\Model\Products as BaseProducts;

/**
 * Class Products
 * @package EshopCatalog\AdminModule\Model
 *
 * @method Product|null|object = getReference($id)
 * @method Product[]|null getAll()
 * @method Product|null get($id)
 */
class Products extends BaseEntityService
{
	use TPublish;

	protected BaseProducts $baseProductsService;

	protected $entityClass = Product::class;

	public function __construct(BaseProducts $baseProductsService)
	{
		$this->baseProductsService = $baseProductsService;
	}

	/**
	 * @param array|int $ids
	 * @param int       $manufacturerId
	 *
	 * @throws \Doctrine\ORM\ORMException
	 */
	public function setManufacturer($ids, $manufacturerId)
	{
		try {
			$manufacturer = $this->em->getReference(Manufacturer::class, $manufacturerId);
			foreach ($this->getEr()->findBy(['id' => !is_array($ids) ? [$ids] : $ids]) as $entity) {
				/** @var Product $entity */
				$entity->setManufacturer($manufacturer);
				$this->em->persist($entity);
			}

			$this->em->flush();

			return true;
		} catch (\Exception $e) {
		}

		return false;
	}

	/**
	 * @param array|int $ids
	 * @param int       $vatRateId
	 *
	 * @throws \Doctrine\ORM\ORMException
	 */
	public function setVatRate($ids, $vatRateId)
	{
		try {
			$vatRate = $this->em->getReference(VatRate::class, $vatRateId);
			foreach ($this->getEr()->findBy(['id' => !is_array($ids) ? [$ids] : $ids]) as $entity) {
				/** @var Product $entity */
				$entity->setVateRate($vatRate);
				$this->em->persist($entity);
			}

			$this->em->flush();

			return true;
		} catch (\Exception $e) {
		}

		return false;
	}

	/** Vrati seznam produktu, ktere obsahuji dany retezec
	 * @return ArrayCollection|Product[]
	 */
	public function getByTerm($term = null)
	{
		if (is_null($term)) {
			return new ArrayCollection();
		}
		$query = $this->getEr()->createQueryBuilder('p', 'p.id')
			->join('p.productTexts', 'pt', 'WITH', 'pt.lang = :lang')
			->join('p.vatRate', 'vr')
			->setParameter('lang', 'cs')//TODO ITranslator
			->orWhere('pt.name LIKE :term')->setParameter('term', '%' . $term . '%')
			->orWhere('p.code1 LIKE :term')
			->orWhere('p.ean LIKE :term')
			->orWhere('p.id LIKE :term')
			->orderBy('pt.name', 'ASC');

		$products = $query->getQuery()->getResult();

		return $products;
	}

	public function deleteDuplicity()
	{
		try {
			set_time_limit(600);
			$data = $this->getEr()->createQueryBuilder('p')->select('p.id, pt.name as name, p.code1, p.code2, count(p) as q')
				->leftJoin('p.productTexts', 'pt')
				->having('count(p) > 1')->groupBy('pt.name, p.code1, p.code2')->getQuery()->getResult();

			foreach ($data as $row) {
				$products = $this->getEr()->createQueryBuilder('p')->where('pt.name = :name AND p.code1 = :code1')
					->join('p.productTexts', 'pt')->orderBy('p.id')
					->setParameters(['code1' => $row['code1'], 'name' => $row['name']]);

				if ($row['code2'] == null) {
					$products->andWhere('p.code2 IS NULL');
				} else {
					$products->andWhere('p.code2 = :code2')->setParameter('code2', $row['code2']);
				}

				$products = $products->getQuery()->getResult();

				if (count($products) >= 2) {
					array_pop($products);

					foreach ($products as $product) {
						if ($product->gallery) {
							FileSystem::delete(WWW_DIR . $product->gallery->generatePath());

							$this->em->createQuery('DELETE FROM ' . Image::class . ' i WHERE i.album = :album')->execute(['album' => $product->gallery->getId()]);
							$this->em->createQuery('DELETE FROM ' . Album::class . ' a WHERE a.id = :album')->execute(['album' => $product->gallery->getId()]);
						}

						$this->em->remove($product);
						$this->em->flush();
					}

					$this->em->clear();
				}
			}
		} catch (\Exception $e) {
			return false;
		}

		return true;
	}

	public function setDiscountDisabled(array $ids, $value): bool
	{
		try {
			foreach ($ids as $id) {
				/** @var Product $product */
				$product = $this->getReference($id);

				$product->discountDisabled = $value ? 1 : 0;
				$this->em->persist($product);
			}

			$this->em->flush();
		} catch (\Exception $e) {
			return false;
		}

		return true;
	}

	public function createVariant(int $id): Product
	{
		$product = $this->get($id);

		if (!$product->isVariant) {
			$baseProductVariant                 = new ProductVariant($product, $this->getLastVariantId() + 1);
			$baseProductVariant->isDefault      = 1;
			$baseProductVariant->createdDefault = $product->getCreated();

			$this->em->persist($baseProductVariant);
		} else {
			if ($product->isVariant->isDefault === 1)
				$baseProductVariant = $product->isVariant;
			else {
				foreach ($product->variants->toArray() as $v)
					if ($v->isDefault === 1) {
						$product            = $v->product;
						$baseProductVariant = $v;
						break;
					}
			}
		}

		if (!isset($baseProductVariant))
			throw new BadRequestException();

		$variant = new Product();

		$texts = [];
		foreach ($product->getTexts()->toArray() as $text) {
			$this->em->detach($text);
			/** @var ProductTexts $text */
			$text->setProduct($variant);
			$texts[] = $text;
		}

		foreach ($product->getProductTags()->toArray() as $tag) {
			$this->em->detach($tag);
			$variant->addProductTag($tag);
			$this->em->persist($tag);
		}

		$variant->setProductTexts($texts);
		$variant->setGallery($product->getGallery());
		$variant->setVateRate($product->getVateRate());
		$this->em->persist($variant);

		$productVariant                 = new ProductVariant($variant, $baseProductVariant->getVariantId());
		$productVariant->createdDefault = $baseProductVariant->createdDefault;
		$this->em->persist($productVariant);

		$this->em->flush();

		return $variant;
	}

	public function removeFromVariants(Product $product): bool
	{
		$variantId = $product->isVariant->getVariantId();

		$allVariants = $this->em->getRepository(ProductVariant::class)->createQueryBuilder('pv')
			->where('pv.variantId = :variantId')->setParameter('variantId', $variantId)
			->getQuery()->getResult();

		if (count($allVariants) == 2) {
			foreach ($allVariants as $v)
				$this->em->remove($v);
			$this->em->flush();
		} else {
			$this->em->remove($product->isVariant);
			$this->em->flush();
		}

		return true;
	}

	public function setAsVariantFor(int $prodId, int $parentId): bool
	{
		if ($prodId === $parentId)
			return false;

		$product = null;
		$parent  = null;
		foreach ($this->getEr()->createQueryBuilder('p')->addSelect('isVar')
			         ->where('p.id IN (:ids)')->setParameter('ids', [$parentId, $prodId])
			         ->leftJoin('p.isVariant', 'isVar')
			         ->getQuery()->getResult() as $row) {
			/** @var Product $row */
			if ($row->getId() == $prodId)
				$product = $row;
			else
				$parent = $row;
		}

		if (!$product || !$parent)
			return false;

		if ($parent->isVariant && $parent->isVariant->isDefault === 0)
			return false;

		if (!$parent->isVariant) {
			$parentVariant                 = new ProductVariant($parent, $this->getLastVariantId() + 1);
			$parentVariant->createdDefault = $parent->getCreated();
			$parentVariant->isDefault      = 1;

			$this->em->persist($parentVariant);
		} else {
			$parentVariant = $parent->isVariant;
		}

		if (!$product->getGallery()) {
			$product->setGallery($parent->getGallery());
			$this->em->persist($product);
		}

		$productVariant = new ProductVariant($product, $parentVariant->getVariantId());
		/** @var ProductTexts $prodText */
		$prodText = $product->getText();

		$productVariant->useName             = $prodText->name ? 1 : 0;
		$productVariant->useName2            = $prodText->name2 ? 1 : 0;
		$productVariant->usePrice            = $product->price ? 1 : 0;
		$productVariant->useRetailPrice      = $product->retailPrice ? 1 : 0;
		$productVariant->useShortDescription = $prodText->shortDescription ? 1 : 0;
		$productVariant->useDescription      = $prodText->description ? 1 : 0;
		$productVariant->usePriceLevels      = $product->getPriceLevels()->count() ? 1 : 0;
		$productVariant->createdDefault      = $parentVariant->createdDefault;

		$this->em->persist($productVariant);

		foreach ($product->getCategoryProducts()->toArray() as $row)
			$this->em->remove($row);
		foreach ($product->sites->toArray() as $row)
			$this->em->remove($row);
		$this->em->persist($product);

		$this->em->flush();

		return true;
	}

	protected function getLastVariantId(): int
	{
		$r = $this->em->getRepository(ProductVariant::class)->createQueryBuilder('v')
			->select('MAX(v.variantId) as variantId')
			->getQuery()->setMaxResults(1)->getArrayResult();

		return isset($r[0]) ? (int) $r[0]['variantId'] : 0;
	}
}
