<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Components;

use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\Products;
use EshopCatalog\Model\Config;
use EshopOrders\FrontModule\Model\Dao\AddedCartItem;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\FrontModule\Model\Event\AddedCartItemEvent;
use Nette\Utils\ArrayHash;

class CartAddForm extends BaseControl
{
	protected Products $productsService;

	/** @var int @persistent */
	protected $productId;

	/** @var Product */
	protected $product;

	public function __construct(Products $products)
	{
		$this->productsService = $products;
	}

	protected function attached($presenter): void
	{
		parent::attached($presenter);

		if ($this->productId && !$this->product) {
			$this->setProduct($this->productId);
		}
	}

	public function render(): void
	{
		$this->template->product = $this->product;

		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$form->addHidden('productId');
		$form->addText('quantity')->setDefaultValue(1)
			->setAttribute('data-add-to-cart-quantity-input')
			->addRule(BaseForm::MIN, 'eshopCatalogFront.cart.minimumQuantity', 1);

		if (Config::load('pseudoWarehouse') && $this->product->unlimitedQuantity === 0) {
			$form['quantity']->addRule(BaseForm::MAX, 'eshopCatalogFront.cart.maximumQuantity', $this->product->getQuantity());
			$form['quantity']->setAttribute('data-max', $this->product->getQuantity());
		}

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		$presenter = $this->getPresenter();

		$canAddToCart = (int) $this->productsService->getEr()->createQueryBuilder('p')
			                      ->select('av.canAddToCart')
			                      ->where('p.id = :id')
			                      ->setParameter('id', $values->productId)
			                      ->innerJoin('p.availability', 'av')
			                      ->getQuery()->setMaxResults(1)->getArrayResult()[0]['canAddToCart'] ?? null;

		if (!$canAddToCart) {
			$presenter->flashMessageInfo('eshopOrderFront.cart.itemCannotBeAddedToCart');
			$presenter->redrawControl('flashes');

			return false;
		}

		$addedItem            = new AddedCartItem();
		$addedItem->quantity  = (int) $values->quantity ?: 1;
		$addedItem->productId = (int) $values->productId;

		$this->eventDispatcher->dispatch(new AddedCartItemEvent($addedItem), 'eshopOrders.cartAddItem');

		return true;
	}

	/**
	 * @param int|Product $product
	 */
	public function setProduct($product): self
	{
		if ($product instanceof Product) {
			$this->product   = $product;
			$this->productId = $product->getId();
		} else {
			$this->productId = $product;
		}

		if ($this->productId) {
			$this['form']->setDefaults([
				'productId' => $this->productId,
			]);
		}

		return $this;
	}
}
