<?php declare(strict_types = 1);

namespace EshopCatalog\FrontModule\Model\Export;

use Core\Model\Dao\Site;
use Core\Model\Dao\SiteDomain;
use Core\Model\Entities\Country;
use Core\Model\Entities\EntityManagerDecorator;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\Provider\IXmlExportService;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Feature;
use EshopOrders\Model\PaymentSpeditions;
use Nette\Utils\FileSystem;

abstract class XmlService implements IXmlExportService
{
	protected string $name;

	public ?string $baseUrl = null;

	protected ?Site $site = null;

	protected ?SiteDomain $domain = null;

	protected $fileHandle      = null;
	protected $fileHandleStock = null;

	/** @var PaymentSpeditions @inject */
	public $paymentSpeditions;

	/** @var EntityManagerDecorator @inject */
	public $em;

	protected ?array $cSpeditions = null;

	protected ?array $cFeatureExport = null;

	public function getFilePathTemp(): string { return TMP_DIR . "/exportproducts/{$this->name}.xml"; }

	public function getStockFilePathTemp(): string { return TMP_DIR . "/exportproducts/{$this->name}-stock.xml"; }

	public function getFilePath(): string
	{
		return sprintf(WWW_DIR . '/exportproducts/%s-%s-%s.xml',
			$this->getName(),
			$this->site->getIdent(),
			$this->domain->getLang(),
		);
	}

	public function getStockFilePath(): string { return str_replace('.xml', '-stock.xml', $this->getFilePath()); }

	public function startExport(): void
	{
		FileSystem::createDir(dirname($this->getFilePathTemp()));
		$this->fileHandle = fopen($this->getFilePathTemp(), 'w');
	}

	public function startExportStock(): void
	{
		FileSystem::createDir(dirname($this->getStockFilePathTemp()));
		$this->fileHandleStock = fopen($this->getStockFilePathTemp(), 'w');
	}

	public function writeToFile(string $data): void
	{
		if ($this->fileHandle)
			fwrite($this->fileHandle, $data);
	}

	public function writeToFileStock(string $data): void
	{
		if ($this->fileHandleStock)
			fwrite($this->fileHandleStock, $data);
	}

	public function setBaseUrl(string $baseUrl) { $this->baseUrl = $baseUrl; }

	public function setSite(Site $site) { $this->site = $site; }

	public function setDomain(SiteDomain $domain) { $this->domain = $domain; }

	public function getName(): string { return $this->name; }

	public function prepareExportData(Product $product): array
	{
		$ef       = $product->getExtraFields();
		$export   = $ef['export'][$this->name];
		$category = [];

		if (isset($ef['categoryExports'][$this->name])) {
			$vals = $ef['categoryExports'][$this->name];

			$category['status']       = $vals['status'];
			$category['categoryText'] = $vals['categoryText'];
		}

		if (!$export)
			$export = ['status' => 2];

		if (empty($export) || $export['status'] == 2)
			$export['status'] = $category['status'];

		if (!$export['categoryText'])
			$export['categoryText'] = $category['categoryText'];

		if (!$export['productName'])
			$export['productName'] = trim(
				(Config::load('exportGrader.addManufacturerToName') && $product->getManufacturer() ? $product->getManufacturer()->name . ' ' : '')
				. $product->getName()
			);

		if (!$export['product'])
			$export['product'] = $export['productName'];

		foreach (['product', 'productName'] as $v)
			$export[$v] = preg_replace('/\s\s+/', ' ', $export[$v]);

		return $export;
	}

	public function endExport()
	{
		fclose($this->fileHandle);
		FileSystem::rename($this->getFilePathTemp(), $this->getFilePath(), true);
	}

	public function endExportStock()
	{
		fclose($this->fileHandleStock);
		FileSystem::rename($this->getStockFilePathTemp(), $this->getStockFilePath(), true);
	}

	public function getAllSpeditions(): array
	{
		$entityValue = '';
		$groupKey    = '';
		$country     = 'cz';
		if ($this->name == 'heureka') {
			$entityValue = 'heurekaId';
			$groupKey    = 'heurekaId';
		} else if ($this->name == 'zbozi') {
			$entityValue = 'zboziId';
			$groupKey    = 'heurekaId';
		} else if ($this->name == 'google') {
			$country  = null;
			$groupKey = 'google';
		}

		if ($this->cSpeditions[$groupKey] === null) {
			$data = [];

			foreach ($this->paymentSpeditions->getAllPublishedByCountry($country) as $row) {
				$s = $row->getSpedition();
				if ($entityValue && !$s->{$entityValue})
					continue;

				$p = $row->getPayment();

				$key = $s->getName() . ' - ' . $p->getName();
				if ($entityValue)
					$key = $s->{$entityValue};

				foreach ($row->countries->toArray() as $country) {
					/** @var Country $country */
					$data[$country->getId()][$key][] = [
						'spedition' => [
							'ident'    => $s->getIdent(),
							'price'    => $s->getPrice(),
							'from'     => $s->getAvailableFrom(),
							'to'       => $s->getAvailableTo(),
							'freeFrom' => $s->getFreeFrom(),
						],
						'payment'   => [
							'ident'    => $p->getIdent(),
							'price'    => $p->getPrice(),
							'from'     => $p->getAvailableFrom(),
							'to'       => $p->getAvailableTo(),
							'freeFrom' => $p->getFreeFrom(),
						],
					];
				}
			}

			$this->cSpeditions[$groupKey] = $data;
		}

		return $this->cSpeditions[$groupKey];
	}

	public function getSpeditionsForPrice(float $prodPrice, bool $getLowest = true): array
	{
		$result = [];

		foreach ($this->getAllSpeditions() as $country => $speditions) {
			foreach ($speditions as $spedId => $rows) {
				$codAvailable = false;
				$d            = [
					'price' => 999999,
					'cod'   => 999999,
				];

				foreach ($rows as $row) {
					$s = $row['spedition'];
					$p = $row['payment'];
					if ($prodPrice < $s['from'] || $prodPrice > $s['to']
						|| $prodPrice < $p['from'] || $prodPrice > $p['to'])
						continue;

					$pPrice = $prodPrice > $p['freeFrom'] ? 0 : $p['price'];
					$sPrice = $prodPrice > $s['freeFrom'] ? 0 : $s['price'];

					$price = $sPrice;

					if (strtolower($p['ident']) === 'cod') {
						$cod          = $price + $pPrice;
						$codAvailable = true;

						if ($d['cod'] > $cod)
							$d['cod'] = $cod;
					} else {
						$price += $pPrice;
					}

					if ($d['price'] > $price)
						$d['price'] = $price;
				}

				if (!$codAvailable)
					unset($d['cod']);

				$result[$country][$spedId] = $d;
			}

			uasort($result[$country], fn($a, $b) => $a['price'] <=> $b['price']);
		}

		return $result;
	}

	public function canExportFeature(int $featureId)
	{
		if ($this->cFeatureExport === null) {
			foreach ($this->em->getRepository(Feature::class)->createQueryBuilder('f')
				         ->select('f.id, f.exportHeureka as heureka, f.exportZbozi as zbozi')
				         ->getQuery()->getArrayResult() as $row) {
				$this->cFeatureExport[$row['id']] = $row;
			}
		}

		return $this->cFeatureExport[$featureId][$this->name] ?? false;
	}
}
