<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Entities;

use Core\Model\Entities\TTranslateListener;
use Core\Model\Helpers\Traits\TExtraField;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Gallery\Model\Entities\Image;
use Gedmo\Mapping\Annotation as Gedmo;
use Gallery\Model\Entities\Album;
use Core\Model\Entities\TId;
use Nette\Utils\DateTime;

/**
 * @ORM\Table(name="eshop_catalog__product", indexes={
 *     @ORM\Index(name="published", columns={"is_published", "id"}),
 *     @ORM\Index(name="published_cat", columns={"is_published", "id_category_default", "id"}),
 * })
 * @ORM\Entity
 * @ORM\EntityListeners({"Core\Model\Entities\TranslateListener", "ProductListener"})
 */
class Product
{
	use TId;
	use TTranslateListener;
	use TExtraField;

	const EXTRA_FIELD_SECTION = 'eshopCatalogProduct';

	/**
	 * @var int
	 * @ORM\Column(name="is_published", type="smallint", length=1)
	 */
	public $isPublished;

	/**
	 * @var int
	 * @ORM\Column(name="in_stock", type="smallint", length=1)
	 */
	public $inStock;

	/**
	 * @var int
	 * @ORM\Column(name="quantity", type="integer", options={"default": 0}, nullable=false)
	 */
	public $quantity;

	/**
	 * @var double
	 * @ORM\Column(name="price", type="decimal", precision=10, scale=2, nullable=true)
	 */
	public $price;

	/**
	 * @var double
	 * @ORM\Column(name="retail_price", type="decimal", precision=10, scale=2, nullable=true)
	 */
	public $retailPrice;

	/**
	 * @var Manufacturer
	 * @ORM\ManyToOne(targetEntity="Manufacturer", cascade={"persist"})
	 * @ORM\JoinColumn(name="id_manufacturer", referencedColumnName="id", onDelete="SET NULL", nullable=true)
	 */
	protected $manufacturer;

	/**
	 * @var ProductTexts[]
	 *
	 * @ORM\OneToMany(targetEntity="ProductTexts", mappedBy="id", indexBy="lang", cascade={"all"})
	 */
	protected $productTexts;

	/**
	 * @var CategoryProduct[]
	 * @ORM\OneToMany(targetEntity="CategoryProduct", mappedBy="product", indexBy="category", cascade={"all"})
	 */
	protected $categoryProducts;

	/**
	 * @ORM\ManyToOne(targetEntity="Category", cascade={"persist"})
	 * @ORM\JoinColumn(name="id_category_default", referencedColumnName="id", onDelete="SET NULL")
	 */
	protected $idCategoryDefault;

	/**
	 * @var FeatureProduct[]
	 * @ORM\OneToMany(targetEntity="FeatureProduct", mappedBy="product", indexBy="product", cascade={"all"})
	 */
	protected $featureProducts;

	/**
	 * @ORM\OneToMany(targetEntity="ProductTag", mappedBy="product", indexBy="product", cascade={"all"})
	 */
	protected $productTags;

	/**
	 * @var ProductSupplier
	 *
	 * @ORM\OneToMany(targetEntity="ProductSupplier", mappedBy="product", indexBy="id_supplier", cascade={"all"})
	 */
	protected $suppliers;

	/**
	 * @var VatRate
	 * @ORM\ManyToOne(targetEntity="VatRate")
	 * @ORM\JoinColumn(name="id_vat_rate", referencedColumnName="id", onDelete="SET NULL")
	 */
	protected $vatRate;

	/**
	 * @var Availability
	 * @ORM\ManyToOne(targetEntity="Availability")
	 * @ORM\JoinColumn(name="id_availability", referencedColumnName="id", onDelete="SET NULL")
	 */
	protected $availability;

	/**
	 * @var string
	 * @ORM\Column(name="ean", type="string", length=20, nullable=true)
	 */
	public $ean;

	/**
	 * @var string
	 * @ORM\Column(name="code1", type="string", length=60, nullable=true)
	 */
	public $code1;

	/**
	 * @var string
	 * @ORM\Column(name="code2", type="string", length=60, nullable=true)
	 */
	public $code2;

	/**
	 * @var Album
	 *
	 * @ORM\ManyToOne(targetEntity="Gallery\Model\Entities\Album", cascade={"persist"})
	 * @ORM\JoinColumn(name="gallery_id", referencedColumnName="id", onDelete="SET NULL", nullable=true)
	 */
	protected $gallery;

	/**
	 * @var DateTime
	 * @ORM\Column(name="created", type="datetime")
	 */
	private $created;

	/**
	 * @var DateTime
	 * @Gedmo\Timestampable(on="update")
	 * @ORM\Column(name="modified", type="datetime", nullable=true)
	 */
	private $modified;

	/**
	 * @var ProductPriceLevel[]
	 * @ORM\OneToMany(targetEntity="ProductPriceLevel", mappedBy="productId", indexBy="group_id")
	 */
	protected $priceLevels;

	/**
	 * @var array
	 * @ORM\Column(name="more_data", type="array", nullable=true)
	 */
	protected $moreData = [];

	/**
	 * @var int
	 * @ORM\Column(name="unlimited_quantity", type="smallint", options={"default": 0}, nullable=false)
	 */
	public $unlimitedQuantity;

	/**
	 * @var int
	 * @ORM\Column(name="discount_disabled", type="smallint", options={"default": 0}, nullable=false)
	 */
	public $discountDisabled;

	/**
	 * @var int
	 * @ORM\Column(name="category_gifts_allowed", type="smallint", options={"default": 1}, nullable=false)
	 */
	protected $categoryGiftsAllowed;

	/**
	 * @var ProductExport
	 * @ORM\OneToMany(targetEntity="ProductExport", mappedBy="id", indexBy="service")
	 */
	public $export;

	/**
	 * @var ProductInSite[];
	 * @ORM\OneToMany(targetEntity="ProductInSite", mappedBy="product", indexBy="site")
	 */
	public $sites;

	/**
	 * @var ProductVariant[]|object
	 */
	public $variants;

	/**
	 * @var ProductVariant
	 * @ORM\OneToOne(targetEntity="ProductVariant", mappedBy="product", cascade={"persist"})
	 */
	public ?ProductVariant $isVariant = null;

	/**
	 * @var ArrayCollection|ProductDocument[]
	 * @ORM\OneToMany(targetEntity="ProductDocument", mappedBy="product", cascade={"all"})
	 */
	public $documents;

	/**
	 * @var ArrayCollection|Product[]
	 * @ORM\ManyToMany(targetEntity="Product")
	 * @ORM\JoinTable(name="eshop_catalog__related_products",
	 *     joinColumns={@ORM\JoinColumn(referencedColumnName="id")},
	 *     inverseJoinColumns={@ORM\JoinColumn(name="related_product_id", referencedColumnName="id")}
	 * )
	 */
	public $relatedProducts;

	/**
	 * @ORM\Column(name="is_assort", type="smallint", length=1, options={"default": 0})
	 */
	public int $isAssort = 0;

	public function __construct()
	{
		$this->productTexts         = new ArrayCollection();
		$this->categoryProducts     = new ArrayCollection();
		$this->featureProducts      = new ArrayCollection();
		$this->suppliers            = new ArrayCollection();
		$this->productTags          = new ArrayCollection();
		$this->quantity             = 0;
		$this->inStock              = 0;
		$this->moreData             = [];
		$this->created              = new DateTime('now');
		$this->priceLevels          = new ArrayCollection();
		$this->extraFields          = new ArrayCollection();
		$this->export               = new ArrayCollection();
		$this->unlimitedQuantity    = 0;
		$this->discountDisabled     = 0;
		$this->categoryGiftsAllowed = 1;
		$this->isPublished          = 1;
		$this->sites                = new ArrayCollection();
		$this->variants             = new ArrayCollection();
		$this->documents            = new ArrayCollection();
		$this->relatedProducts      = new ArrayCollection();
	}

	public function addProductText($lang)
	{
		$this->productTexts->set($lang, new ProductTexts($this, $lang));
	}

	public function setProductTexts(array $texts): self
	{
		$this->productTexts = new ArrayCollection($texts);

		return $this;
	}

	public function setProductText(ProductTexts $productTexts)
	{
		$this->productTexts->set($productTexts->getLang(), $productTexts);
	}

	/**
	 * @deprecated use getText
	 */
	public function getProductText($lang = null)
	{
		return $this->productTexts->get($lang ?: $this->locale) ?? null;
	}

	public function getText($lang = null)
	{
		return $this->productTexts->get($lang ?: $this->locale) ?? null;
	}

	/**
	 * @deprecated use getTexts
	 */
	public function getProductTexts() { return $this->productTexts; }

	public function getTexts() { return $this->productTexts; }

	public function getFeatureProducts()
	{
		return $this->featureProducts;
	}

	/**
	 * @param CategoryProduct $categoryProduct
	 *
	 * @return $this
	 */
	public function addCategoryProduct(CategoryProduct $categoryProduct)
	{
		if (!$this->categoryProducts->contains($categoryProduct))
			$this->categoryProducts->add($categoryProduct);

		return $this;
	}

	public function removeCategoryProduct(CategoryProduct $categoryProduct)
	{
		$this->categoryProducts->removeElement($categoryProduct);
	}

	/**
	 * @return ArrayCollection|CategoryProduct[]
	 */
	public function getCategoryProducts()
	{
		return $this->categoryProducts;
	}

	public function removeAllCategoryProducts()
	{
		$this->categoryProducts->clear();
	}

	public function setCategories(array $categories): self
	{
		$this->categoryProducts = new ArrayCollection($categories);

		return $this;
	}

	public function removeProductTag(ProductTag $tag): self
	{
		$this->productTags->removeElement($tag);

		return $this;
	}

	public function addProductTag(ProductTag $productTag)
	{
		$productTag->setProduct($this);
		$this->productTags->add($productTag);
	}

	public function getProductTags()
	{
		return $this->productTags;
	}

	public function setManufacturer(?Manufacturer $manufacturer = null)
	{
		$this->manufacturer = $manufacturer;
	}

	public function getManufacturer()
	{
		return $this->manufacturer;
	}

	/**
	 * @deprecated
	 */
	public function setDefaultCategory(Category $category)
	{
		$this->idCategoryDefault = $category;
	}

	/**
	 * @deprecated
	 */
	public function getDefaultCategory()
	{
		return $this->idCategoryDefault;
	}

	public function setVateRate(VatRate $vatRate)
	{
		$this->vatRate = $vatRate;
	}

	public function getVateRate(): ?VatRate
	{
		return $this->vatRate;
	}

	public function setAvailability(?Availability $availability = null): self
	{
		$this->availability = $availability;

		return $this;
	}

	public function getAvailability(): ?Availability { return $this->availability; }

	/*************************************
	 * == Gallery
	 */

	public function getGallery() { return $this->gallery; }

	public function setGallery(Album $album): self
	{
		$this->gallery = $album;

		return $this;
	}

	public function getImage(): ?Image
	{
		$img = null;
		if ($this->gallery) {
			if ($this->isVariant && $this->isVariant->defaultImage)
				$img = $this->gallery->getImages()[$this->isVariant->defaultImage] ?? null;

			if (!$img)
				$img = $this->gallery->getCoverImage();
		}

		return $img;
	}

	/*************************************
	 * == ProductPriceLevel
	 */

	public function getPriceLevels()
	{
		return $this->priceLevels;
	}

	/*************************************
	 * == Supplier
	 */

	public function getSupplier($id)
	{
		return $this->suppliers->get($id);
	}

	public function getSuppliers()
	{
		return $this->suppliers;
	}

	public function setSupplier(ProductSupplier $ps)
	{
		if (!$this->suppliers->containsKey($ps->getSupplier()->getId())) {
			$this->suppliers->clear();
			$this->suppliers->add($ps);
		}
	}

	/*************************************
	 * == Dates
	 */

	public function getCreated() { return $this->created; }

	public function getModified() { return $this->modified; }

	public function validateCreated()
	{
		if ($this->created->format('y-m-d') <= 0)
			$this->created = new DateTime();
	}

	/*************************************
	 * == MoreData
	 */

	/**
	 * @param string $key
	 *
	 * @return mixed|null
	 */
	public function getMoreDataValue($key) { return $this->moreData[$key] ?? null; }

	/**
	 * @param string $key
	 * @param string $value
	 *
	 * @return $this
	 */
	public function setMoreDataValue($key, $value)
	{
		if (!is_array($this->moreData))
			$this->moreData = [];

		$this->moreData[$key] = $value;

		return $this;
	}

	/**
	 * @param array $data
	 *
	 * @return $this
	 */
	public function setMoreData($data)
	{
		$this->moreData = $data;

		return $this;
	}

	/*************************************
	 * == Booleans
	 */

	public function isCategoryGiftsAllowed(): int { return (int) $this->categoryGiftsAllowed; }

	public function setCategoryGiftsAllowed($allowed): self
	{
		$this->categoryGiftsAllowed = (int) $allowed;

		return $this;
	}

	public function getIsVariantParent(): bool { return $this->isVariant && $this->isVariant->isDefault === 1; }

	public function getVariantParent(): ?Product
	{
		if ($this->getIsVariantParent())
			return $this;

		if ($this->variants)
			foreach ($this->variants->toArray() as $v)
				if ($v->isDefault === 1)
					return $v->getProduct();

		return null;
	}

	/**
	 * @return ProductDocument[]
	 */
	public function getDocuments(): array
	{
		return $this->documents->toArray();
	}

	/**
	 * @param ProductDocument $document
	 */
	public function addDocument(ProductDocument $document): void
	{
		$this->documents->add($document);
	}

	/**
	 * @return Product[]
	 */
	public function getRelatedProducts(): array
	{
		return $this->relatedProducts->toArray();
	}

	/**
	 * @param Product $product
	 */
	public function addRelatedProduct(Product $product): void
	{
		$contains = false;
		foreach ($this->relatedProducts as $relatedProduct) {
			if ($relatedProduct->getId() === $product->getId()) {
				$contains = true;
				break;
			}
		}

		if ($this->getId() !== $product->getId() && !$contains) {
			$this->relatedProducts->add($product);
		}
	}

}
