<?php declare(strict_types = 1);

namespace EshopCatalog\Model\Subscribers;

use Core\Model\Entities\ExtraField;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Category;
use Navigations\Model\Event\RouteInEvent;
use Navigations\Model\Navigations;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Core\Model\Entities\EntityManagerDecorator;
use Navigations\Model\Event\RouteOutEvent;
use EshopCatalog\Model\Navigation\BaseNavigation;
use EshopCatalog\Model\NavigationFilter;
use Navigations\Model\Event\RouteUpdateCacheDep;
use Nette\Caching\Cache;

class NavigationSubscriber implements EventSubscriberInterface
{
	protected EntityManagerDecorator $entityManager;

	protected NavigationFilter $navigationFilter;

	protected Categories $categories;

	protected Navigations $navigations;

	protected static $outFilter = [];

	public function __construct(EntityManagerDecorator $entityManager, NavigationFilter $navigationFilter,
	                            Categories $categories, Navigations $navigations)
	{
		$this->entityManager    = $entityManager;
		$this->navigationFilter = $navigationFilter;
		$this->categories       = $categories;
		$this->navigations      = $navigations;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			BaseNavigation::class . '::routeOut'            => ['routeOut', 100],
			BaseNavigation::class . '::routeUpdateCacheDep' => ['routeUpdateCacheDep', 100],
			Navigations::class . '::routeInNotFound'        => ['routeInNotFound', 110],
		];
	}

	public function routeUpdateCacheDep(RouteUpdateCacheDep $event)
	{
		if (!empty(self::$outFilter)) {
			foreach (self::$outFilter as $type => $v)
				foreach ($v as $k => $vv)
					$event->cache[Cache::TAGS][] = 'categoriesFilter_' . $type . '_' . $k;

			self::$outFilter = [];
		}
	}

	public function routeOut(RouteOutEvent $event): void
	{
		if ($event->disableDefaultFilter === true)
			return;

		$urlParams = &$event->urlParams;
		$url       = &$event->url;

		$event->urlParams = $this->navigationFilter->processRouterOut($urlParams);

		if ($urlParams['productsFilter-filter'])
			self::$outFilter = $urlParams['productsFilter-filter'];
	}

	public function routeInNotFound(RouteInEvent $event)
	{
		if (!Config::load('allowCategoryFullUrlField', false))
			return;

		$catId = $this->entityManager->getRepository(ExtraField::class)->createQueryBuilder('ef')
				->select('ef.sectionKey')
				->where('ef.sectionName = :sn')
				->andWhere('ef.key = :key')
				->andWhere('ef.value = :value')
				->andWhere('ef.lang = :lang')
				->setParameters([
					'sn'    => Category::EXTRA_FIELD_SECTION,
					'key'   => 'fullUrl',
					'value' => $event->urlParams['path'],
					'lang'  => $event->urlParams['locale'],
				])->getQuery()->getScalarResult()[0]['sectionKey'] ?? null;

		if ($catId) {
			$catId = (int) $catId;
			$cat   = $this->categories->get($catId);

			if ($cat) {
				$r   = $cat->getParent();
				$nav = null;
				while ($r) {
					$nav = $this->categories->findNavigationId($r->getId());

					if ($nav)
						break;

					$r = $cat->getParent();
				}

				if ($nav) {
					$nav                               = $this->navigations->getNavigation($nav);
					$event->return['presenter']        = 'EshopCatalog:Front:Default';
					$event->return['action']           = 'category';
					$event->return['id']               = $catId;
					$event->return['activeNavigation'] = $nav;
					$event->stopPropagation();
				}
			}
		}
	}
}
