<?php declare(strict_types = 1);

namespace EshopCatalog\AdminModule\Components\Products;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Model\Features;
use EshopCatalog\AdminModule\Model\FeatureValues;
use EshopCatalog\AdminModule\Model\ProductVariantCombinations;
use EshopCatalog\AdminModule\Model\ProductVariants;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Entities\FeatureProduct;
use EshopCatalog\Model\Entities\ProductVariant;
use EshopCatalog\Model\Entities\ProductVariantCombination;
use EshopCatalog\Model\Entities\Variant;
use EshopCatalog\Model\Entities\VariantValue;
use Nette\Application\UI\Form;
use Nette\Http\IResponse;
use Nette\Http\Request;
use Nette\Utils\ArrayHash;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductTexts;
use EshopCatalog\AdminModule\Model\Products;
use Nette\Utils\Json;

/**
 * Class ProductsFeatureForm
 * @package EshopCatalog\AdminModule\Components\Products
 */
class ProductsFeatureForm extends BaseControl
{
	/** @var array */
	public $products;

	/** @var string */
	private $ids;

	/** @var Products */
	protected $productServices;

	/** @var Features */
	protected $featuresService;

	/** @var FeatureValues */
	protected $featureValuesService;

	/** @var array */
	protected $features;

	public function __construct(Products $products, Features $features, FeatureValues $featureValues)
	{
		$this->productServices      = $products;
		$this->featuresService      = $features;
		$this->featureValuesService = $featureValues;
	}

	public function render()
	{
		$this->template->featuresJson = Json::encode($this->getFeatures());
		$this->template->render($this->getTemplateFile());
	}

	protected function attached($presenter)
	{
		parent::attached($presenter);
		$feature = $this->getPresenter()->getHttpRequest()->getPost('feature');
		if ($feature)
			$this['form']['value']->setItems($this->getFeatures()['values'][$feature]);
	}

	/*******************************************************************************************************************
	 * ========================  COMPONENTS
	 */

	protected function createComponentForm()
	{
		$form = $this->createForm();

		$form->addHidden('produtcIds', $this->ids);

		$form->addSelect('feature', 'eshopCatalog.productFeature.feature', $this->getFeatures()['features']);
		$form->addSelect('value', 'eshopCatalog.productFeature.value', $this->getFeatures()['values'][key($this->getFeatures()['features'])] ?? []);

		$form->addSaveCancelControl('saveControl');
		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			$featureId = $values->feature;
			$valueId   = $this->getPresenter()->getHttpRequest()->getPost('value');

			$tmp = $this->products;
			foreach ($this->products as $product) {
				/** @var FeatureProduct[] $exists */
				$exists = $this->em->getRepository(FeatureProduct::class)->createQueryBuilder('fp')
					->where('fp.product = :productId')->setParameter('productId', $product->getId())
					->andWhere('fp.feature= :featureId')->setParameter('featureId', $featureId)
					->getQuery()->getResult();

				foreach ($exists as $exist) {
					unset($tmp[$product->getId()]);
					$exist->setFeature($this->featuresService->getReference($featureId));
					$exist->setFeatureValue($this->featureValuesService->getReference($valueId));
					$this->em->persist($exist);
				}
			}

			foreach ($tmp as $product) {
				$featureProduct = new FeatureProduct(
					$product,
					$this->featuresService->getReference($featureId),
					$this->featureValuesService->getReference($valueId)
				);

				$this->em->persist($featureProduct);
			}

			$this->em->flush();
			$form->addCustomData('productIds', $values->produtcIds);
			$this->getPresenter()->flashMessageSuccess('eshopCatalog.productForm.productSaved');
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}
	}

	/*******************************************************************************************************************
	 * ========================  GET / SET
	 */

	public function setProducts($ids)
	{
		$this->ids  = $ids;
		$productIds = explode('-', $ids);
		foreach ($productIds as $id) {
			$this->products[$id] = $this->em->getRepository(Product::class)->find($id);
		}

		if (!$this->products)
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}

	protected function getFeatures()
	{
		if (empty($this->features)) {
			$features = [];
			foreach ($this->featuresService->getEr()->createQueryBuilder('f')->select('f.id, ft.name')
				         ->join('f.featureTexts', 'ft', 'WITH', 'ft.lang = :lang')
				         ->setParameter('lang', $this->translator->getLocale())
				         ->where('f.isPublished = 1')
				         ->getQuery()->getResult() as $v)
				$features[$v['id']] = $v['name'];

			$values = [];
			foreach ($this->featureValuesService->getEr()->createQueryBuilder('fv')->select('fv.id, fvt.name, IDENTITY(fv.feature) as featureId')
				         ->join('fv.featureValueTexts', 'fvt', 'WITH', 'fvt.lang = :lang')
				         ->setParameter('lang', $this->translator->getLocale())
				         ->where('fv.isPublished = 1')
				         ->getQuery()->getResult() as $v)
				$values[$v['featureId']][$v['id']] = $v['name'];

			$this->features = ['features' => $features, 'values' => $values];
		}

		return $this->features;
	}
}
